package fr.ird.observe.navigation.model;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import fr.ird.observe.dto.IdDto;
import io.ultreia.java4all.bean.AbstractJavaBean;

import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Created by tchemit on 26/05/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public abstract class DtoModelNavigationNodeSupport<D extends IdDto> extends AbstractJavaBean implements DtoModelNavigationNode<D> {

    private final Class<D> type;
    private DtoModelNavigationNode<?> parent;
    private String id;
    private ImmutableList<DtoModelNavigationNode<?>> children;
    private Integer level;

    @SuppressWarnings({"unchecked", "WeakerAccess"})
    public DtoModelNavigationNodeSupport() {
        this.type = (Class) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
        this.children = ImmutableList.of();
    }

    @Override
    public Class<D> getType() {
        return type;
    }

    @Override
    public int getLevel() {
        return level == null ? level = (isRoot() ? 0 : 1 + parent.getLevel()) : level;
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public void setId(String id) {
        String oldValue = getId();
        this.id = id;
        firePropertyChange(PROPERTY_ID, oldValue, id);
        firePropertyChange(PROPERTY_ENABLED, oldValue != null, id != null);
        if (id == null) {
            getChildren().forEach(n -> n.setId(null));
        }
    }

    @Override
    public DtoModelNavigationNode<?> getParent() {
        return parent;
    }

    @Override
    public ImmutableList<DtoModelNavigationNode<?>> getChildren() {
        return children;
    }

    @Override
    public <N extends DtoModelNavigationNode<?>> void init(N parentNode, ImmutableList<DtoModelNavigationNode<?>> children) {
        this.parent = parentNode;
        this.children = children;
    }

    @Override
    public <N extends DtoModelNavigationNode<?>> ImmutableList<N> getNodesFromAncestor(DtoModelNavigationNode<?> other) {
        List<N> result = new ArrayList<>(getLevel());
        getAncestorNodes(other, result);
        Collections.reverse(result);
        return ImmutableList.copyOf(result);
    }

    @Override
    public <N extends DtoModelNavigationNode<?>> ImmutableList<N> getNodesToAncestor(DtoModelNavigationNode<?> other) {
        List<N> result = new ArrayList<>(getLevel());
        getAncestorNodes(other, result);
        return ImmutableList.copyOf(result);
    }

    @SuppressWarnings("unchecked")
    private <N extends DtoModelNavigationNode<?>> void getAncestorNodes(DtoModelNavigationNode<?> other, List<N> builder) {
        builder.add((N) this);
        if (!this.equals(other)) {
            ((DtoModelNavigationNodeSupport) parent).getAncestorNodes(other, builder);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        DtoModelNavigationNodeSupport<?> that = (DtoModelNavigationNodeSupport<?>) o;
        return Objects.equals(type, that.type)
                && Objects.equals(id, that.id)
                && Objects.equals(getModel(), that.getModel());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getModel(), type, id);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "{" +
                "parent=" + (parent == null ? "" : parent.getClass().getSimpleName()) +
                ", level=" + getLevel() +
                (isEnabled() ? ", id='" + id + '\'' : ", disabled") + '}';
    }
}
