package fr.ird.observe.navigation.model;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import fr.ird.observe.dto.IdDto;
import io.ultreia.java4all.bean.AbstractJavaBean;
import io.ultreia.java4all.lang.Objects2;
import io.ultreia.java4all.util.ServiceLoaders;

import java.lang.reflect.ParameterizedType;
import java.util.Objects;

/**
 * Created by tchemit on 26/05/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("WeakerAccess")
public abstract class DtoModelNavigationModelSupport<N extends DtoModelNavigationNode<?>> extends AbstractJavaBean implements DtoModelNavigationModel<N> {

    private final ImmutableList<N> nodes;
    private final N root;

    protected DtoModelNavigationModelSupport() {
        @SuppressWarnings("unchecked")
        Class<N> nodeType = (Class) ((ParameterizedType) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0]).getRawType();
        ImmutableMap.Builder<N, Class<? extends DtoModelNavigationNode<?>>[]> nodesAnnotationBuilders = ImmutableMap.builder();
        ImmutableMap.Builder<Class<? extends DtoModelNavigationNode<?>>, N> parentBuilders = ImmutableMap.builder();
        this.nodes = Objects.requireNonNull(loadNodes(nodeType, nodesAnnotationBuilders, parentBuilders));
        this.root = Objects.requireNonNull(loadRoot(nodeType, parentBuilders.build(), nodesAnnotationBuilders.build()));
        root.addPropertyChangeListener(DtoModelNavigationNode.PROPERTY_ENABLED, e -> {
            boolean oldValue = (boolean) e.getOldValue();
            boolean newValue = (boolean) e.getNewValue();
            firePropertyChange(PROPERTY_ENABLED, oldValue, newValue);
        });
    }

    @Override
    public boolean accept(IdDto dto) {
        return nodes.stream().anyMatch(n -> n.accept(dto));
    }


    @SuppressWarnings("unchecked")
    @Override
    public <D extends IdDto> DtoModelNavigationNode<D> getDtoNode(Class<D> type) {
        return (DtoModelNavigationNode<D>) nodes.stream().filter(n -> type.equals(n.getType())).findFirst().orElse(null);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <D extends IdDto> DtoModelNavigationNode<D> getNavigationNode(DtoModelNavigationNode<?> selectNode) {
        DtoModelNavigationNode<?> editNode = null;
        while (selectNode != null && editNode == null) {
            Class<? extends IdDto> type = selectNode.getType();
            DtoModelNavigationNode<? extends IdDto> n = getDtoNode(type);
            if (n == null) {
                selectNode = selectNode.getParent();
            } else {
                editNode = n;
            }
        }
        return (DtoModelNavigationNode<D>) editNode;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <D extends DtoModelNavigationNode> D getNode(Class<D> type) {
        return (D) nodes.stream().filter(n -> type.equals(n.getClass())).findFirst().orElse(null);
    }

    @Override
    public N getRoot() {
        return root;
    }

    @Override
    public ImmutableList<N> getNodes() {
        return nodes;
    }

    @Override
    public ImmutableList<N> getNodesWithIds() {
        ImmutableList.Builder<N> builder = ImmutableList.builder();
        root.getIds(builder);
        return builder.build();
    }

    protected ImmutableList<N> loadNodes(Class<N> nodeType,
                                         ImmutableMap.Builder<N, Class<? extends DtoModelNavigationNode<?>>[]> nodesAnnotationBuilders,
                                         ImmutableMap.Builder<Class<? extends DtoModelNavigationNode<?>>, N> parentBuilders) {
        ImmutableList.Builder<N> nodesBuilders = ImmutableList.builder();
        for (Class<N> thisNodeType : ServiceLoaders.loadTypes(nodeType)) {
            N node = Objects2.newInstance(thisNodeType);
            nodesBuilders.add(node);
            DtoModelNavigationNodeDefinition annotation = Objects.requireNonNull(node.getClass().getAnnotation(DtoModelNavigationNodeDefinition.class));
            Class<? extends DtoModelNavigationNode<?>>[] children = annotation.value();
            if (children.length > 0) {
                nodesAnnotationBuilders.put(node, children);
                for (Class<? extends DtoModelNavigationNode<?>> child : children) {
                    parentBuilders.put(child, node);
                }
            }
        }
        return nodesBuilders.build();
    }

    protected N loadRoot(Class<N> nodeType, ImmutableMap<Class<? extends DtoModelNavigationNode<?>>, N> parentMap, ImmutableMap<N, Class<? extends DtoModelNavigationNode<?>>[]> annotations) {
        N root = null;

        for (N node : nodes) {
            @SuppressWarnings("SuspiciousMethodCalls") N parentNode = parentMap.get(node.getClass());
            Class<? extends DtoModelNavigationNode<?>>[] children = annotations.get(node);
            ImmutableList.Builder<DtoModelNavigationNode<?>> childrenNodes = ImmutableList.builder();
            if (children != null) {
                ImmutableMap<? extends Class<? extends DtoModelNavigationNode>, N> nodesByClass = Maps.uniqueIndex(nodes, DtoModelNavigationNode::getClass);
                for (Class<? extends DtoModelNavigationNode<?>> child : children) {
                    childrenNodes.add(Objects.requireNonNull(nodesByClass.get(child), "can't find node of type: " + child));
                }
            }
            node.init(parentNode, childrenNodes.build());
        }
        for (N node : nodes) {
            if (node.isRoot()) {
                if (root != null) {
                    throw new IllegalStateException(String.format("Found two root node: %s and %s", root, node));
                }
                root = node;
            }
        }
        return Objects.requireNonNull(root, String.format("Could not find a root node of type: %s", nodeType));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        DtoModelNavigationModelSupport<?> that = (DtoModelNavigationModelSupport<?>) o;
        return Objects.equals(nodes, that.nodes) &&
                Objects.equals(root, that.root);
    }

    @Override
    public int hashCode() {
        return Objects.hash(nodes, root);
    }

}
