package fr.ird.observe.dto.reference;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.ObserveDto;

import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * Created on 21/11/15.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public abstract class DtoReference<D extends IdDto, R extends DtoReference<D, R>> extends IdDto implements ObserveDto, DtoReferenceAware {


    public static final String PROPERTY_TYPE = "type";

    public static final String PROPERTY_CREATE_DATE = "createDate";

    public static final String PROPERTY_VERSION = "version";

    public abstract DtoReferenceDefinition<D, R> getDefinition();

    protected DtoReference(DtoReferenceAware dto) {
        Objects.requireNonNull(dto);
        this.id = dto.getTopiaId();
        this.version = dto.getTopiaVersion();
        this.createDate = dto.getTopiaCreateDate();
        this.lastUpdateDate = dto.getLastUpdateDate();
    }

    public static <BeanType extends DtoReference> Predicate<BeanType> newIdPredicate(String id) {
        return input -> Objects.equals(id, input.getId());
    }

    public static <BeanType extends DtoReference> Map<String, BeanType> splitById(Collection<BeanType> dtos) {
        return Maps.uniqueIndex(dtos, DtoReference::getId);
    }

    public static <BeanType extends DtoReference> Collection<BeanType> filterById(Collection<BeanType> source, String id) {
        return source.stream().filter(newIdPredicate(id)).collect(Collectors.toList());
    }

    public static <BeanType extends DtoReference> BeanType find(Collection<BeanType> source, String id) {
        return source.stream().filter(newIdPredicate(id)).findFirst().orElse(null);
    }

    @Override
    public final Date getCreateDate() {
        return createDate;
    }

    @Override
    public final long getVersion() {
        return version;
    }

    @SuppressWarnings("unchecked")
    public <O> O getPropertyValue(String propertyName) {
        Function<R, O> propertyFunction = getDefinition().getPropertyFunction(propertyName);
        return propertyFunction.apply((R) this);
    }

    public Class<D> getDtoType() {
        return getDefinition().getDtoType();
    }

    public Class<R> getReferenceType() {
        return getDefinition().getType();
    }

    public ImmutableSet<String> getPropertyNames() {
        return getDefinition().getPropertyNames();
    }

}
