package fr.ird.observe.dto;

/*
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2019 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.reference.DtoReference;
import org.apache.commons.lang3.StringUtils;

import java.util.Collection;
import java.util.Objects;
import java.util.function.Predicate;

public class IdHelper {

    private static final String DTO_ROOT_PACKAGE = "fr.ird.observe.dto";
    private static final int DTO_ROOT_PACKAGE_LENGTH = DTO_ROOT_PACKAGE.length() + 1;
    private static final String DTO_REFERENTIAL_PACKAGE = DTO_ROOT_PACKAGE + ".referential";
    private static final String DTO_DATA_PACKAGE = DTO_ROOT_PACKAGE + ".data";
    private static final String ENTITY_ROOT_PACKAGE = "fr.ird.observe.entities";
    private static final int ENTITY_ROOT_PACKAGE_LENGTH = ENTITY_ROOT_PACKAGE.length() + 1;

    public static boolean isReferential(Class type) {
        return isReferentialFromPackageName(type.getPackage().getName());
    }

    public static boolean isReferentialFromPackageName(String packageName) {
        return packageName.startsWith(DTO_REFERENTIAL_PACKAGE);
    }

    public static boolean isData(Class type) {
        return isDataFromPackageName(type.getPackage().getName());
    }

    private static boolean isDataFromPackageName(String packageName) {
        return packageName.startsWith(DTO_DATA_PACKAGE);
    }

    public static String getDtoSimplifiedName(Class<?> dataType) {
        Objects.requireNonNull(dataType, " Null type given");
        return cleanId(dataType.getName().substring(DTO_ROOT_PACKAGE_LENGTH));
    }

    public static String getEntitySimplifiedName(Class<?> dataType) {
        Objects.requireNonNull(dataType, " Null type given");
        return cleanId(dataType.getName().substring(ENTITY_ROOT_PACKAGE_LENGTH));
    }

    public static String getEntityShortName(String id) {
        String simplifiedName = getEntitySimplifiedName(id);
        int i = simplifiedName.indexOf("#");
        simplifiedName = simplifiedName.substring(0, i);
        int j = simplifiedName.lastIndexOf(".");
        return j == -1 ? simplifiedName : simplifiedName.substring(j + 1);
    }

    public static String getEntitySimplifiedName(String dataType) {
        Objects.requireNonNull(dataType, " Null type given");
        return cleanId(dataType.substring(ENTITY_ROOT_PACKAGE_LENGTH));
    }

    public static String getDtoShortName(Class<?> dataType) {
        Objects.requireNonNull(dataType, " Null type given");
        return cleanId(dataType);
    }

    public static String cleanId(Class id) {
        return cleanId(id.getSimpleName());
    }

    public static boolean equalsCleanId(Class one, Class two) {
        return IdHelper.cleanId(one.getSimpleName()).equals(IdHelper.cleanId(two.getSimpleName()));
    }

    public static String cleanId(String id) {
        if (id.endsWith("Dto")) {
            return id.substring(0, id.length() - 3);
        }
        if (id.endsWith("Reference")) {
            return id.substring(0, id.length() - 9);
        }
        // for entity, hibernate proxy gives use some class name like Entity_xxx
        int index = id.indexOf("_");
        if (index > -1) {
            id = id.substring(0, index);
        } else {
            // for entity, hibernate
            index = id.indexOf("@");
            if (index > -1) {
                id = id.substring(0, index);
            }
        }
        if (id.endsWith("Impl")) {
            return id.substring(0, id.length() - 4);
        }
        return id;
    }

    public static boolean isTrip(DtoReference dto) {
        return isTripSeineId(dto.getId()) || isTripLonglineId(dto.getId());
    }

    public static boolean isTripLonglineId(String id) {
        return id.contains("TripLongline");
    }

    public static boolean isTripSeineId(String id) {
        return id.contains("TripSeine");
    }

    public static <BeanType extends IdDto> BeanType findById(Collection<BeanType> source, String id) {
        return source.stream().filter(newIdPredicate(id)::test).findFirst().orElse(null);
    }

    public static <BeanType extends IdDto> boolean exists(Collection<BeanType> source, String id) {
        return source.stream().anyMatch(newIdPredicate(id)::test);
    }

    public static <BeanType extends IdDto> Predicate<BeanType> newIdPredicate(String id) {
        return o -> Objects.equals(id, o.getId());
    }

    public static String getRelativeDtoPackage(String dtoClassName) {
        return StringUtils.removeStart(dtoClassName, DTO_ROOT_PACKAGE);
    }

}
