package fr.ird.observe.dto.form;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2017 - 2020 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableMap;
import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.reference.DtoReferenceDefinition;
import fr.ird.observe.dto.reference.ReferentialDtoReference;
import fr.ird.observe.dto.reference.ReferentialDtoReferenceDefinition;
import fr.ird.observe.dto.referential.ReferentialDto;
import fr.ird.observe.spi.DtoModelHelper;
import fr.ird.observe.spi.initializer.DtoReferencesInitializerSupport;
import fr.ird.observe.spi.map.ImmutableDtoMap;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * For a given of type {@link #type}, define all dependencies of referential set it need.
 * <p>
 * Means, for each property which is a reference to a dto, let's grab mark it, this make use possible then to download
 * <p>
 * in one request, all referential references required by this object.
 * Created on 11/11/15.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class FormDefinition<D extends IdDto> {

    private final Class<D> type;
    private final ImmutableMap<String, ReferentialDtoReferenceDefinition<?, ?>> properties;

    public static <D extends IdDto> Builder<D> builder(Class<D> type) {
        return new Builder<>(type, DtoModelHelper.getReferencesInitializer());
    }

    public ImmutableMap<String, ReferentialDtoReferenceDefinition<?, ?>> getProperties() {
        return properties;
    }

    FormDefinition(Class<D> type, ImmutableMap<String, ReferentialDtoReferenceDefinition<?, ?>> properties) {
        this.type = type;
        this.properties = properties;
    }

    public Class<D> getType() {
        return type;
    }

    public Set<Class<?>> getPropertiesTypes() {
        return properties.values().stream().map(DtoReferenceDefinition::getDtoType).collect(Collectors.toSet());
    }

    public Set<Class<?>> getPropertiesReferenceTypes() {
        return properties.values().stream().map(DtoReferenceDefinition::getType).collect(Collectors.toSet());
    }

    public static class Builder<D extends IdDto> {

        private final Class<D> type;
        private final DtoReferencesInitializerSupport referencesInitializer;

        private final ImmutableMap.Builder<String, ReferentialDtoReferenceDefinition<?, ?>> propertiesBuilder;

        public Builder(Class<D> type, DtoReferencesInitializerSupport referencesInitializer) {
            this.type = type;
            this.referencesInitializer = referencesInitializer;
            this.propertiesBuilder = ImmutableMap.builder();
        }

        public FormDefinition<D> build(ImmutableDtoMap.Builder<FormDefinition<? extends IdDto>> formDefinitionBuilder) {
            FormDefinition<D> definition = new FormDefinition<>(type, propertiesBuilder.build());
            formDefinitionBuilder.put(type, definition);
            return definition;
        }

        public <DD extends ReferentialDto, RR extends ReferentialDtoReference<DD, RR>> Builder<D> addProperty(String name, Class<RR> referenceType) {
            ReferentialDtoReferenceDefinition<DD, RR> definition = referenceSetDefinition(referenceType);
            propertiesBuilder.put(name, definition);
            return this;
        }

        @SuppressWarnings("unchecked")
        private <DD extends ReferentialDto, R extends ReferentialDtoReference<DD, R>> ReferentialDtoReferenceDefinition<DD, R> referenceSetDefinition(Class<R> referenceType) {
            ReferentialDtoReferenceDefinition referenceDefinition = referencesInitializer.getReferentialReferenceSetDefinition().get(referenceType);
            Objects.requireNonNull(referenceDefinition, "Could not find referenceSet for type: " + referenceType);
            return referenceDefinition;

        }

    }


}
