package fr.ird.observe.spi;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2008 - 2018 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.IdHelper;
import fr.ird.observe.dto.data.DataDto;
import fr.ird.observe.dto.form.FormDefinition;
import fr.ird.observe.dto.reference.DataDtoReference;
import fr.ird.observe.dto.reference.ReferentialDtoReference;
import fr.ird.observe.dto.referential.ReferentialDto;
import fr.ird.observe.spi.context.DataDtoContext;
import fr.ird.observe.spi.context.DataFormContext;
import fr.ird.observe.spi.context.ReferentialDtoContext;
import fr.ird.observe.spi.context.ReferentialFormContext;
import fr.ird.observe.spi.initializer.DtoFormsInitializerSupport;
import fr.ird.observe.spi.initializer.DtoReferencesInitializerSupport;
import fr.ird.observe.spi.map.ImmutableDtoMap;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Created by tchemit on 02/09/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("unchecked")
public class DtoModelHelper {

    /** Logger. */
    private static final Log log = LogFactory.getLog(DtoModelHelper.class);

    private static final DtoModelHelper INSTANCE = new DtoModelHelper();


    public static Set<Class<? extends ReferentialDto>> filterReferentialDto(Collection<Class<? extends IdDto>> types) {
        return (Set) types.stream().filter(ReferentialDto.class::isAssignableFrom).collect(Collectors.toSet());
    }


    public static Set<Class<? extends DataDto>> filterDataDto(Collection<Class<? extends IdDto>> types) {
        return (Set) types.stream().filter(DataDto.class::isAssignableFrom).collect(Collectors.toSet());
    }

    private final ImmutableDtoMap<ReferentialDtoContext> referentialDtoContext;
    private final ImmutableDtoMap<ReferentialFormContext> referentialFormContext;
    private final ImmutableDtoMap<DataDtoContext> dataDtoContext;
    private final ImmutableDtoMap<DataFormContext> dataFormContext;


    public static <D extends ReferentialDto, R extends ReferentialDtoReference<D, R>> ReferentialDtoContext<D, R> fromReferentialDto(Class<D> dtoType) {
        return INSTANCE.referentialDtoContext.get(dtoType);
    }

    public static <D extends DataDto, R extends DataDtoReference<D, R>> DataDtoContext<D, R> fromDataDto(Class<D> dtoType) {
        return INSTANCE.dataDtoContext.get(dtoType);
    }

    public static <D extends IdDto> Optional<FormDefinition<D>> getOptionalFormDefinition(Class<? extends IdDto> dtoType) {
        if (IdHelper.isReferential(dtoType)) {
            Optional<ReferentialFormContext> optional = Optional.ofNullable(DtoModelHelper.fromReferentialForm((Class) dtoType));
            return optional.map(ReferentialFormContext::toFormDefinition);
        }
        Optional<DataFormContext> optional = Optional.ofNullable(DtoModelHelper.fromDataForm((Class) dtoType));
        return optional.map(DataFormContext::toFormDefinition);
    }

    static <D extends ReferentialDto, F extends ReferentialDto> ReferentialFormContext<D, F> fromReferentialForm(Class<D> dtoType) {
        return INSTANCE.referentialFormContext.get(dtoType);
    }

    static <D extends DataDto, F extends DataDto> DataFormContext<D, F> fromDataForm(Class<D> dtoType) {
        return INSTANCE.dataFormContext.get(dtoType);
    }


    protected DtoModelHelper() {

        log.info("Dto model helper initialization  (" + this + ").");

        DtoReferencesInitializerSupport referencesInitializer = getReferencesInitializer();
        DtoFormsInitializerSupport formInitializer = getFormInitializer();

        ImmutableDtoMap.Builder<ReferentialDtoContext> referentialDtoContextBuilder = ImmutableDtoMap.builder();
        ImmutableDtoMap.Builder<ReferentialFormContext> referentialFormContextBuilder = ImmutableDtoMap.builder();
        ImmutableDtoMap.Builder<DataDtoContext> dataDtoContextBuilder = ImmutableDtoMap.builder();
        ImmutableDtoMap.Builder<DataFormContext> dataFormContextBuilder = ImmutableDtoMap.builder();

        Collection<Class<? extends IdDto>> dtoTypesWithReference = referencesInitializer.getReferenceToDtoClassMapping().values();

        int referentialCount = 0;

        for (Class<? extends ReferentialDto> dtoType : referencesInitializer.getReferentialTypes()) {

            registerReferential(dtoType, referencesInitializer, formInitializer, referentialDtoContextBuilder, referentialFormContextBuilder);

            referentialCount++;
        }
        log.info("Load " + referentialCount + " referential definitions.");

        Collection<Class> dtoTypesWithForm = formInitializer.getDtoToFormClassMapping().keys();
        int dataCountWithReference = 0;
        int dataCountWithForm = 0;
        Set<Class<? extends DataDto>> dataTypes = new LinkedHashSet<>(referencesInitializer.getDataTypes());
        dataTypes.addAll((Collection) dtoTypesWithForm);

        for (Class<? extends DataDto> dtoType : dataTypes) {

            Class<? extends DataDto> formType = formInitializer.getDtoToFormClassMapping().forData(dtoType);

            if (dtoTypesWithForm.contains(formType)) {

                FormDefinition formDefinition = formInitializer.getFormDefinitions().get(formType);
                if (formDefinition != null) {
                    registerDataForm(dtoType, formType, formDefinition, dataFormContextBuilder);
                    dataCountWithForm++;
                }
            }

            if (referencesInitializer.getDtoToReferenceClassMapping().forData(dtoType) == null) {
                continue;
            }

            if (dtoTypesWithReference.contains(dtoType)) {
                registerDataWithReference(dtoType, referencesInitializer, dataDtoContextBuilder);
                dataCountWithReference++;
            } else {
                registerData(dtoType, referencesInitializer, dataDtoContextBuilder);
            }

        }

        log.info("Load " + dataCountWithReference + " reference definitions.");
        log.info("Load " + dataCountWithForm + " form definitions.");
        log.info("Load " + referencesInitializer.getReferentialBinders().size() + " dto referential binders.");
        log.info("Load " + referencesInitializer.getDataBinders().size() + " dto data binders.");

        this.referentialDtoContext = referentialDtoContextBuilder.build();
        this.referentialFormContext = referentialFormContextBuilder.build();
        this.dataDtoContext = dataDtoContextBuilder.build();
        this.dataFormContext = dataFormContextBuilder.build();

        log.info("Dto model helper is initialized (" + this + ").");
    }

    private static DtoFormsInitializerSupport FORM_INITIALIZER;

    private static DtoReferencesInitializerSupport REFERENCES_INITIALIZER;

    public static DtoReferencesInitializerSupport getReferencesInitializer() {
        if (REFERENCES_INITIALIZER == null) {
            REFERENCES_INITIALIZER = getService(DtoReferencesInitializerSupport.class);
        }
        return REFERENCES_INITIALIZER;
    }

    static DtoFormsInitializerSupport getFormInitializer() {
        if (FORM_INITIALIZER == null) {
            FORM_INITIALIZER = getService(DtoFormsInitializerSupport.class);
        }
        return FORM_INITIALIZER;
    }

    private static <D extends ReferentialDto> void registerReferential(Class<D> dtoType,
                                                                       DtoReferencesInitializerSupport referencesInitializer,
                                                                       DtoFormsInitializerSupport formInitializer,
                                                                       ImmutableDtoMap.Builder<ReferentialDtoContext> referentialDtoContextBuilder, ImmutableDtoMap.Builder<ReferentialFormContext> referentialFormContextBuilder) {

        referentialDtoContextBuilder.put(dtoType, new ReferentialDtoContext<>(dtoType, referencesInitializer));
        referentialFormContextBuilder.put(dtoType, new ReferentialFormContext(dtoType, dtoType, formInitializer.getFormDefinitions().get(dtoType)));

    }

    private static <D extends DataDto> void registerData(Class<D> dtoType, DtoReferencesInitializerSupport referencesInitializer, ImmutableDtoMap.Builder<DataDtoContext> dataDtoContextBuilder) {

        dataDtoContextBuilder.put(dtoType, new DataDtoContext<>(dtoType, referencesInitializer));

    }

    private static <D extends DataDto, F extends DataDto> void registerDataForm(Class<D> dtoType, Class<F> formType, FormDefinition<F> referencesInitializer, ImmutableDtoMap.Builder<DataFormContext> dataFormContextBuilder) {
        dataFormContextBuilder.put(dtoType, new DataFormContext<>(dtoType, formType, referencesInitializer));
    }

    private static <D extends DataDto> void registerDataWithReference(Class<D> dtoType, DtoReferencesInitializerSupport referencesInitializer, ImmutableDtoMap.Builder<DataDtoContext> dataDtoContextBuilder) {

        dataDtoContextBuilder.put(dtoType, new DataDtoContext<>(dtoType, referencesInitializer));
    }

    public static <S> S getService(Class<S> serviceType) {
        Iterator<S> iterator = ServiceLoader.load(serviceType).iterator();
        if (iterator.hasNext()) {

            return iterator.next();
        } else {
            throw new IllegalStateException("No instance found for " + serviceType.getName());
        }
    }
}
