package fr.ird.observe.dto.reference;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2008 - 2018 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import fr.ird.observe.dto.IdDto;
import java.io.Serializable;
import java.util.function.Function;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Describe the definition of a reference on a dto.
 * <p>
 * Created on 11/11/15.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public abstract class DtoReferenceDefinition<D extends IdDto, R extends DtoReference<D, R>> implements Serializable {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(DtoReferenceDefinition.class);

    private final Class<D> dtoType;
    private final Class<R> referenceType;
    private final ImmutableMap<String, Class<?>> properties;
    private final ImmutableMap<String, Function<R, ?>> functions;

    protected DtoReferenceDefinition(Class<D> dtoType, Class<R> referenceType, ImmutableMap<String, Class<?>> properties, ImmutableMap<String, Function<R, ?>> functions) {
        this.dtoType = dtoType;
        this.referenceType = referenceType;
        this.properties = properties;
        this.functions = functions;
    }

    public Class<R> getType() {
        return referenceType;
    }

    public Class<D> getDtoType() {
        return dtoType;
    }

    ImmutableSet<String> getPropertyNames() {
        return properties.keySet();
    }

    <O> Function<R, O> getPropertyFunction(String propertyName) {
        //noinspection unchecked
        return (Function<R, O>) functions.get(propertyName);
    }

    @Override
    public String toString() {
        MoreObjects.ToStringHelper toStringHelper = MoreObjects.toStringHelper(this)
                .add("referenceType", referenceType.getSimpleName());
        if (log.isDebugEnabled()) {
            toStringHelper.add("properties", properties);
        }
        return toStringHelper.toString();
    }

    public abstract static class Builder<D extends IdDto, R extends DtoReference<D, R>, X extends DtoReferenceDefinition<D, R>> {

        final Class<D> dtoType;
        final Class<R> referenceType;
        final ImmutableMap.Builder<String, Class<?>> propertiesBuilder;
        final ImmutableMap.Builder<String, Function<R, ?>> functionsBuilder;

        public Builder(Class<D> dtoType, Class<R> referenceType) {
            this.dtoType = dtoType;
            this.referenceType = referenceType;
            propertiesBuilder = ImmutableMap.builder();
            functionsBuilder = ImmutableMap.builder();
        }

        public <O> Builder<D, R, X> addProperty(Class<O> type, String name, Function<R, O> function) {
            propertiesBuilder.put(name, type);
            functionsBuilder.put(name, function);
            return this;
        }

        public Class<R> getType() {
            return referenceType;
        }

        public abstract X build();
    }

}
