package fr.ird.observe.dto;

/*
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2008 - 2018 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ird.observe.dto.reference.DtoReference;
import java.util.Collection;
import java.util.Objects;
import java.util.function.Predicate;

public class IdHelper {

    private static final String DTO_ROOT_PACKAGE = "fr.ird.observe.dto";
    private static final int DTO_ROOT_PACKAGE_LENGTH = DTO_ROOT_PACKAGE.length() + 1;
    private static final String DTO_REFERENTIAL_PACKAGE = DTO_ROOT_PACKAGE + ".referential";
    private static final String DTO_REFERENTIAL_SEINE_PACKAGE = DTO_REFERENTIAL_PACKAGE + ".seine";
    private static final String DTO_REFERENTIAL_LONGLINE_PACKAGE = DTO_REFERENTIAL_PACKAGE + ".longline";

    private static final String DTO_DATA_PACKAGE = DTO_ROOT_PACKAGE + ".data";
    private static final String DTO_DATA_SEINE_PACKAGE = DTO_DATA_PACKAGE + ".seine";
    private static final String DTO_DATA_LONGLINE_PACKAGE = DTO_DATA_PACKAGE + ".longline";

    private static final String ENTITY_ROOT_PACKAGE = "fr.ird.observe.entities";
    private static final int ENTITY_ROOT_PACKAGE_LENGTH = ENTITY_ROOT_PACKAGE.length() + 1;
    private static final String ENTITY_REFERENTIAL_PACKAGE = ENTITY_ROOT_PACKAGE + ".referentiel";
    private static final String ENTITY_REFERENTIAL_SEINE_PACKAGE = ENTITY_REFERENTIAL_PACKAGE + ".seine";
    private static final String ENTITY_REFERENTIAL_LONGLINE_PACKAGE = ENTITY_REFERENTIAL_PACKAGE + ".longline";

    private static final String ENTITY_DATA_PACKAGE = ENTITY_ROOT_PACKAGE;
    private static final String ENTITY_DATA_SEINE_PACKAGE = ENTITY_DATA_PACKAGE + ".seine";
    private static final String ENTITY_DATA_LONGLINE_PACKAGE = ENTITY_DATA_PACKAGE + ".longline";

    public static boolean isReferential(Class type) {
        return isReferentialFromPackageName(type.getPackage().getName());
    }

    public static boolean isReferentialFromPackageName(String packageName) {
        return packageName.startsWith(DTO_REFERENTIAL_PACKAGE);
    }

    public static boolean isReferentialFromClassName(String packageName) {
        return packageName.startsWith(DTO_REFERENTIAL_PACKAGE);
    }

    public static boolean isReferentialSeineFromPackageName(String packageName) {
        return DTO_REFERENTIAL_SEINE_PACKAGE.equals(packageName);
    }

    public static boolean isReferentialLonglineFromPackageName(String packageName) {
        return DTO_REFERENTIAL_LONGLINE_PACKAGE.equals(packageName);
    }

    public static boolean isReferentialSeineFromClassName(String packageName) {
        return DTO_REFERENTIAL_SEINE_PACKAGE.startsWith(packageName);
    }

    public static boolean isReferentialLonglineFromClassName(String packageName) {
        return DTO_REFERENTIAL_LONGLINE_PACKAGE.startsWith(packageName);
    }

    public static boolean isData(Class type) {
        return isDataFromPackageName(type.getPackage().getName());
    }

    public static boolean isDataFromPackageName(String packageName) {
        return packageName.startsWith(DTO_DATA_PACKAGE);
    }

    public static boolean isDataSeineFromPackageName(String packageName) {
        return DTO_DATA_SEINE_PACKAGE.equals(packageName);
    }

    public static boolean isDataLonglineFromPackageName(String packageName) {
        return DTO_DATA_LONGLINE_PACKAGE.equals(packageName);
    }

    public static boolean isDataFromClassName(String className) {
        return DTO_DATA_PACKAGE.startsWith(className);
    }

    public static boolean isDataSeineFromClassName(String className) {
        return DTO_DATA_SEINE_PACKAGE.startsWith(className);
    }

    public static boolean isDataLonglineFromClassName(String className) {
        return DTO_DATA_LONGLINE_PACKAGE.startsWith(className);
    }

    public static String getDtoSimplifiedName(Class<?> dataType) {
        Objects.requireNonNull(dataType, " Null type given");
        return cleanId(dataType.getName().substring(DTO_ROOT_PACKAGE_LENGTH));
    }

    public static String getEntitySimplifiedName(Class<?> dataType) {
        Objects.requireNonNull(dataType, " Null type given");
        return cleanId(dataType.getName().substring(ENTITY_ROOT_PACKAGE_LENGTH));
    }
    public static String getEntitySimplifiedName(String  dataType) {
        Objects.requireNonNull(dataType, " Null type given");
        return cleanId(dataType.substring(ENTITY_ROOT_PACKAGE_LENGTH));
    }

    public static String getDtoShortName(Class<?> dataType) {
        Objects.requireNonNull(dataType, " Null type given");
        return cleanId(dataType);
    }

    public static String cleanId(Class id) {
        return cleanId(id.getSimpleName());
    }

    public static boolean equalsCleanId(Class one, Class two) {
        return IdHelper.cleanId(one.getSimpleName()).equals(IdHelper.cleanId(two.getSimpleName()));
    }

    public static String cleanId(String id) {
        if (id.endsWith("Dto")) {
            return id.substring(0, id.length() - 3);
        }
        if (id.endsWith("Reference")) {
            return id.substring(0, id.length() - 9);
        }
        // for entity, hibernate proxy gives use some class name like Entity_xxx
        int index = id.indexOf("_");
        if (index > -1) {
            id = id.substring(0, index);
        } else {
            // for entity, hibernate
            index = id.indexOf("@");
            if (index > -1) {
                id = id.substring(0, index);
            }
        }
        if (id.endsWith("Impl")) {
            return id.substring(0, id.length() - 4);
        }
        return id;
    }

    public static <BeanType extends IdDto> BeanType findById(Collection<BeanType> source, String id) {
        return source.stream().filter(newIdPredicate(id)::test).findFirst().orElse(null);
    }

    public static <BeanType extends IdDto> boolean exists(Collection<BeanType> source, String id) {
        return source.stream().anyMatch(newIdPredicate(id)::test);
    }

    public static boolean isProgramId(String id) {
        return id.contains("Program");
    }

    public static boolean isTrip(IdDto dto) {
        return isTripSeineId(dto.getId()) || isTripLonglineId(dto.getId());
    }

    public static boolean isTrip(DtoReference dto) {
        return isTripSeineId(dto.getId()) || isTripLonglineId(dto.getId());
    }

    public static boolean isTripLonglineId(String id) {
        return id.contains("TripLongline");
    }

    public static boolean isTripSeineId(String id) {
        return id.contains("TripSeine");
    }

    public static boolean isRouteId(String id) {
        return id.contains("Route");
    }

    public static boolean isActivitySeine(IdDto dto) {
        return isActivitySeineId(dto.getId());
    }

    public static boolean isActivitySeineId(String id) {
        return id.contains("ActivitySeine");
    }

    public static boolean isActivityLongline(IdDto dto) {
        return isActivityLonglineId(dto.getId());
    }

    public static boolean isActivityLonglineId(String id) {
        return id.contains("ActivityLongline");
    }

    public static boolean isSetSeine(IdDto dto) {
        return isSetSeineId(dto.getId());
    }

    public static boolean isSetSeineId(String id) {
        return id.contains("SetSeine");
    }

    public static boolean isSetLongline(IdDto dto) {
        return isSetLonglineId(dto.getId());
    }

    public static boolean isSetLonglineId(String id) {
        return id.contains("SetLongline");
    }

    public static boolean isFloatingObject(IdDto dto) {
        return isFloatingObjectId(dto.getId());
    }

    public static boolean isFloatingObjectId(String id) {
        return id.contains("FloatingObject");
    }

    public static boolean isSeineId(String id) {
        return id.contains("Seine");
    }

    public static boolean isLonglineId(String id) {
        return id.contains("Longline");
    }

    public static <BeanType extends IdDto> Predicate<BeanType> newIdPredicate(String id) {
        return o -> Objects.equals(id, o.getId());

    }
}
