package fr.ird.observe.dto.form;

/*-
 * #%L
 * ObServe Toolkit :: Common Dto
 * %%
 * Copyright (C) 2008 - 2017 IRD, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableMap;
import fr.ird.observe.dto.IdDto;
import fr.ird.observe.dto.IdHelper;
import fr.ird.observe.dto.reference.DtoReferenceDefinition;
import fr.ird.observe.dto.reference.ReferentialDtoReferenceSupport;
import fr.ird.observe.dto.referential.ReferentialDto;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * For a given of type {@link #type}, define all dependencies of referential set it need.
 * <p>
 * Means, for each property which is a reference to a dto, let's grab mark it, this make use possible then to download
 * <p>
 * in one request, all referential references required by this object.
 * Created on 11/11/15.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class FormDefinition<D extends IdDto> {

    private final String key;
    private final Class<D> type;
    private final ImmutableMap<String, DtoReferenceDefinition<?, ?>> properties;

    public static <D extends IdDto> Builder<D> builder(Class<D> type, ImmutableMap<String, DtoReferenceDefinition> referentialDtoReferenceDefinitions) {
        return new Builder<>(type, referentialDtoReferenceDefinitions);
    }

    public ImmutableMap<String, DtoReferenceDefinition<?, ?>> getProperties() {
        return properties;
    }

    FormDefinition(Class<D> type, ImmutableMap<String, DtoReferenceDefinition<?, ?>> properties) {
        this.type = type;
        this.properties = properties;
        this.key = IdHelper.cleanId(type.getName());
    }

    public Class<D> getType() {
        return type;
    }

    public String name() {
        return type.getName();
    }

    public Set<Class<?>> getPropertiesTypes() {
        return properties.values().stream().map(DtoReferenceDefinition::getDtoType).collect(Collectors.toSet());
    }

    public Set<Class<?>> getPropertiesReferenceTypes() {
        return properties.values().stream().map(DtoReferenceDefinition::getType).collect(Collectors.toSet());
    }

    public static class Builder<D extends IdDto> {

        private final Class<D> type;

        private final ImmutableMap<String, DtoReferenceDefinition> referentialDtoReferenceDefinitions;
        private final ImmutableMap.Builder<String, DtoReferenceDefinition<?, ?>> propertiesBuilder;

        public Builder(Class<D> type, ImmutableMap<String, DtoReferenceDefinition> referentialDtoReferenceDefinitions) {
            this.type = type;
            this.referentialDtoReferenceDefinitions = referentialDtoReferenceDefinitions;
            this.propertiesBuilder = ImmutableMap.builder();
        }

        public FormDefinition<D> build(ImmutableMap.Builder<String, FormDefinition<? extends IdDto>> formDefinitionBuilder) {
            FormDefinition<D> definition = new FormDefinition<>(type, propertiesBuilder.build());
            formDefinitionBuilder.put(type.getName().replace("Dto", ""), definition);
            return definition;
        }

        public <DD extends ReferentialDto, RR extends ReferentialDtoReferenceSupport<DD, RR>> Builder<D> addProperty(String name, Class<RR> referenceType) {
            DtoReferenceDefinition<DD, RR> definition = referenceSetDefinition(referenceType);
            propertiesBuilder.put(name, definition);
            return this;
        }

        private <DD extends ReferentialDto, R extends ReferentialDtoReferenceSupport<DD, R>> DtoReferenceDefinition<DD, R> referenceSetDefinition(Class<R> referenceType) {
            DtoReferenceDefinition referenceDefinition = referentialDtoReferenceDefinitions.get(IdHelper.cleanId(referenceType.getName()));
            Objects.requireNonNull(referenceDefinition, "Could not find referenceSet for type: " + referenceType);
            //noinspection unchecked
            return referenceDefinition;

        }

    }


}
