package eu.livotov.labs.android.robotools.settings;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.StringRes;
import android.text.TextUtils;

import eu.livotov.labs.android.robotools.crypt.RTDataCryptEngine;
import eu.livotov.labs.android.robotools.text.RTBase64;

/**
 * An encrypted analogue of RTPrefs.
 * All data will be encrypted with the target phone specific key (autogenerated) and decrypted "on the fly" when calling getXXX methods.
 */

public class RTSecurePrefs {
    private static RTSecurePrefs defaultPreferences;

    protected RTPrefs innerPrefs;
    private RTDataCryptEngine cryptEngine;

    /**
     * Creates non-transferable secure prefs instance with the default file name
     *
     * @param ctx
     */
    public RTSecurePrefs(final Context ctx) {
        innerPrefs = new RTPrefs(ctx, "defaultsecure", false);
        init(ctx, false);
    }

    /**
     * Creates non-transferable secure prefs object using custom file name
     *
     * @param ctx
     * @param preferenceStorageName file name to store properties in
     */
    public RTSecurePrefs(final Context ctx, final String preferenceStorageName) {
        innerPrefs = new RTPrefs(ctx, preferenceStorageName, false);
        init(ctx, false);
    }

    /**
     * Creates custom secure prefs object
     *
     * @param ctx
     * @param preferenceStorageName file name to store properties in
     * @param transferable          if set to <code>true</code>, this prefs file can be read on any phone (it will survive ap data backup and restore). When set to <code>false</code> - phone-tied encryption keys will be created and even app,
     *                              reinstalled on the same phone, will not be able to read the prefs file, created by a previous app installation. This is mainly important when you need to backup and restore your prefs using android backup service.
     */
    public RTSecurePrefs(final Context ctx, final String preferenceStorageName, boolean transferable) {
        innerPrefs = new RTPrefs(ctx, preferenceStorageName, true);
        init(ctx, transferable);
    }

    private void init(Context ctx, boolean transferable) {
        cryptEngine = new RTDataCryptEngine(ctx, transferable);
    }

    public static synchronized RTSecurePrefs getDefault(final Context ctx) {
        if (defaultPreferences == null) {
            defaultPreferences = new RTSecurePrefs(ctx);
        }

        return defaultPreferences;
    }

    public boolean contains(@StringRes int key) {
        return contains(innerPrefs.ctx.getString(key));
    }

    public boolean contains(String key) {
        return !TextUtils.isEmpty(innerPrefs.getString(key, null));
    }

    public String getString(@StringRes int key, final String defaultValue) {
        return getString(innerPrefs.ctx.getString(key), defaultValue);
    }

    public synchronized String getString(@NonNull String key, final String defaultValue) {
        try {

            final String encryptedValue = innerPrefs.getString(key, defaultValue);

            if (TextUtils.isEmpty(encryptedValue)
                    || encryptedValue.equalsIgnoreCase(defaultValue)) {
                return encryptedValue;
            }

            return cryptEngine.decryptString(encryptedValue);
        } catch (Throwable err) {
            reset();
            return null;
        }
    }

    public int getInt(@StringRes int key, int defaultValue) {
        try {
            return Integer.parseInt(getString(key, "" + defaultValue));
        } catch (Throwable err) {
            return defaultValue;
        }
    }

    public int getInt(@NonNull String key, int defaultValue) {
        try {
            return Integer.parseInt(getString(key, "" + defaultValue));
        } catch (Throwable err) {
            return defaultValue;
        }
    }

    public void setInt(@StringRes int key, int value) {
        setString(key, "" + value);
    }

    public void setString(@StringRes int key, String value) {
        setString(innerPrefs.ctx.getString(key), value);
    }

    public synchronized void setString(@NonNull String key, String value) {
        if (TextUtils.isEmpty(value)) {
            innerPrefs.remove(key);
        } else {
            try {
                innerPrefs.setString(key, cryptEngine.encryptString(value));
            } catch (Throwable throwable) {
                reset();
            }
        }
    }

    /**
     * Completely resets the keychain with all data and keys. After calling this method
     * the keychain becomes empty
     */
    public void reset() {
        innerPrefs.clear();
        cryptEngine.reset();
    }

    public void setInt(@NonNull String key, int value) {
        setString(key, "" + value);
    }

    public long getLong(@StringRes int key, long defaultValue) {
        try {
            return Long.parseLong(getString(key, "" + defaultValue));
        } catch (Throwable err) {
            return defaultValue;
        }
    }

    public long getLong(@NonNull String key, long defaultValue) {
        try {
            return Long.parseLong(getString(key, "" + defaultValue));
        } catch (Throwable err) {
            return defaultValue;
        }
    }

    public void setIntArray(@StringRes int key, int[] array) {
        setString(key, RTPrefs.arrayToString(array));
    }

    public void setIntArray(@NonNull String key, int[] array) {
        setString(key, RTPrefs.arrayToString(array));
    }

    public void setLongArray(@StringRes int key, long[] array) {
        setString(key, RTPrefs.arrayToString(array));
    }

    public void setLongArray(@NonNull String key, long[] array) {
        setString(key, RTPrefs.arrayToString(array));
    }

    public void setByteArray(@StringRes int key, byte[] array) {
        setString(key, RTBase64.encodeToString(array, RTBase64.NO_WRAP));
    }

    public void setByteArray(@NonNull String key, byte[] array) {
        setString(key, RTBase64.encodeToString(array, RTBase64.NO_WRAP));
    }

    public int[] getIntArray(@StringRes int key) {
        return RTPrefs.stringToIntegerArray(getString(key, ""));
    }

    public int[] getIntArray(@NonNull String key) {
        return RTPrefs.stringToIntegerArray(getString(key, ""));
    }

    public long[] getLongArray(@StringRes int key) {
        return innerPrefs.stringToLongArray(getString(key, ""));
    }

    public long[] getLongArray(@NonNull String key) {
        return RTPrefs.stringToLongArray(getString(key, ""));
    }

    public byte[] getByteArray(@StringRes int key) {
        return RTBase64.decode(getString(key, ""), RTBase64.NO_WRAP);
    }

    public byte[] getByteArray(@NonNull String key) {
        return RTBase64.decode(getString(key, ""), RTBase64.NO_WRAP);
    }

    public void setLong(@StringRes int key, long value) {
        setString(key, "" + value);
    }

    public void setLong(@NonNull String key, long value) {
        setString(key, "" + value);
    }

    public void setDouble(@StringRes int key, double value) {
        setString(key, "" + value);
    }

    public void setDouble(String key, double value) {
        setString(key, "" + value);
    }

    public double getDouble(@StringRes int key, double defaultValue) {
        try {
            return Double.parseDouble(getString(key, "" + defaultValue));
        } catch (Throwable err) {
            return defaultValue;
        }
    }

    public double getDouble(@NonNull String key, double defaultValue) {
        try {
            return Double.parseDouble(getString(key, "" + defaultValue));
        } catch (Throwable err) {
            return defaultValue;
        }
    }

    public boolean getBoolean(@StringRes int key, boolean defaultValue) {
        try {
            return "1".equals(getString(key, defaultValue ? "1" : "0"));
        } catch (Throwable err) {
            return defaultValue;
        }
    }

    public boolean getBoolean(@NonNull String key, boolean defaultValue) {
        try {
            return "1".equals(getString(key, defaultValue ? "1" : "0"));
        } catch (Throwable err) {
            return defaultValue;
        }
    }

    public void setBoolean(@StringRes int key, boolean value) {
        setString(key, value ? "1" : "0");
    }

    public void setBoolean(@NonNull String key, boolean value) {
        setString(key, value ? "1" : "0");
    }

    public <T> T getObject(Class<T> clazz, @StringRes int key, T defaultValue) {
        return getObject(clazz, innerPrefs.ctx.getString(key), defaultValue);
    }

    public <T> T getObject(Class<T> clazz, @NonNull String key, T defaultValue) {
        try {
            return innerPrefs.gson.fromJson(getString(key, ""), clazz);
        } catch (Throwable ignored) {
            return defaultValue;
        }
    }

    public void setObject(@StringRes int key, Object object) {
        setObject(innerPrefs.ctx.getString(key), object);
    }

    public void setObject(@NonNull String key, Object object) {
        try {
            if (object != null) {
                setString(key, innerPrefs.gson.toJson(object));
            } else {
                innerPrefs.remove(key);
            }
        } catch (Throwable err) {
            throw new IllegalArgumentException("Cannot convert to JSON: " + object.toString(), err);
        }
    }

}
