package io.embrace.android.embracesdk;

import androidx.annotation.Nullable;

import com.google.gson.annotations.SerializedName;

import java.util.ArrayList;
import java.util.List;

import java9.util.Lists;

class ThreadInfo {

    private final Long threadId;

    /**
     * The name of the thread.
     */
    @SerializedName("n")
    private final String name;

    /**
     * The priority of the thread
     */
    @SerializedName("p")
    private final Integer priority;

    /**
     * String representation of each line of the stack trace.
     */
    @SerializedName("tt")
    private final List<String> lines;

    /**
     * Thread state when the ANR is happening {@link Thread.State}
     */
    private final Thread.State state;

    ThreadInfo(long threadId, Thread.State state, String name, int priority, List<String> lines) {
        this.threadId = threadId;
        this.state = state;
        this.name = name;
        this.priority = priority;
        this.lines = Lists.copyOf(lines);
    }

    String getName() {
        return name;
    }

    int getPriority() {
        return priority;
    }

    List<String> getLines() {
        return lines;
    }

    public long getThreadId() {
        return threadId;
    }

    /**
     * Creates a {@link ThreadInfo} from the {@link Thread}, {@link StackTraceElement}[] pair,
     * using the thread name and priority, and each stacktrace element as each line.
     *
     * @param thread the exception
     * @return the stacktrace instance
     */
    static ThreadInfo ofThread(Thread thread, StackTraceElement[] stackTraceElements) {
        return ofThread(thread, stackTraceElements, null);
    }

    /**
     * Creates a {@link ThreadInfo} from the {@link Thread}, {@link StackTraceElement}[] pair,
     * using the thread name and priority, and each stacktrace element as each line with a limited length.
     *
     * @param thread           the exception
     * @param maxStacktraceSize the maximum length of a stacktrace
     * @return the stacktrace instance
     */
    static ThreadInfo ofThread(Thread thread, StackTraceElement[] stackTraceElements, Integer maxStacktraceSize) {
        String name = thread.getName();
        int priority = thread.getPriority();

        List<String> lines = new ArrayList<>();
        for (StackTraceElement line : stackTraceElements) {
            String stacktrace = line.toString();
            if (maxStacktraceSize != null && maxStacktraceSize != 0 && maxStacktraceSize <= stacktrace.length()) {
                stacktrace = stacktrace.substring(0, maxStacktraceSize);
            }

            lines.add(stacktrace);
        }

        return new ThreadInfo(thread.getId(), thread.getState(), name, priority, lines);
    }

    @Override
    public int hashCode() {
        int result = threadId != null ? threadId.hashCode() : 0;
        result = 31 * result + (name != null ? name.hashCode() : 0);
        result = 31 * result + (priority != null ? priority.hashCode() : 0);
        result = 31 * result + (lines != null ? lines.hashCode() : 0);
        result = 31 * result + (state != null ? state.hashCode() : 0);
        return result;
    }

    @Override
    public boolean equals(@Nullable Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ThreadInfo threadInfo = (ThreadInfo) o;

        if (threadId != null ? !threadId.equals(threadInfo.threadId) : threadInfo.threadId != null) {
            return false;
        }
        if (name != null ? !name.equals(threadInfo.name) : threadInfo.name != null) {
            return false;
        }
        if (priority != null ? !priority.equals(threadInfo.priority) : threadInfo.priority != null) {
            return false;
        }
        if (lines != null ? !lines.equals(threadInfo.lines) : threadInfo.lines != null) {
            return false;
        }
        return state != null ? state.equals(threadInfo.state) : threadInfo.state == null;
    }
}
