package io.embrace.android.embracesdk;

import android.util.Base64;

import com.google.gson.annotations.SerializedName;

import java.util.HashMap;
import java.util.Locale;
import java.util.UUID;

import io.embrace.android.embracesdk.network.NetworkCaptureData;

class NetworkCaptureCall {

    /**
     * UUID identifying the network request captured.
     */
    @SerializedName("id")
    private final String networkId;

    /**
     * Session ID that the network request occurred during.
     */
    @SerializedName("sid")
    private final String sessionId;

    /**
     * The start time of the request.
     */
    @SerializedName("st")
    private final Long startTime;

    /**
     * The end time of the request.
     */
    @SerializedName("et")
    private final Long endTime;

    /**
     * The duration of the network request.
     */
    @SerializedName("dur")
    private final Long duration;

    /**
     * The matched URL from the rule.
     */
    @SerializedName("mu")
    private final String matchedURL;

    /**
     * The URL being requested.
     */
    @SerializedName("url")
    private final String url;

    /**
     * The HTTP method the network request corresponds to.
     */
    @SerializedName("m")
    private final String httpMethod;

    /**
     * Contents of the body in a network request.
     */
    @SerializedName("sb")
    private final String responseBody;

    /**
     * UUID identifying the network request captured.
     */
    @SerializedName("sc")
    private final Integer responseStatus;

    /**
     * A dictionary containing the HTTP response headers.
     */
    @SerializedName("sh")
    private final HashMap<String, String> responseHeaders;

    /**
     * Response body size in bytes.
     */
    @SerializedName("sz")
    private final Integer responseSize;

    /**
     * Captured response body size in bytes.
     */
    @SerializedName("si")
    private final Integer responseBodySize;

    /**
     * Request body.
     */
    @SerializedName("qb")
    private final String requestBody;

    /**
     * The query string for the request, if present.
     */
    @SerializedName("qq")
    private final String requestQuery;

    /**
     * A dictionary containing the HTTP query headers.
     */
    @SerializedName("qh")
    private final HashMap<String, String> requestQueryHeaders;

    /**
     * Request body size in bytes.
     */
    @SerializedName("qz")
    private final Integer requestSize;

    /**
     * Captured request body size in bytes.
     */
    @SerializedName("qi")
    private final Integer requestBodySize;

    private NetworkCaptureCall(NetworkCaptureCall.Builder builder) {
        url = builder.url;
        httpMethod = builder.httpMethod;
        networkId = builder.requestId;
        sessionId = builder.sessionId;
        startTime = builder.startTime;
        endTime = builder.endTime;
        duration = builder.duration;
        matchedURL = builder.matchedURL;
        responseBody = builder.responseBody;
        responseStatus = builder.responseStatus;
        responseHeaders = builder.responseHeaders;
        responseSize = builder.responseSize;
        responseBodySize = builder.responseBodySize;
        requestQuery = builder.requestQueryParams;
        requestQueryHeaders = builder.requestQueryHeaders;
        requestBody = builder.requestBody;
        requestSize = builder.requestSize;
        requestBodySize = builder.requestBodySize;
    }

    static NetworkCaptureCall.Builder newBuilder() {
        return new NetworkCaptureCall.Builder();
    }

    static class Builder {
        private String requestId = createRequestId();
        private String sessionId;
        private Long startTime;
        private Long endTime;
        private Long duration;
        private String matchedURL;
        private String url;
        private String httpMethod;
        private String responseBody;
        private Integer responseStatus;
        private HashMap<String, String> responseHeaders;
        private Integer responseSize;
        private Integer responseBodySize;
        private String requestQueryParams;
        private HashMap<String, String> requestQueryHeaders;
        private String requestBody;
        private Integer requestSize;
        private Integer requestBodySize;

        private Builder() {
        }

        NetworkCaptureCall.Builder withUrl(String url) {
            this.url = url;
            return this;
        }

        NetworkCaptureCall.Builder withHttpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        NetworkCaptureCall.Builder withSessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        NetworkCaptureCall.Builder withStartTime(long startTime) {
            this.startTime = startTime;
            return this;
        }

        NetworkCaptureCall.Builder withEndTime(long endTime) {
            this.endTime = endTime;
            return this;
        }


        NetworkCaptureCall.Builder withDuration(long duration) {
            this.duration = duration;
            return this;
        }

        NetworkCaptureCall.Builder withMatchedURL(String matchedURL) {
            this.matchedURL = matchedURL;
            return this;
        }

        NetworkCaptureCall.Builder withResponseStatus(int responseStatus) {
            this.responseStatus = responseStatus;
            return this;
        }

        NetworkCaptureCall.Builder withNetworkCaptureData(NetworkCaptureData networkCaptureData) {
            this.requestQueryParams = networkCaptureData.getRequestQueryParams();
            this.requestQueryHeaders = (HashMap<String, String>) networkCaptureData.getRequestHeaders();
            this.requestBody = networkCaptureData.getCapturedRequestBody();
            this.responseHeaders = (HashMap<String, String>) networkCaptureData.getResponseHeaders();
            this.responseBody = networkCaptureData.getCapturedResponseBody();
            this.requestSize = networkCaptureData.getRequestBodySize();
            this.responseSize = networkCaptureData.getResponseBodySize();
            this.requestBodySize = networkCaptureData.getCapturedRequestBodySize();
            this.responseBodySize = networkCaptureData.getCapturedResponseBodySize();
            return this;
        }

        NetworkCaptureCall build() {
            return new NetworkCaptureCall(this);
        }

        private static String createRequestId() {
            return String.format("%.32s", Base64.encodeToString(UUID.randomUUID().toString().getBytes(), Base64.NO_WRAP)
                    .replace('+', '0')
                    .replace(' ', '0')).toUpperCase(Locale.getDefault());
        }
    }
}
