package io.embrace.android.embracesdk;

import com.google.gson.annotations.SerializedName;

/**
 * Intervals during which the UI thread was blocked for more than 1 second, which
 * determines that the application is not responding (ANR).
 */
final class AnrInterval {
    /**
     * The type of thread not responding. Currently only the UI thread is monitored.
     */
    enum Type {
        @SerializedName("ui")
        UI
    }

    /**
     * The time at which the application stopped responding.
     */
    @SerializedName("st")
    private final long startTime;

    /**
     * The last time the thread was alive.
     */
    @SerializedName("lk")
    private final long lastKnownTime;

    /**
     * The time the application started responding.
     */
    @SerializedName("en")
    private final Long endTime;

    /**
     * The component of the application which stopped responding.
     */
    @SerializedName("v")
    private final Type type;

    /**
     * The captured stacktraces of the anr interval.
     */
    @SerializedName("se")
    private final AnrStacktraces stacktraces;

    private AnrInterval(Builder builder) {
        this.startTime = builder.startTime;
        this.lastKnownTime = builder.lastKnownTime;
        this.endTime = builder.endTime;
        this.type = builder.type;
        this.stacktraces = builder.stacktraces;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static final class Builder {

        private long startTime;
        private long lastKnownTime;
        private Long endTime;
        private Type type;
        private AnrStacktraces stacktraces;

        private Builder() {
        }

        Builder withStartTime(long startTime) {
            this.startTime = startTime;
            return this;
        }

        Builder withLastKnownTime(long lastKnownTime) {
            this.lastKnownTime = lastKnownTime;
            return this;
        }

        Builder withEndTime(Long endTime) {
            this.endTime = endTime;
            return this;
        }

        Builder withType(Type type) {
            this.type = type;
            return this;
        }

        Builder withStacktraces(AnrStacktraces stacktraces) {
            this.stacktraces = stacktraces;
            return this;
        }

        public AnrInterval build() {
            return new AnrInterval(this);
        }
    }
}
