package io.embrace.android.embracesdk;

import android.os.Looper;

import com.fernandocejas.arrow.checks.Preconditions;

import java.io.Closeable;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

/**
 * Background worker which executes work on one or more threads.
 */
final class BackgroundWorker implements Closeable {
    private final ExecutorService executorService;

    BackgroundWorker(ExecutorService executorService) {
        this.executorService = Preconditions.checkNotNull(executorService, "executorService must not be null");
    }

    /**
     * Submit the task and wait for the result.
     * <p>
     * Retrieving the task will time out after 4 seconds if this is executed on the main thread.
     *
     * @param task the task to execute
     * @param <T> the return type of the task
     * @return the result of the task
     */
    <T> T submitAndGet(Callable<T> task) {
        try {
            Future<T> future = this.executorService.submit(task);
            if (Looper.getMainLooper() == Looper.myLooper()) {
                // Ensure that we don't block the main thread for more than 4 seconds
                return future.get(4L, TimeUnit.SECONDS);
            } else {
                return future.get();
            }
        } catch (Exception ex) {
            throw new RuntimeException("Failed to execute task", ex);
        }
    }

    /**
     * Submits a task to be executed on another thread.
     *
     * @param task the task to be executed
     * @param <T> the return type of the task
     * @return a future wrapping the result of the task
     */
    <T> Future<T> submit(final Callable<T> task) {
        return executorService.submit(task);
    }

    /**
     * Creates a worker with a single thread.
     *
     * @param workerName the name of the worker to use in the name of the thread
     * @return the worker
     */
    public static BackgroundWorker ofSingleThread(String workerName) {
        return new BackgroundWorker(createSingleThreadExecutor(workerName));
    }

    private static ExecutorService createSingleThreadExecutor(String name) {
        ExecutorService executor = Executors.newSingleThreadExecutor(WorkerUtils.createTheadFactory(name));
        WorkerUtils.shutdownHook(executor, name);
        return executor;
    }

    @Override
    public void close() {
        executorService.shutdown();
    }
}
