package io.embrace.android.embracesdk.network;

import com.fernandocejas.arrow.checks.Preconditions;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.Proxy;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;

/**
 * Custom implementation of URLStreamHandler that wraps a base URLStreamHandler and provides a context for executing
 * Embrace-specific logic.
 */
public abstract class EmbraceUrlStreamHandler extends URLStreamHandler {
    protected static final String METHOD_NAME_OPEN_CONNECTION = "openConnection";

    protected static final String MSG_ERROR_OPEN_CONNECTION =
            "An exception was thrown while attempting to open a connection";

    protected final URLStreamHandler handler;

    /**
     * Method that corresponds to URLStreamHandler.openConnection(URL).
     */
    private Method methodOpenConnection1;

    /**
     * Method that corresponds to URLStreamHandler.openConnection(URL, Proxy).
     */
    private Method methodOpenConnection2;

    /**
     * Given the base URLStreamHandler that will be wrapped, constructs the instance.
     */
    public EmbraceUrlStreamHandler(URLStreamHandler handler) {

        Preconditions.checkNotNull(handler, "Handler is null.");

        this.handler = handler;
        try {
            this.methodOpenConnection1 = getMethodOpenConnection(URL.class);
            this.methodOpenConnection2 = getMethodOpenConnection(URL.class, Proxy.class);
        } catch (NoSuchMethodException e) {
            throw new IllegalStateException("Failed to initialize EmbraceUrlStreamHandler instance.", e);
        }
    }

    @Override
    public abstract int getDefaultPort();

    /**
     * Given the URL class instance, returns the Java method that corresponds to the URLStreamHandler.openConnection(URL)
     * method.
     */
    protected abstract Method getMethodOpenConnection(Class<URL> url) throws NoSuchMethodException;

    /**
     * Given the URL class and Proxy class instances, returns the Java method that corresponds to the
     * URLStreamHandler.openConnection(URL, Proxy) method.
     */
    protected abstract Method getMethodOpenConnection(Class<URL> url, Class<Proxy> proxy) throws NoSuchMethodException;

    /**
     * Given an instance of URLConnection, returns a new URLConnection that wraps the provided instance with additional
     * Embrace-specific logic.
     */
    protected abstract URLConnection newEmbraceUrlConnection(URLConnection connection);

    @Override
    protected URLConnection openConnection(URL url) throws IOException {
        try {
            return newEmbraceUrlConnection((URLConnection) this.methodOpenConnection1.invoke(this.handler, url));
        } catch (Exception e) {
            // We catch Exception here instead of the specific exceptions that can be thrown due to a change in the way some
            // of these exceptions are compiled on different OS versions.

            throw new IOException(MSG_ERROR_OPEN_CONNECTION, e);
        }
    }

    @Override
    protected URLConnection openConnection(URL url, Proxy proxy) throws IOException {
        try {
            return newEmbraceUrlConnection((URLConnection) this.methodOpenConnection2.invoke(this.handler, url, proxy));
        } catch (Exception e) {
            // We catch Exception here instead of the specific exceptions that can be thrown due to a change in the way some
            // of these exceptions are compiled on different OS versions.

            throw new IOException(MSG_ERROR_OPEN_CONNECTION, e);
        }
    }
}
