package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.optional.Optional;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

/**
 * Retries failed API calls for a particular message type every {@value RETRY_WORKER_PERIOD} seconds.
 * <p>
 * This service is called by the {@link ApiClient} in the event that an API call fails.
 *
 * @param <T> the message type to retry
 */
class ApiClientRetryWorker<T> {

    private static final int RETRY_WORKER_PERIOD = 10;

    private final Class<T> clazz;

    private final CacheService cacheService;

    private final Queue<FailedApiCall<T>> failedApiCalls = new ConcurrentLinkedQueue<>();

    private final String cacheKey;

    private final ApiClient apiClient;

    private final ScheduledWorker worker;


    ApiClientRetryWorker(Class<T> clazz, CacheService cacheService, ApiClient apiClient, ScheduledWorker worker) {
        this.clazz = clazz;
        this.cacheService = cacheService;
        this.apiClient = apiClient;
        this.cacheKey = "network_failed_" + clazz.getSimpleName().toLowerCase(Locale.ENGLISH);
        this.worker = worker;
        start();
    }

    /**
     * Add a failed call to the failed list.
     *
     * @param request the API request to retry
     * @param payload the payload to retry
     */
    synchronized void addFailedCall(ApiRequest request, T payload) {
        failedApiCalls.add(new FailedApiCall<>(request, payload));
        List<FailedApiCall<T>> retries = StreamSupport.stream(failedApiCalls).collect(Collectors.toList());
        cacheService.cacheObject(cacheKey, new FailedCalls<>(retries), FailedCalls.class);
    }

    private void start() {
        // Load any cached failed API calls
        Optional<FailedCalls> optionalCalls = cacheService.loadObject(cacheKey, FailedCalls.class);
        if (optionalCalls.isPresent()) {
            failedApiCalls.addAll(optionalCalls.get().failedApiCalls);
        }
        
        // Retry any failed API calls every a number of seconds.
        worker.scheduleAtFixedRate(() -> {
            synchronized (this) {
                try {
                    List<FailedApiCall<T>> reattempts = new ArrayList<>();
                    FailedApiCall<T> lastCall = failedApiCalls.poll();
                    while (lastCall != null) {
                        reattempts.add(lastCall);
                        lastCall = failedApiCalls.poll();
                    }
                    cacheService.cacheObject(cacheKey, new FailedCalls<>(reattempts), FailedCalls.class);
                    for (FailedApiCall<T> apiCall : reattempts) {
                        apiClient.jsonPost(apiCall.apiRequest, apiCall.payload, clazz, this::addFailedCall);
                    }
                } catch (Exception ex) {
                    EmbraceLogger.logDebug("Error in ApiClient retry worker", ex);
                }
            }
        }, 0, RETRY_WORKER_PERIOD, TimeUnit.SECONDS);
    }

    private static class FailedApiCall<P> {
        private final ApiRequest apiRequest;

        private final P payload;

        FailedApiCall(ApiRequest apiRequest, P payload) {
            this.apiRequest = apiRequest;
            this.payload = payload;
        }
    }

    private static class FailedCalls<P> {
        private final List<FailedApiCall<P>> failedApiCalls;

        FailedCalls(List<FailedApiCall<P>> failedApiCalls) {
            this.failedApiCalls = failedApiCalls;
        }
    }

}
