package io.embrace.android.embracesdk;

import com.fernandocejas.arrow.checks.Preconditions;

import java.util.Arrays;
import java.util.HashSet;

import java9.util.stream.Collectors;
import java9.util.stream.StreamSupport;

import static io.embrace.android.embracesdk.EmbraceSessionService.APPLICATION_STATE_ACTIVE;
import static io.embrace.android.embracesdk.EmbraceSessionService.APPLICATION_STATE_BACKGROUND;

/**
 * Intercepts Embrace SDK's exceptions errors and forwards them to the Embrace API.
 */
final class EmbraceExceptionService {

    private final EmbraceActivityService activityService;

    private ConfigService configService;

    private ExceptionError exceptionError;

    // ignore network-related exceptions since they are expected
    private static final Class[] ignoredExceptionClasses = new Class[] {
            java.net.BindException.class,
            java.net.ConnectException.class,
            java.net.HttpRetryException.class,
            java.net.NoRouteToHostException.class,
            java.net.PortUnreachableException.class,
            java.net.ProtocolException.class,
            java.net.SocketException.class,
            java.net.SocketTimeoutException.class,
            java.net.UnknownHostException.class,
            java.net.UnknownServiceException.class,
    };
    private final HashSet ignoredExceptions = new HashSet<>(Arrays.asList(ignoredExceptionClasses));
    private final HashSet ignoredExceptionStrings = new HashSet<>(
            StreamSupport.stream(Arrays.asList(ignoredExceptionClasses))
                    .map(c -> c.getName())
                    .collect(Collectors.toList()));

    EmbraceExceptionService(EmbraceActivityService activityService) {
        this.activityService = Preconditions.checkNotNull(activityService);
        this.configService = null;
    }

    public void setConfigService(ConfigService configService) {
        this.configService = configService;
    }

    synchronized void handleExceptionError(Throwable throwable) {
        if (ignoredExceptions.contains(throwable.getClass())) {
            return;
        }

        // If the exception has been wrapped in another exception, the ignored exception name will
        // show up as the start of the message, delimited by a semicolon.
        String message = throwable.getMessage();
        if (message != null && ignoredExceptionStrings.contains(message.split(":")[0])) {
            return;
        }

        if (exceptionError == null) {
            exceptionError = new ExceptionError();
        }

        // if the config service has not been set yet, capture the exception
        if ((configService == null) || configService.isInternalExceptionCaptureEnabled()) {
            exceptionError.addException(
                    throwable,
                    getApplicationState());
        }
    }

    private String getApplicationState() {
        if (activityService.isInBackground()) {
            return APPLICATION_STATE_BACKGROUND;
        } else {
            return APPLICATION_STATE_ACTIVE;
        }
    }

    ExceptionError getCurrentExceptionError() {
        return exceptionError;
    }

    synchronized void resetExceptionErrorObject() {
        this.exceptionError = null;
    }
}
