/*
 * Copyright 2024 Sergio Belda
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package dev.sergiobelda.compose.vectorize.generator

import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.CodeBlock
import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.FunSpec
import com.squareup.kotlinpoet.KModifier
import com.squareup.kotlinpoet.MemberName
import com.squareup.kotlinpoet.PropertySpec
import com.squareup.kotlinpoet.buildCodeBlock
import dev.sergiobelda.compose.vectorize.generator.utils.setIndent
import dev.sergiobelda.compose.vectorize.generator.vector.StrokeCap
import dev.sergiobelda.compose.vectorize.generator.vector.StrokeJoin
import dev.sergiobelda.compose.vectorize.generator.vector.Vector
import dev.sergiobelda.compose.vectorize.generator.vector.VectorColor
import dev.sergiobelda.compose.vectorize.generator.vector.VectorNode
import dev.sergiobelda.compose.vectorize.generator.vector.VectorNode.Path.Companion.DefaultFillAlpha
import dev.sergiobelda.compose.vectorize.generator.vector.VectorNode.Path.Companion.DefaultFillType
import dev.sergiobelda.compose.vectorize.generator.vector.VectorNode.Path.Companion.DefaultStrokeAlpha
import dev.sergiobelda.compose.vectorize.generator.vector.VectorNode.Path.Companion.DefaultStrokeCap
import dev.sergiobelda.compose.vectorize.generator.vector.VectorNode.Path.Companion.DefaultStrokeLineJoin
import dev.sergiobelda.compose.vectorize.generator.vector.VectorNode.Path.Companion.DefaultStrokeLineMiter
import dev.sergiobelda.compose.vectorize.generator.vector.VectorNode.Path.Companion.DefaultStrokeWidth
import java.util.Locale

/**
 * Generator for creating a Kotlin source file with an ImageVector property for the given [vector],
 * with name [imageName].
 *
 * @param imageName the name for the generated property, which is also used for the generated file.
 * I.e if the name is `Menu`, the property will be `Menu` (inside a theme receiver object) and
 * the file will be `Menu.kt` (under the theme package name).
 * @param vector the parsed vector to generate ImageVector.Builder commands for
 */
class ImageVectorGenerator(
    private val imageName: String,
    private val imagePackageName: String,
    private val imageCategoryName: String,
    private val vector: Vector,
) {
    private var isComposable: Boolean = false

    /**
     * @return a [FileSpec] representing a Kotlin source file containing the property for this
     * programmatic [vector] representation.
     *
     * The package name and hence file location of the generated file is: [imagePackageName].
     */
    fun createFileSpec(): FileSpec {
        val builder = createFileSpecBuilder()
        val backingProperty = getBackingProperty()
        val propertySpecBuilder =
            PropertySpec.builder(name = imageName, type = ClassNames.ImageVector)
                .receiver(
                    ClassName(
                        imagePackageName,
                        "Images",
                        imageCategoryName,
                    ),
                )
                .getter(
                    imageGetter(
                        backingProperty = backingProperty,
                        imageName = imageName,
                    ),
                )
        builder.addProperty(propertySpecBuilder.build())
        builder.addProperty(backingProperty)
        return builder.setIndent().build()
    }

    private fun createFileSpecBuilder(): FileSpec.Builder {
        val imagesPackage =
            imagePackageName
                .plus(".")
                .plus(imageCategoryName.lowercase(Locale.ROOT))
        return FileSpec.builder(
            packageName = imagesPackage,
            fileName = imageName,
        )
    }

    private fun getBackingProperty(): PropertySpec {
        val backingPropertyName = "_" + imageName.replaceFirstChar { it.lowercase(Locale.ROOT) }
        return backingProperty(name = backingPropertyName)
    }

    private fun imageGetter(
        backingProperty: PropertySpec,
        imageName: String,
    ): FunSpec {
        val imageVectorCodeBlock = buildCodeBlock {
            val parameterList = listOfNotNull(
                "name = \"%N\"",
                "width = ${vector.width}f",
                "height = ${vector.height}f",
                "viewportWidth = ${vector.viewportWidth}f",
                "viewportHeight = ${vector.viewportHeight}f",
                "autoMirror = ${vector.autoMirror}",
            )
            val parameters = if (parameterList.isNotEmpty()) {
                parameterList.joinToString(
                    prefix = "%N = %M(\n\t",
                    postfix = "\n)",
                    separator = ",\n\t",
                )
            } else {
                ""
            }
            beginControlFlow(
                parameters,
                backingProperty,
                MemberNames.ImageVector,
                imageName,
            )
            vector.nodes.forEach { node ->
                addVectorNodeCode(node)
            }
            endControlFlow()
        }
        return FunSpec.getterBuilder().apply {
            if (!isComposable) {
                addCode(
                    buildCodeBlock {
                        beginControlFlow("if (%N != null)", backingProperty)
                        addStatement("return %N!!", backingProperty)
                        endControlFlow()
                    },
                )
            }
            addCode(imageVectorCodeBlock)
            addStatement("return %N!!", backingProperty)
            if (isComposable) {
                addAnnotation(AnnotationNames.Composable)
            }
        }.build()
    }

    private fun backingProperty(name: String): PropertySpec {
        val nullableImageVector = ClassNames.ImageVector.copy(nullable = true)
        return PropertySpec.builder(name = name, type = nullableImageVector)
            .mutable()
            .addModifiers(KModifier.PRIVATE)
            .initializer("null")
            .build()
    }

    /**
     * Recursively adds function calls to construct the given [vectorNode] and its children.
     */
    private fun CodeBlock.Builder.addVectorNodeCode(vectorNode: VectorNode) {
        when (vectorNode) {
            is VectorNode.Group -> {
                beginControlFlow("%M", MemberNames.Group)
                vectorNode.paths.forEach { path ->
                    addVectorNodeCode(path)
                }
                endControlFlow()
            }

            is VectorNode.Path -> {
                addPathCode(vectorNode) {
                    vectorNode.nodes.forEach { pathNode ->
                        addStatement(pathNode.asFunctionCall())
                    }
                }
            }
        }
    }

    /**
     * Adds a function call to create the given [path], with [pathBody] containing the commands for
     * the path.
     */
    private fun CodeBlock.Builder.addPathCode(
        path: VectorNode.Path,
        pathBody: CodeBlock.Builder.() -> Unit,
    ) {
        val parameterList = mutableListOf<String>()
        val memberList = mutableListOf<MemberName>()

        with(path) {
            memberList.add(MemberNames.Path)

            fillAlpha.takeIf { it != DefaultFillAlpha }?.let {
                parameterList.add("fillAlpha = ${it}f")
            }
            fillColor?.let {
                when (it) {
                    is VectorColor.Hexadecimal -> {
                        parameterList.add("fill = %M(%M(${it.processValue()}))")
                        memberList.add(MemberNames.SolidColor)
                        memberList.add(MemberNames.Color)
                    }

                    is VectorColor.Attribute -> {
                        parameterList.add("fill = %M(%M.${it.processValue()})")
                        memberList.add(MemberNames.SolidColor)
                        memberList.add(MemberNames.Material3ColorScheme)
                        isComposable = true
                    }
                }
            }
            fillType.takeIf { it != DefaultFillType }?.let {
                parameterList.add("pathFillType = %M")
                memberList.add(MemberNames.PathFillType.EvenOdd)
            }
            strokeAlpha.takeIf { it != DefaultStrokeAlpha }?.let {
                parameterList.add("strokeAlpha = ${it}f")
            }
            strokeColor?.let {
                when (it) {
                    is VectorColor.Hexadecimal -> {
                        parameterList.add("stroke = %M(%M(${it.processValue()}))")
                        memberList.add(MemberNames.SolidColor)
                        memberList.add(MemberNames.Color)
                    }

                    is VectorColor.Attribute -> {
                        parameterList.add("stroke = %M(%M.${it.processValue()})")
                        memberList.add(MemberNames.SolidColor)
                        memberList.add(MemberNames.Material3ColorScheme)
                        isComposable = true
                    }
                }
            }
            strokeCap.takeIf { it != DefaultStrokeCap }?.let {
                parameterList.add("strokeLineCap = %M")
                when (strokeCap) {
                    StrokeCap.Round -> memberList.add(MemberNames.StrokeCapType.Round)
                    StrokeCap.Square -> memberList.add(MemberNames.StrokeCapType.Square)
                    else -> memberList.add(MemberNames.StrokeCapType.Butt)
                }
            }
            strokeLineMiter.takeIf { it != DefaultStrokeLineMiter }?.let {
                parameterList.add("strokeLineMiter = ${strokeLineMiter}f")
            }
            strokeLineJoin.takeIf { it != DefaultStrokeLineJoin }?.let {
                parameterList.add("strokeLineJoin = %M")
                when (strokeLineJoin) {
                    StrokeJoin.Bevel -> memberList.add(MemberNames.StrokeJoinType.Bevel)
                    StrokeJoin.Round -> memberList.add(MemberNames.StrokeJoinType.Round)
                    else -> memberList.add(MemberNames.StrokeJoinType.Miter)
                }
            }
            strokeWidth.takeIf { it != DefaultStrokeWidth }?.let {
                parameterList.add("strokeLineWidth = ${strokeWidth}f")
            }
        }
        addPathParameters(parameterList, memberList)
        pathBody()
        endControlFlow()
    }

    private fun CodeBlock.Builder.addPathParameters(
        parameterList: List<String>,
        memberList: List<MemberName>,
    ) {
        val parameters = if (parameterList.isNotEmpty()) {
            parameterList.joinToString(
                prefix = "(\n\t",
                postfix = "\n)",
                separator = ",\n\t",
            )
        } else {
            ""
        }

        beginControlFlow("%M$parameters", args = memberList.toTypedArray())
    }

    private fun VectorColor.Attribute.processValue(): String =
        this.value.replace("?color", "").replaceFirstChar(Char::lowercase)

    private fun VectorColor.Hexadecimal.processValue(): String =
        this.value.replace("#", "0x")
}
