/*
 * Decompiled with CFR 0.152.
 */
package com.android.volley.cache;

import android.os.SystemClock;
import com.android.volley.Cache;
import com.android.volley.VolleyLog;
import com.android.volley.misc.IOUtils;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

public class DiskBasedCache
implements Cache {
    private final int CACHE_LOAD_THREADS = 2;
    private final CacheContainer mEntries = new CacheContainer();
    private final File mRootDirectory;
    private final int mMaxCacheSizeInBytes;
    private static final int DEFAULT_DISK_USAGE_BYTES = 0x500000;
    private static final float HYSTERESIS_FACTOR = 0.9f;
    private static final int CACHE_MAGIC = 538247942;

    public DiskBasedCache(File rootDirectory, int maxCacheSizeInBytes) {
        this.mRootDirectory = rootDirectory;
        this.mMaxCacheSizeInBytes = maxCacheSizeInBytes;
    }

    public DiskBasedCache(File rootDirectory) {
        this(rootDirectory, 0x500000);
    }

    @Override
    public synchronized void clear() {
        File[] files = this.mRootDirectory.listFiles();
        if (files != null) {
            for (File file : files) {
                file.delete();
            }
        }
        this.mEntries.clear();
        VolleyLog.d("Cache cleared.", new Object[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized Cache.Entry get(String key) {
        CacheHeader entry = this.mEntries.get(key);
        if (entry == null) {
            return null;
        }
        File file = this.getFileForKey(key);
        FilterInputStream cis = null;
        try {
            cis = new IOUtils.CountingInputStream(new BufferedInputStream(new FileInputStream(file)));
            CacheHeader.readHeader(cis);
            byte[] data = IOUtils.streamToBytes(cis, (int)(file.length() - ((IOUtils.CountingInputStream)cis).getBytesRead()));
            Cache.Entry entry2 = entry.toCacheEntry(data);
            return entry2;
        }
        catch (IOException e) {
            VolleyLog.d("%s: %s", file.getAbsolutePath(), e.toString());
            this.remove(key);
            Cache.Entry entry3 = null;
            return entry3;
        }
        catch (OutOfMemoryError e) {
            VolleyLog.e("Caught OOM for %d byte image, path=%s: %s", file.length(), file.getAbsolutePath(), e.toString());
            Cache.Entry entry4 = null;
            return entry4;
        }
        finally {
            if (cis != null) {
                try {
                    cis.close();
                }
                catch (IOException ioe) {
                    return null;
                }
            }
        }
    }

    @Override
    public synchronized void initialize() {
        this.mEntries.initialize();
    }

    @Override
    public synchronized void invalidate(String key, boolean fullExpire) {
        Cache.Entry entry = this.get(key);
        if (entry != null) {
            entry.softTtl = -1L;
            if (fullExpire) {
                entry.ttl = -1L;
            }
            this.put(key, entry);
        }
    }

    @Override
    public synchronized void put(String key, Cache.Entry entry) {
        this.pruneIfNeeded(entry.data.length);
        File file = this.getFileForKey(key);
        try {
            BufferedOutputStream fos = new BufferedOutputStream(new FileOutputStream(file));
            CacheHeader e = new CacheHeader(key, entry);
            boolean success = e.writeHeader(fos);
            if (!success) {
                fos.close();
                VolleyLog.d("Failed to write header for %s", file.getAbsolutePath());
                throw new IOException();
            }
            fos.write(entry.data);
            fos.close();
            this.putEntry(key, e);
            return;
        }
        catch (IOException e) {
            boolean deleted = file.delete();
            if (!deleted) {
                VolleyLog.d("Could not clean up file %s", file.getAbsolutePath());
            }
            return;
        }
    }

    @Override
    public synchronized void remove(String key) {
        boolean deleted = this.getFileForKey(key).delete();
        this.removeEntry(key);
        if (!deleted) {
            VolleyLog.d("Could not delete cache entry for key=%s, filename=%s", key, this.getFilenameForKey(key));
        }
    }

    private String getFilenameForKey(String key) {
        int firstHalfLength = key.length() / 2;
        String localFilename = String.valueOf(key.substring(0, firstHalfLength).hashCode());
        localFilename = localFilename + String.valueOf(key.substring(firstHalfLength).hashCode());
        return localFilename;
    }

    public File getFileForKey(String key) {
        return new File(this.mRootDirectory, this.getFilenameForKey(key));
    }

    private void pruneIfNeeded(int neededSpace) {
        if (!this.mEntries.isLoaded()) {
            return;
        }
        if (this.mEntries.getTotalSize() + (long)neededSpace < (long)this.mMaxCacheSizeInBytes) {
            return;
        }
        if (VolleyLog.DEBUG) {
            VolleyLog.v("Pruning old cache entries.", new Object[0]);
        }
        long before = this.mEntries.getTotalSize();
        int prunedFiles = 0;
        long startTime = SystemClock.elapsedRealtime();
        Iterator iterator = this.mEntries.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry entry = iterator.next();
            CacheHeader e = (CacheHeader)entry.getValue();
            boolean deleted = this.getFileForKey(e.key).delete();
            if (!deleted) {
                VolleyLog.d("Could not delete cache entry for key=%s, filename=%s", e.key, this.getFilenameForKey(e.key));
            }
            iterator.remove();
            ++prunedFiles;
            if (!((float)(this.mEntries.getTotalSize() + (long)neededSpace) < (float)this.mMaxCacheSizeInBytes * 0.9f)) continue;
            break;
        }
        if (VolleyLog.DEBUG) {
            VolleyLog.v("pruned %d files, %d bytes, %d ms", prunedFiles, this.mEntries.getTotalSize() - before, SystemClock.elapsedRealtime() - startTime);
        }
    }

    private void putEntry(String key, CacheHeader entry) {
        this.mEntries.put(key, entry);
    }

    private void removeEntry(String key) {
        CacheHeader entry = this.mEntries.get(key);
        if (entry != null) {
            this.mEntries.remove(key);
        }
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() {
    }

    static class CacheHeader {
        public long size;
        public String key;
        public String etag;
        public long serverDate;
        public long lastModified;
        public long ttl;
        public long softTtl;
        public Map<String, String> responseHeaders;

        private CacheHeader() {
        }

        public CacheHeader(String key, Cache.Entry entry) {
            this.key = key;
            this.size = entry.data.length;
            this.etag = entry.etag;
            this.serverDate = entry.serverDate;
            this.lastModified = entry.lastModified;
            this.ttl = entry.ttl;
            this.softTtl = entry.softTtl;
            this.responseHeaders = entry.responseHeaders;
        }

        public static CacheHeader readHeader(InputStream is) throws IOException {
            CacheHeader entry = new CacheHeader();
            int magic = IOUtils.readInt(is);
            if (magic != 538247942) {
                throw new IOException();
            }
            entry.key = IOUtils.readString(is);
            entry.etag = IOUtils.readString(is);
            if (entry.etag.equals("")) {
                entry.etag = null;
            }
            entry.serverDate = IOUtils.readLong(is);
            entry.lastModified = IOUtils.readLong(is);
            entry.ttl = IOUtils.readLong(is);
            entry.softTtl = IOUtils.readLong(is);
            entry.responseHeaders = IOUtils.readStringStringMap(is);
            return entry;
        }

        public Cache.Entry toCacheEntry(byte[] data) {
            Cache.Entry e = new Cache.Entry();
            e.data = data;
            e.etag = this.etag;
            e.serverDate = this.serverDate;
            e.lastModified = this.lastModified;
            e.ttl = this.ttl;
            e.softTtl = this.softTtl;
            e.responseHeaders = this.responseHeaders;
            return e;
        }

        public boolean writeHeader(OutputStream os) {
            try {
                IOUtils.writeInt(os, 538247942);
                IOUtils.writeString(os, this.key);
                IOUtils.writeString(os, this.etag == null ? "" : this.etag);
                IOUtils.writeLong(os, this.serverDate);
                IOUtils.writeLong(os, this.lastModified);
                IOUtils.writeLong(os, this.ttl);
                IOUtils.writeLong(os, this.softTtl);
                IOUtils.writeStringStringMap(this.responseHeaders, os);
                os.flush();
                return true;
            }
            catch (IOException e) {
                VolleyLog.d("%s", e.toString());
                return false;
            }
        }
    }

    private class CacheContainer
    extends ConcurrentHashMap<String, CacheHeader> {
        private final PriorityBlockingQueue<Runnable> mQueue;
        private final Map<String, Future<CacheHeader>> mLoadingFiles;
        private AtomicLong mTotalSize;
        private boolean mInitialized;

        public CacheContainer() {
            super(16, 0.75f, 2);
            this.mQueue = new PriorityBlockingQueue();
            this.mLoadingFiles = new ConcurrentHashMap<String, Future<CacheHeader>>();
            this.mTotalSize = new AtomicLong(0L);
            this.mInitialized = false;
        }

        public synchronized void initialize() {
            if (this.mInitialized) {
                return;
            }
            this.mInitialized = true;
            if (!DiskBasedCache.this.mRootDirectory.exists()) {
                if (!DiskBasedCache.this.mRootDirectory.mkdirs()) {
                    VolleyLog.e("Unable to create cache dir %s", DiskBasedCache.this.mRootDirectory.getAbsolutePath());
                }
                return;
            }
            File[] files = DiskBasedCache.this.mRootDirectory.listFiles();
            if (files == null) {
                return;
            }
            VolleyLog.d("Loading %d files from cache", files.length);
            ThreadPoolExecutor executor = new ThreadPoolExecutor(2, 2, 10L, TimeUnit.MILLISECONDS, this.mQueue);
            for (File file : files) {
                HeaderParserCallable callable = new HeaderParserCallable(file);
                ReorderingFutureTask submit = new ReorderingFutureTask(callable);
                this.mLoadingFiles.put(file.getName(), submit);
                executor.execute(submit);
            }
        }

        private void waitForCache() {
            while (this.mLoadingFiles.size() > 0) {
                Iterator<Map.Entry<String, Future<CacheHeader>>> iterator = this.mLoadingFiles.entrySet().iterator();
                if (!iterator.hasNext()) continue;
                Map.Entry<String, Future<CacheHeader>> entry = iterator.next();
                try {
                    entry.getValue().get();
                }
                catch (InterruptedException ignored) {
                }
                catch (ExecutionException executionException) {}
            }
        }

        private void waitForKey(Object key) {
            if (this.isLoaded()) {
                return;
            }
            String filename = DiskBasedCache.this.getFilenameForKey((String)key);
            Future<CacheHeader> future = this.mLoadingFiles.get(filename);
            if (future != null) {
                try {
                    future.get();
                }
                catch (InterruptedException ignored) {
                }
                catch (ExecutionException ignored) {
                    // empty catch block
                }
            }
        }

        public boolean isLoaded() {
            return this.mLoadingFiles.size() == 0;
        }

        public long getTotalSize() {
            return this.mTotalSize.get();
        }

        @Override
        public CacheHeader get(Object key) {
            this.waitForKey(key);
            return (CacheHeader)super.get(key);
        }

        @Override
        public boolean containsKey(Object key) {
            this.waitForKey(key);
            return super.containsKey(key);
        }

        @Override
        public CacheHeader put(String key, CacheHeader entry) {
            this.waitForKey(key);
            if (super.containsKey(key)) {
                this.mTotalSize.getAndAdd(entry.size - ((CacheHeader)super.get((Object)key)).size);
            } else {
                this.mTotalSize.getAndAdd(entry.size);
            }
            return super.put(key, entry);
        }

        @Override
        public CacheHeader remove(Object key) {
            this.waitForKey(key);
            if (super.containsKey(key)) {
                this.mTotalSize.getAndAdd(-1L * ((CacheHeader)super.get((Object)key)).size);
            }
            return (CacheHeader)super.remove(key);
        }

        @Override
        public void clear() {
            this.waitForCache();
            this.mTotalSize.getAndSet(0L);
            super.clear();
        }

        private class HeaderParserCallable
        implements Callable<CacheHeader> {
            private final File file;

            public HeaderParserCallable(File file) {
                this.file = file;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public CacheHeader call() throws Exception {
                BufferedInputStream fis = null;
                try {
                    fis = new BufferedInputStream(new FileInputStream(this.file));
                    CacheHeader entry = CacheHeader.readHeader(fis);
                    entry.size = this.file.length();
                    CacheContainer.super.put(entry.key, entry);
                    CacheContainer.this.mTotalSize.getAndAdd(entry.size);
                    CacheHeader cacheHeader = entry;
                    return cacheHeader;
                }
                catch (IOException e) {
                    if (this.file != null) {
                        this.file.delete();
                    }
                }
                finally {
                    try {
                        if (fis != null) {
                            fis.close();
                        }
                    }
                    catch (IOException iOException) {}
                    CacheContainer.this.mLoadingFiles.remove(this.file.getName());
                }
                return null;
            }
        }

        private class ReorderingFutureTask
        extends FutureTask<CacheHeader>
        implements Comparable<ReorderingFutureTask> {
            private int mGetRequests;

            public ReorderingFutureTask(Callable<CacheHeader> callable) {
                super(callable);
                this.mGetRequests = 0;
            }

            @Override
            public CacheHeader get() throws InterruptedException, ExecutionException {
                ++this.mGetRequests;
                if (CacheContainer.this.mQueue.contains(this)) {
                    CacheContainer.this.mQueue.remove(this);
                    CacheContainer.this.mQueue.add(this);
                }
                return (CacheHeader)super.get();
            }

            @Override
            public int compareTo(ReorderingFutureTask another) {
                return this.mGetRequests > another.mGetRequests ? -1 : (this.mGetRequests < another.mGetRequests ? 1 : 0);
            }
        }
    }
}

