package de.wenzlaff.blockchain;

import java.io.IOException;
import java.math.BigDecimal;
import java.net.ConnectException;
import java.util.concurrent.ExecutionException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.web3j.crypto.Credentials;
import org.web3j.crypto.WalletUtils;
import org.web3j.protocol.Web3j;
import org.web3j.protocol.core.DefaultBlockParameterName;
import org.web3j.protocol.core.methods.response.EthGasPrice;
import org.web3j.protocol.core.methods.response.TransactionReceipt;
import org.web3j.protocol.core.methods.response.Web3ClientVersion;
import org.web3j.protocol.http.HttpService;
import org.web3j.utils.Convert;
import org.web3j.utils.Convert.Unit;

/**
 * Klasse zum abfragen des Bestands einer Adresse auf der ETH Blockchain und
 * versenden von Eth von einem File Wallet.
 * 
 * https://docs.web3j.io/4.8.7/
 * 
 * @author Thomas Wenzlaff
 *
 */
public class Transfer {

	// Ganache RPC SERVER URL
	private static final String BLOCKCHAIN_SERVER_URL = "http://127.0.0.1:7545";

	private static final String ZIEL_JSON_ADDRSSE = "0x1486df323a85999fb6a540492dc8725c8ecdfb9b";
	// Ganache Address z.B. die 1. aus der Liste
	private static final String ZIEL_ADDRESSE = ZIEL_JSON_ADDRSSE; // "0xC87b5463c07b6064754fD2b6bdb17774977B745b";

	// das gleich Passwort das auch in der Klasse LocalWallet verwendet wurde
	private static final String WALLET_PASSWORD = "password";

	private static final Logger LOG = LogManager.getLogger(Transfer.class);

	public static void main(String[] args) throws Exception {

		Transfer transfer = new Transfer();
		Web3j web3 = transfer.connect(BLOCKCHAIN_SERVER_URL);
		LOG.info("Ziel Addresse");
		transfer.printBalance(web3, ZIEL_ADDRESSE);

		Credentials credentials = WalletUtils.loadCredentials(WALLET_PASSWORD, "UTC--2021-09-25T15-09-47.203000000Z--8e031f1c47537910fe104936dcd0f77f8a97c6f1.json");

		LOG.info("Von Wallet Adresse");
		transfer.printBalance(web3, credentials.getAddress());

		double ueberweiseBetragInEther = 0.015;
		transfer.send(web3, credentials, ZIEL_ADDRESSE, ueberweiseBetragInEther);
	}

	private void send(Web3j web3, Credentials credentials, String zielAdresse, double ether) throws Exception {
		LOG.info("Send " + ether + " Ether  = " + Convert.toWei(BigDecimal.valueOf(ether), Unit.ETHER) + " WEI");
		TransactionReceipt transactionReceipt = org.web3j.tx.Transfer.sendFunds(web3, credentials, zielAdresse, BigDecimal.valueOf(ether), Convert.Unit.ETHER).send();
		LOG.info("Betrag von Adresse: " + transactionReceipt.getFrom());
		LOG.info("In Block Nummer   : " + transactionReceipt.getBlockNumber());
		printBalance(web3, zielAdresse);
	}

	private Web3j connect(String blockchainServerUrl) {
		Web3j web3 = Web3j.build(new HttpService(BLOCKCHAIN_SERVER_URL));
		LOG.info("OK, verbunden mit Blockchain " + blockchainServerUrl + " ...");
		return web3;
	}

	private void printBalance(Web3j web3, String addresse) throws IOException, InterruptedException, ExecutionException {

		Web3ClientVersion clientVersion = null;
		try {
			clientVersion = web3.web3ClientVersion().send();
		} catch (ConnectException e) {
			LOG.error("Die Blockchain ist nicht erreichbar. Starten vergessen? Oder URL oder Port falsch?");
			throw e;

		}
		LOG.debug("Client Version: " + clientVersion.getWeb3ClientVersion());

		EthGasPrice gasPrice = web3.ethGasPrice().send();
		LOG.debug("Default Gas Preis: " + gasPrice.getGasPrice());
		LOG.info("Addresse:           " + addresse);
		LOG.info("Bestand:            " + Convert.fromWei(web3.ethGetBalance(addresse, DefaultBlockParameterName.LATEST).send().getBalance().toString(), Unit.ETHER) + " Ether =  "
				+ Convert.fromWei(web3.ethGetBalance(addresse, DefaultBlockParameterName.LATEST).send().getBalance().toString(), Unit.GWEI) + " GWEI");
	}
}
