package de.flapdoodle.embed.process.transitions;

import de.flapdoodle.embed.process.archives.ExtractedFileSet;
import de.flapdoodle.embed.process.config.store.Package;
import de.flapdoodle.embed.process.io.directories.TempDir;
import de.flapdoodle.embed.process.store.ExtractedFileSetStore;
import de.flapdoodle.embed.process.types.Archive;
import de.flapdoodle.embed.process.types.Name;
import de.flapdoodle.reverse.StateID;
import de.flapdoodle.reverse.naming.HasLabel;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Immutable implementation of {@link ExtractPackage}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableExtractPackage.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableExtractPackage extends ExtractPackage {
  private final String transitionLabel;
  private final StateID<Name> name;
  private final StateID<ExtractedFileSet> destination;
  private final StateID<Archive> archive;
  private final StateID<Package> distPackage;
  private final StateID<TempDir> tempDir;
  private final StateID<ExtractedFileSetStore> extractedFileSetStore;

  private ImmutableExtractPackage(ImmutableExtractPackage.Builder builder) {
    this.extractedFileSetStore = builder.extractedFileSetStore;
    if (builder.transitionLabel != null) {
      initShim.transitionLabel(builder.transitionLabel);
    }
    if (builder.name != null) {
      initShim.name(builder.name);
    }
    if (builder.destination != null) {
      initShim.destination(builder.destination);
    }
    if (builder.archive != null) {
      initShim.archive(builder.archive);
    }
    if (builder.distPackage != null) {
      initShim.distPackage(builder.distPackage);
    }
    if (builder.tempDir != null) {
      initShim.tempDir(builder.tempDir);
    }
    this.transitionLabel = initShim.transitionLabel();
    this.name = initShim.name();
    this.destination = initShim.destination();
    this.archive = initShim.archive();
    this.distPackage = initShim.distPackage();
    this.tempDir = initShim.tempDir();
    this.initShim = null;
  }

  private ImmutableExtractPackage(
      String transitionLabel,
      StateID<Name> name,
      StateID<ExtractedFileSet> destination,
      StateID<Archive> archive,
      StateID<Package> distPackage,
      StateID<TempDir> tempDir,
      StateID<ExtractedFileSetStore> extractedFileSetStore) {
    this.transitionLabel = transitionLabel;
    this.name = name;
    this.destination = destination;
    this.archive = archive;
    this.distPackage = distPackage;
    this.tempDir = tempDir;
    this.extractedFileSetStore = extractedFileSetStore;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte transitionLabelBuildStage = STAGE_UNINITIALIZED;
    private String transitionLabel;

    String transitionLabel() {
      if (transitionLabelBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (transitionLabelBuildStage == STAGE_UNINITIALIZED) {
        transitionLabelBuildStage = STAGE_INITIALIZING;
        this.transitionLabel = Objects.requireNonNull(ImmutableExtractPackage.super.transitionLabel(), "transitionLabel");
        transitionLabelBuildStage = STAGE_INITIALIZED;
      }
      return this.transitionLabel;
    }

    void transitionLabel(String transitionLabel) {
      this.transitionLabel = transitionLabel;
      transitionLabelBuildStage = STAGE_INITIALIZED;
    }

    private byte nameBuildStage = STAGE_UNINITIALIZED;
    private StateID<Name> name;

    StateID<Name> name() {
      if (nameBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (nameBuildStage == STAGE_UNINITIALIZED) {
        nameBuildStage = STAGE_INITIALIZING;
        this.name = Objects.requireNonNull(ImmutableExtractPackage.super.name(), "name");
        nameBuildStage = STAGE_INITIALIZED;
      }
      return this.name;
    }

    void name(StateID<Name> name) {
      this.name = name;
      nameBuildStage = STAGE_INITIALIZED;
    }

    private byte destinationBuildStage = STAGE_UNINITIALIZED;
    private StateID<ExtractedFileSet> destination;

    StateID<ExtractedFileSet> destination() {
      if (destinationBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (destinationBuildStage == STAGE_UNINITIALIZED) {
        destinationBuildStage = STAGE_INITIALIZING;
        this.destination = Objects.requireNonNull(ImmutableExtractPackage.super.destination(), "destination");
        destinationBuildStage = STAGE_INITIALIZED;
      }
      return this.destination;
    }

    void destination(StateID<ExtractedFileSet> destination) {
      this.destination = destination;
      destinationBuildStage = STAGE_INITIALIZED;
    }

    private byte archiveBuildStage = STAGE_UNINITIALIZED;
    private StateID<Archive> archive;

    StateID<Archive> archive() {
      if (archiveBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (archiveBuildStage == STAGE_UNINITIALIZED) {
        archiveBuildStage = STAGE_INITIALIZING;
        this.archive = Objects.requireNonNull(ImmutableExtractPackage.super.archive(), "archive");
        archiveBuildStage = STAGE_INITIALIZED;
      }
      return this.archive;
    }

    void archive(StateID<Archive> archive) {
      this.archive = archive;
      archiveBuildStage = STAGE_INITIALIZED;
    }

    private byte distPackageBuildStage = STAGE_UNINITIALIZED;
    private StateID<Package> distPackage;

    StateID<Package> distPackage() {
      if (distPackageBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (distPackageBuildStage == STAGE_UNINITIALIZED) {
        distPackageBuildStage = STAGE_INITIALIZING;
        this.distPackage = Objects.requireNonNull(ImmutableExtractPackage.super.distPackage(), "distPackage");
        distPackageBuildStage = STAGE_INITIALIZED;
      }
      return this.distPackage;
    }

    void distPackage(StateID<Package> distPackage) {
      this.distPackage = distPackage;
      distPackageBuildStage = STAGE_INITIALIZED;
    }

    private byte tempDirBuildStage = STAGE_UNINITIALIZED;
    private StateID<TempDir> tempDir;

    StateID<TempDir> tempDir() {
      if (tempDirBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (tempDirBuildStage == STAGE_UNINITIALIZED) {
        tempDirBuildStage = STAGE_INITIALIZING;
        this.tempDir = Objects.requireNonNull(ImmutableExtractPackage.super.tempDir(), "tempDir");
        tempDirBuildStage = STAGE_INITIALIZED;
      }
      return this.tempDir;
    }

    void tempDir(StateID<TempDir> tempDir) {
      this.tempDir = tempDir;
      tempDirBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (transitionLabelBuildStage == STAGE_INITIALIZING) attributes.add("transitionLabel");
      if (nameBuildStage == STAGE_INITIALIZING) attributes.add("name");
      if (destinationBuildStage == STAGE_INITIALIZING) attributes.add("destination");
      if (archiveBuildStage == STAGE_INITIALIZING) attributes.add("archive");
      if (distPackageBuildStage == STAGE_INITIALIZING) attributes.add("distPackage");
      if (tempDirBuildStage == STAGE_INITIALIZING) attributes.add("tempDir");
      return "Cannot build ExtractPackage, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code transitionLabel} attribute
   */
  @Override
  public String transitionLabel() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.transitionLabel()
        : this.transitionLabel;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @Override
  protected StateID<Name> name() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.name()
        : this.name;
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public StateID<ExtractedFileSet> destination() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.destination()
        : this.destination;
  }

  /**
   * @return The value of the {@code archive} attribute
   */
  @Override
  protected StateID<Archive> archive() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.archive()
        : this.archive;
  }

  /**
   * @return The value of the {@code distPackage} attribute
   */
  @Override
  protected StateID<Package> distPackage() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.distPackage()
        : this.distPackage;
  }

  /**
   * @return The value of the {@code tempDir} attribute
   */
  @Override
  protected StateID<TempDir> tempDir() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.tempDir()
        : this.tempDir;
  }

  /**
   * @return The value of the {@code extractedFileSetStore} attribute
   */
  @Override
  protected Optional<StateID<ExtractedFileSetStore>> extractedFileSetStore() {
    return Optional.ofNullable(extractedFileSetStore);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExtractPackage#transitionLabel() transitionLabel} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for transitionLabel
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExtractPackage withTransitionLabel(String value) {
    String newValue = Objects.requireNonNull(value, "transitionLabel");
    if (this.transitionLabel.equals(newValue)) return this;
    return new ImmutableExtractPackage(
        newValue,
        this.name,
        this.destination,
        this.archive,
        this.distPackage,
        this.tempDir,
        this.extractedFileSetStore);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExtractPackage#name() name} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExtractPackage withName(StateID<Name> value) {
    if (this.name == value) return this;
    StateID<Name> newValue = Objects.requireNonNull(value, "name");
    return new ImmutableExtractPackage(
        this.transitionLabel,
        newValue,
        this.destination,
        this.archive,
        this.distPackage,
        this.tempDir,
        this.extractedFileSetStore);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExtractPackage#destination() destination} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExtractPackage withDestination(StateID<ExtractedFileSet> value) {
    if (this.destination == value) return this;
    StateID<ExtractedFileSet> newValue = Objects.requireNonNull(value, "destination");
    return new ImmutableExtractPackage(
        this.transitionLabel,
        this.name,
        newValue,
        this.archive,
        this.distPackage,
        this.tempDir,
        this.extractedFileSetStore);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExtractPackage#archive() archive} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for archive
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExtractPackage withArchive(StateID<Archive> value) {
    if (this.archive == value) return this;
    StateID<Archive> newValue = Objects.requireNonNull(value, "archive");
    return new ImmutableExtractPackage(
        this.transitionLabel,
        this.name,
        this.destination,
        newValue,
        this.distPackage,
        this.tempDir,
        this.extractedFileSetStore);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExtractPackage#distPackage() distPackage} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for distPackage
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExtractPackage withDistPackage(StateID<Package> value) {
    if (this.distPackage == value) return this;
    StateID<Package> newValue = Objects.requireNonNull(value, "distPackage");
    return new ImmutableExtractPackage(
        this.transitionLabel,
        this.name,
        this.destination,
        this.archive,
        newValue,
        this.tempDir,
        this.extractedFileSetStore);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExtractPackage#tempDir() tempDir} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for tempDir
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableExtractPackage withTempDir(StateID<TempDir> value) {
    if (this.tempDir == value) return this;
    StateID<TempDir> newValue = Objects.requireNonNull(value, "tempDir");
    return new ImmutableExtractPackage(
        this.transitionLabel,
        this.name,
        this.destination,
        this.archive,
        this.distPackage,
        newValue,
        this.extractedFileSetStore);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link ExtractPackage#extractedFileSetStore() extractedFileSetStore} attribute.
   * @param value The value for extractedFileSetStore
   * @return A modified copy of {@code this} object
   */
  public final ImmutableExtractPackage withExtractedFileSetStore(StateID<ExtractedFileSetStore> value) {
    StateID<ExtractedFileSetStore> newValue = Objects.requireNonNull(value, "extractedFileSetStore");
    if (this.extractedFileSetStore == newValue) return this;
    return new ImmutableExtractPackage(
        this.transitionLabel,
        this.name,
        this.destination,
        this.archive,
        this.distPackage,
        this.tempDir,
        newValue);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link ExtractPackage#extractedFileSetStore() extractedFileSetStore} attribute.
   * A shallow reference equality check is used on unboxed optional value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for extractedFileSetStore
   * @return A modified copy of {@code this} object
   */
  @SuppressWarnings("unchecked") // safe covariant cast
  public final ImmutableExtractPackage withExtractedFileSetStore(Optional<? extends StateID<ExtractedFileSetStore>> optional) {
    StateID<ExtractedFileSetStore> value = optional.orElse(null);
    if (this.extractedFileSetStore == value) return this;
    return new ImmutableExtractPackage(
        this.transitionLabel,
        this.name,
        this.destination,
        this.archive,
        this.distPackage,
        this.tempDir,
        value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableExtractPackage} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableExtractPackage
        && equalTo(0, (ImmutableExtractPackage) another);
  }

  private boolean equalTo(int synthetic, ImmutableExtractPackage another) {
    return transitionLabel.equals(another.transitionLabel)
        && name.equals(another.name)
        && destination.equals(another.destination)
        && archive.equals(another.archive)
        && distPackage.equals(another.distPackage)
        && tempDir.equals(another.tempDir)
        && Objects.equals(extractedFileSetStore, another.extractedFileSetStore);
  }

  /**
   * Computes a hash code from attributes: {@code transitionLabel}, {@code name}, {@code destination}, {@code archive}, {@code distPackage}, {@code tempDir}, {@code extractedFileSetStore}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + transitionLabel.hashCode();
    h += (h << 5) + name.hashCode();
    h += (h << 5) + destination.hashCode();
    h += (h << 5) + archive.hashCode();
    h += (h << 5) + distPackage.hashCode();
    h += (h << 5) + tempDir.hashCode();
    h += (h << 5) + Objects.hashCode(extractedFileSetStore);
    return h;
  }

  /**
   * Prints the immutable value {@code ExtractPackage} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("ExtractPackage{");
    builder.append("transitionLabel=").append(transitionLabel);
    builder.append(", ");
    builder.append("name=").append(name);
    builder.append(", ");
    builder.append("destination=").append(destination);
    builder.append(", ");
    builder.append("archive=").append(archive);
    builder.append(", ");
    builder.append("distPackage=").append(distPackage);
    builder.append(", ");
    builder.append("tempDir=").append(tempDir);
    if (extractedFileSetStore != null) {
      builder.append(", ");
      builder.append("extractedFileSetStore=").append(extractedFileSetStore);
    }
    return builder.append("}").toString();
  }

  /**
   * Creates an immutable copy of a {@link ExtractPackage} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ExtractPackage instance
   */
  public static ImmutableExtractPackage copyOf(ExtractPackage instance) {
    if (instance instanceof ImmutableExtractPackage) {
      return (ImmutableExtractPackage) instance;
    }
    return ImmutableExtractPackage.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableExtractPackage ImmutableExtractPackage}.
   * <pre>
   * ImmutableExtractPackage.builder()
   *    .transitionLabel(String) // optional {@link ExtractPackage#transitionLabel() transitionLabel}
   *    .name(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.types.Name&amp;gt;) // optional {@link ExtractPackage#name() name}
   *    .destination(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.archives.ExtractedFileSet&amp;gt;) // optional {@link ExtractPackage#destination() destination}
   *    .archive(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.types.Archive&amp;gt;) // optional {@link ExtractPackage#archive() archive}
   *    .distPackage(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.config.store.Package&amp;gt;) // optional {@link ExtractPackage#distPackage() distPackage}
   *    .tempDir(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.io.directories.TempDir&amp;gt;) // optional {@link ExtractPackage#tempDir() tempDir}
   *    .extractedFileSetStore(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.store.ExtractedFileSetStore&amp;gt;) // optional {@link ExtractPackage#extractedFileSetStore() extractedFileSetStore}
   *    .build();
   * </pre>
   * @return A new ImmutableExtractPackage builder
   */
  public static ImmutableExtractPackage.Builder builder() {
    return new ImmutableExtractPackage.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableExtractPackage ImmutableExtractPackage}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private String transitionLabel;
    private StateID<Name> name;
    private StateID<ExtractedFileSet> destination;
    private StateID<Archive> archive;
    private StateID<Package> distPackage;
    private StateID<TempDir> tempDir;
    private StateID<ExtractedFileSetStore> extractedFileSetStore;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.embed.process.transitions.ExtractPackage} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ExtractPackage instance) {
      Objects.requireNonNull(instance, "instance");
      from((short) 0, (Object) instance);
      return this;
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.reverse.naming.HasLabel} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(HasLabel instance) {
      Objects.requireNonNull(instance, "instance");
      from((short) 0, (Object) instance);
      return this;
    }

    private void from(short _unused, Object object) {
      long bits = 0;
      if (object instanceof ExtractPackage) {
        ExtractPackage instance = (ExtractPackage) object;
        Optional<StateID<ExtractedFileSetStore>> extractedFileSetStoreOptional = instance.extractedFileSetStore();
        if (extractedFileSetStoreOptional.isPresent()) {
          extractedFileSetStore(extractedFileSetStoreOptional);
        }
        if ((bits & 0x1L) == 0) {
          transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
        destination(instance.destination());
        name(instance.name());
        distPackage(instance.distPackage());
        tempDir(instance.tempDir());
        archive(instance.archive());
      }
      if (object instanceof HasLabel) {
        HasLabel instance = (HasLabel) object;
        if ((bits & 0x1L) == 0) {
          transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
      }
    }

    /**
     * Initializes the value for the {@link ExtractPackage#transitionLabel() transitionLabel} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ExtractPackage#transitionLabel() transitionLabel}.</em>
     * @param transitionLabel The value for transitionLabel 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder transitionLabel(String transitionLabel) {
      this.transitionLabel = Objects.requireNonNull(transitionLabel, "transitionLabel");
      return this;
    }

    /**
     * Initializes the value for the {@link ExtractPackage#name() name} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ExtractPackage#name() name}.</em>
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder name(StateID<Name> name) {
      this.name = Objects.requireNonNull(name, "name");
      return this;
    }

    /**
     * Initializes the value for the {@link ExtractPackage#destination() destination} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ExtractPackage#destination() destination}.</em>
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder destination(StateID<ExtractedFileSet> destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      return this;
    }

    /**
     * Initializes the value for the {@link ExtractPackage#archive() archive} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ExtractPackage#archive() archive}.</em>
     * @param archive The value for archive 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder archive(StateID<Archive> archive) {
      this.archive = Objects.requireNonNull(archive, "archive");
      return this;
    }

    /**
     * Initializes the value for the {@link ExtractPackage#distPackage() distPackage} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ExtractPackage#distPackage() distPackage}.</em>
     * @param distPackage The value for distPackage 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder distPackage(StateID<Package> distPackage) {
      this.distPackage = Objects.requireNonNull(distPackage, "distPackage");
      return this;
    }

    /**
     * Initializes the value for the {@link ExtractPackage#tempDir() tempDir} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link ExtractPackage#tempDir() tempDir}.</em>
     * @param tempDir The value for tempDir 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder tempDir(StateID<TempDir> tempDir) {
      this.tempDir = Objects.requireNonNull(tempDir, "tempDir");
      return this;
    }

    /**
     * Initializes the optional value {@link ExtractPackage#extractedFileSetStore() extractedFileSetStore} to extractedFileSetStore.
     * @param extractedFileSetStore The value for extractedFileSetStore
     * @return {@code this} builder for chained invocation
     */
    public final Builder extractedFileSetStore(StateID<ExtractedFileSetStore> extractedFileSetStore) {
      this.extractedFileSetStore = Objects.requireNonNull(extractedFileSetStore, "extractedFileSetStore");
      return this;
    }

    /**
     * Initializes the optional value {@link ExtractPackage#extractedFileSetStore() extractedFileSetStore} to extractedFileSetStore.
     * @param extractedFileSetStore The value for extractedFileSetStore
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder extractedFileSetStore(Optional<? extends StateID<ExtractedFileSetStore>> extractedFileSetStore) {
      this.extractedFileSetStore = extractedFileSetStore.orElse(null);
      return this;
    }

    /**
     * Builds a new {@link ImmutableExtractPackage ImmutableExtractPackage}.
     * @return An immutable instance of ExtractPackage
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableExtractPackage build() {
      return new ImmutableExtractPackage(this);
    }
  }
}
