package de.flapdoodle.embed.process.transitions;

import de.flapdoodle.embed.process.config.DownloadConfig;
import de.flapdoodle.embed.process.config.store.Package;
import de.flapdoodle.embed.process.distribution.Distribution;
import de.flapdoodle.embed.process.io.directories.TempDir;
import de.flapdoodle.embed.process.io.progress.ProgressListener;
import de.flapdoodle.embed.process.net.DownloadToPath;
import de.flapdoodle.embed.process.store.DownloadCache;
import de.flapdoodle.embed.process.types.Archive;
import de.flapdoodle.embed.process.types.Name;
import de.flapdoodle.reverse.StateID;
import de.flapdoodle.reverse.naming.HasLabel;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link DownloadPackage}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableDownloadPackage.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableDownloadPackage extends DownloadPackage {
  private final String transitionLabel;
  private final StateID<Name> name;
  private final StateID<DownloadCache> downloadCache;
  private final StateID<ProgressListener> progressListener;
  private final DownloadConfig downloadConfig;
  private final StateID<Distribution> distribution;
  private final StateID<Package> distPackage;
  private final StateID<TempDir> tempDirectory;
  private final StateID<Archive> destination;
  private final DownloadToPath downloadToPath;

  private ImmutableDownloadPackage(ImmutableDownloadPackage.Builder builder) {
    if (builder.transitionLabel != null) {
      initShim.transitionLabel(builder.transitionLabel);
    }
    if (builder.name != null) {
      initShim.name(builder.name);
    }
    if (builder.downloadCache != null) {
      initShim.downloadCache(builder.downloadCache);
    }
    if (builder.progressListener != null) {
      initShim.progressListener(builder.progressListener);
    }
    if (builder.downloadConfig != null) {
      initShim.downloadConfig(builder.downloadConfig);
    }
    if (builder.distribution != null) {
      initShim.distribution(builder.distribution);
    }
    if (builder.distPackage != null) {
      initShim.distPackage(builder.distPackage);
    }
    if (builder.tempDirectory != null) {
      initShim.tempDirectory(builder.tempDirectory);
    }
    if (builder.destination != null) {
      initShim.destination(builder.destination);
    }
    if (builder.downloadToPath != null) {
      initShim.downloadToPath(builder.downloadToPath);
    }
    this.transitionLabel = initShim.transitionLabel();
    this.name = initShim.name();
    this.downloadCache = initShim.downloadCache();
    this.progressListener = initShim.progressListener();
    this.downloadConfig = initShim.downloadConfig();
    this.distribution = initShim.distribution();
    this.distPackage = initShim.distPackage();
    this.tempDirectory = initShim.tempDirectory();
    this.destination = initShim.destination();
    this.downloadToPath = initShim.downloadToPath();
    this.initShim = null;
  }

  private ImmutableDownloadPackage(
      String transitionLabel,
      StateID<Name> name,
      StateID<DownloadCache> downloadCache,
      StateID<ProgressListener> progressListener,
      DownloadConfig downloadConfig,
      StateID<Distribution> distribution,
      StateID<Package> distPackage,
      StateID<TempDir> tempDirectory,
      StateID<Archive> destination,
      DownloadToPath downloadToPath) {
    this.transitionLabel = transitionLabel;
    this.name = name;
    this.downloadCache = downloadCache;
    this.progressListener = progressListener;
    this.downloadConfig = downloadConfig;
    this.distribution = distribution;
    this.distPackage = distPackage;
    this.tempDirectory = tempDirectory;
    this.destination = destination;
    this.downloadToPath = downloadToPath;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte transitionLabelBuildStage = STAGE_UNINITIALIZED;
    private String transitionLabel;

    String transitionLabel() {
      if (transitionLabelBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (transitionLabelBuildStage == STAGE_UNINITIALIZED) {
        transitionLabelBuildStage = STAGE_INITIALIZING;
        this.transitionLabel = Objects.requireNonNull(ImmutableDownloadPackage.super.transitionLabel(), "transitionLabel");
        transitionLabelBuildStage = STAGE_INITIALIZED;
      }
      return this.transitionLabel;
    }

    void transitionLabel(String transitionLabel) {
      this.transitionLabel = transitionLabel;
      transitionLabelBuildStage = STAGE_INITIALIZED;
    }

    private byte nameBuildStage = STAGE_UNINITIALIZED;
    private StateID<Name> name;

    StateID<Name> name() {
      if (nameBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (nameBuildStage == STAGE_UNINITIALIZED) {
        nameBuildStage = STAGE_INITIALIZING;
        this.name = Objects.requireNonNull(ImmutableDownloadPackage.super.name(), "name");
        nameBuildStage = STAGE_INITIALIZED;
      }
      return this.name;
    }

    void name(StateID<Name> name) {
      this.name = name;
      nameBuildStage = STAGE_INITIALIZED;
    }

    private byte downloadCacheBuildStage = STAGE_UNINITIALIZED;
    private StateID<DownloadCache> downloadCache;

    StateID<DownloadCache> downloadCache() {
      if (downloadCacheBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (downloadCacheBuildStage == STAGE_UNINITIALIZED) {
        downloadCacheBuildStage = STAGE_INITIALIZING;
        this.downloadCache = Objects.requireNonNull(ImmutableDownloadPackage.super.downloadCache(), "downloadCache");
        downloadCacheBuildStage = STAGE_INITIALIZED;
      }
      return this.downloadCache;
    }

    void downloadCache(StateID<DownloadCache> downloadCache) {
      this.downloadCache = downloadCache;
      downloadCacheBuildStage = STAGE_INITIALIZED;
    }

    private byte progressListenerBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProgressListener> progressListener;

    StateID<ProgressListener> progressListener() {
      if (progressListenerBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (progressListenerBuildStage == STAGE_UNINITIALIZED) {
        progressListenerBuildStage = STAGE_INITIALIZING;
        this.progressListener = Objects.requireNonNull(ImmutableDownloadPackage.super.progressListener(), "progressListener");
        progressListenerBuildStage = STAGE_INITIALIZED;
      }
      return this.progressListener;
    }

    void progressListener(StateID<ProgressListener> progressListener) {
      this.progressListener = progressListener;
      progressListenerBuildStage = STAGE_INITIALIZED;
    }

    private byte downloadConfigBuildStage = STAGE_UNINITIALIZED;
    private DownloadConfig downloadConfig;

    DownloadConfig downloadConfig() {
      if (downloadConfigBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (downloadConfigBuildStage == STAGE_UNINITIALIZED) {
        downloadConfigBuildStage = STAGE_INITIALIZING;
        this.downloadConfig = Objects.requireNonNull(ImmutableDownloadPackage.super.downloadConfig(), "downloadConfig");
        downloadConfigBuildStage = STAGE_INITIALIZED;
      }
      return this.downloadConfig;
    }

    void downloadConfig(DownloadConfig downloadConfig) {
      this.downloadConfig = downloadConfig;
      downloadConfigBuildStage = STAGE_INITIALIZED;
    }

    private byte distributionBuildStage = STAGE_UNINITIALIZED;
    private StateID<Distribution> distribution;

    StateID<Distribution> distribution() {
      if (distributionBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (distributionBuildStage == STAGE_UNINITIALIZED) {
        distributionBuildStage = STAGE_INITIALIZING;
        this.distribution = Objects.requireNonNull(ImmutableDownloadPackage.super.distribution(), "distribution");
        distributionBuildStage = STAGE_INITIALIZED;
      }
      return this.distribution;
    }

    void distribution(StateID<Distribution> distribution) {
      this.distribution = distribution;
      distributionBuildStage = STAGE_INITIALIZED;
    }

    private byte distPackageBuildStage = STAGE_UNINITIALIZED;
    private StateID<Package> distPackage;

    StateID<Package> distPackage() {
      if (distPackageBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (distPackageBuildStage == STAGE_UNINITIALIZED) {
        distPackageBuildStage = STAGE_INITIALIZING;
        this.distPackage = Objects.requireNonNull(ImmutableDownloadPackage.super.distPackage(), "distPackage");
        distPackageBuildStage = STAGE_INITIALIZED;
      }
      return this.distPackage;
    }

    void distPackage(StateID<Package> distPackage) {
      this.distPackage = distPackage;
      distPackageBuildStage = STAGE_INITIALIZED;
    }

    private byte tempDirectoryBuildStage = STAGE_UNINITIALIZED;
    private StateID<TempDir> tempDirectory;

    StateID<TempDir> tempDirectory() {
      if (tempDirectoryBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (tempDirectoryBuildStage == STAGE_UNINITIALIZED) {
        tempDirectoryBuildStage = STAGE_INITIALIZING;
        this.tempDirectory = Objects.requireNonNull(ImmutableDownloadPackage.super.tempDirectory(), "tempDirectory");
        tempDirectoryBuildStage = STAGE_INITIALIZED;
      }
      return this.tempDirectory;
    }

    void tempDirectory(StateID<TempDir> tempDirectory) {
      this.tempDirectory = tempDirectory;
      tempDirectoryBuildStage = STAGE_INITIALIZED;
    }

    private byte destinationBuildStage = STAGE_UNINITIALIZED;
    private StateID<Archive> destination;

    StateID<Archive> destination() {
      if (destinationBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (destinationBuildStage == STAGE_UNINITIALIZED) {
        destinationBuildStage = STAGE_INITIALIZING;
        this.destination = Objects.requireNonNull(ImmutableDownloadPackage.super.destination(), "destination");
        destinationBuildStage = STAGE_INITIALIZED;
      }
      return this.destination;
    }

    void destination(StateID<Archive> destination) {
      this.destination = destination;
      destinationBuildStage = STAGE_INITIALIZED;
    }

    private byte downloadToPathBuildStage = STAGE_UNINITIALIZED;
    private DownloadToPath downloadToPath;

    DownloadToPath downloadToPath() {
      if (downloadToPathBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (downloadToPathBuildStage == STAGE_UNINITIALIZED) {
        downloadToPathBuildStage = STAGE_INITIALIZING;
        this.downloadToPath = Objects.requireNonNull(ImmutableDownloadPackage.super.downloadToPath(), "downloadToPath");
        downloadToPathBuildStage = STAGE_INITIALIZED;
      }
      return this.downloadToPath;
    }

    void downloadToPath(DownloadToPath downloadToPath) {
      this.downloadToPath = downloadToPath;
      downloadToPathBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (transitionLabelBuildStage == STAGE_INITIALIZING) attributes.add("transitionLabel");
      if (nameBuildStage == STAGE_INITIALIZING) attributes.add("name");
      if (downloadCacheBuildStage == STAGE_INITIALIZING) attributes.add("downloadCache");
      if (progressListenerBuildStage == STAGE_INITIALIZING) attributes.add("progressListener");
      if (downloadConfigBuildStage == STAGE_INITIALIZING) attributes.add("downloadConfig");
      if (distributionBuildStage == STAGE_INITIALIZING) attributes.add("distribution");
      if (distPackageBuildStage == STAGE_INITIALIZING) attributes.add("distPackage");
      if (tempDirectoryBuildStage == STAGE_INITIALIZING) attributes.add("tempDirectory");
      if (destinationBuildStage == STAGE_INITIALIZING) attributes.add("destination");
      if (downloadToPathBuildStage == STAGE_INITIALIZING) attributes.add("downloadToPath");
      return "Cannot build DownloadPackage, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code transitionLabel} attribute
   */
  @Override
  public String transitionLabel() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.transitionLabel()
        : this.transitionLabel;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @Override
  protected StateID<Name> name() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.name()
        : this.name;
  }

  /**
   * @return The value of the {@code downloadCache} attribute
   */
  @Override
  protected StateID<DownloadCache> downloadCache() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.downloadCache()
        : this.downloadCache;
  }

  /**
   * @return The value of the {@code progressListener} attribute
   */
  @Override
  protected StateID<ProgressListener> progressListener() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.progressListener()
        : this.progressListener;
  }

  /**
   * @return The value of the {@code downloadConfig} attribute
   */
  @Override
  protected DownloadConfig downloadConfig() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.downloadConfig()
        : this.downloadConfig;
  }

  /**
   * @return The value of the {@code distribution} attribute
   */
  @Override
  protected StateID<Distribution> distribution() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.distribution()
        : this.distribution;
  }

  /**
   * @return The value of the {@code distPackage} attribute
   */
  @Override
  protected StateID<Package> distPackage() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.distPackage()
        : this.distPackage;
  }

  /**
   * @return The value of the {@code tempDirectory} attribute
   */
  @Override
  protected StateID<TempDir> tempDirectory() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.tempDirectory()
        : this.tempDirectory;
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public StateID<Archive> destination() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.destination()
        : this.destination;
  }

  /**
   * @return The value of the {@code downloadToPath} attribute
   */
  @Override
  public DownloadToPath downloadToPath() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.downloadToPath()
        : this.downloadToPath;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#transitionLabel() transitionLabel} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for transitionLabel
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withTransitionLabel(String value) {
    String newValue = Objects.requireNonNull(value, "transitionLabel");
    if (this.transitionLabel.equals(newValue)) return this;
    return new ImmutableDownloadPackage(
        newValue,
        this.name,
        this.downloadCache,
        this.progressListener,
        this.downloadConfig,
        this.distribution,
        this.distPackage,
        this.tempDirectory,
        this.destination,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#name() name} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withName(StateID<Name> value) {
    if (this.name == value) return this;
    StateID<Name> newValue = Objects.requireNonNull(value, "name");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        newValue,
        this.downloadCache,
        this.progressListener,
        this.downloadConfig,
        this.distribution,
        this.distPackage,
        this.tempDirectory,
        this.destination,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#downloadCache() downloadCache} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for downloadCache
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withDownloadCache(StateID<DownloadCache> value) {
    if (this.downloadCache == value) return this;
    StateID<DownloadCache> newValue = Objects.requireNonNull(value, "downloadCache");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        this.name,
        newValue,
        this.progressListener,
        this.downloadConfig,
        this.distribution,
        this.distPackage,
        this.tempDirectory,
        this.destination,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#progressListener() progressListener} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for progressListener
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withProgressListener(StateID<ProgressListener> value) {
    if (this.progressListener == value) return this;
    StateID<ProgressListener> newValue = Objects.requireNonNull(value, "progressListener");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        this.name,
        this.downloadCache,
        newValue,
        this.downloadConfig,
        this.distribution,
        this.distPackage,
        this.tempDirectory,
        this.destination,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#downloadConfig() downloadConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for downloadConfig
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withDownloadConfig(DownloadConfig value) {
    if (this.downloadConfig == value) return this;
    DownloadConfig newValue = Objects.requireNonNull(value, "downloadConfig");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        this.name,
        this.downloadCache,
        this.progressListener,
        newValue,
        this.distribution,
        this.distPackage,
        this.tempDirectory,
        this.destination,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#distribution() distribution} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for distribution
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withDistribution(StateID<Distribution> value) {
    if (this.distribution == value) return this;
    StateID<Distribution> newValue = Objects.requireNonNull(value, "distribution");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        this.name,
        this.downloadCache,
        this.progressListener,
        this.downloadConfig,
        newValue,
        this.distPackage,
        this.tempDirectory,
        this.destination,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#distPackage() distPackage} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for distPackage
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withDistPackage(StateID<Package> value) {
    if (this.distPackage == value) return this;
    StateID<Package> newValue = Objects.requireNonNull(value, "distPackage");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        this.name,
        this.downloadCache,
        this.progressListener,
        this.downloadConfig,
        this.distribution,
        newValue,
        this.tempDirectory,
        this.destination,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#tempDirectory() tempDirectory} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for tempDirectory
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withTempDirectory(StateID<TempDir> value) {
    if (this.tempDirectory == value) return this;
    StateID<TempDir> newValue = Objects.requireNonNull(value, "tempDirectory");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        this.name,
        this.downloadCache,
        this.progressListener,
        this.downloadConfig,
        this.distribution,
        this.distPackage,
        newValue,
        this.destination,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#destination() destination} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withDestination(StateID<Archive> value) {
    if (this.destination == value) return this;
    StateID<Archive> newValue = Objects.requireNonNull(value, "destination");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        this.name,
        this.downloadCache,
        this.progressListener,
        this.downloadConfig,
        this.distribution,
        this.distPackage,
        this.tempDirectory,
        newValue,
        this.downloadToPath);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link DownloadPackage#downloadToPath() downloadToPath} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for downloadToPath
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableDownloadPackage withDownloadToPath(DownloadToPath value) {
    if (this.downloadToPath == value) return this;
    DownloadToPath newValue = Objects.requireNonNull(value, "downloadToPath");
    return new ImmutableDownloadPackage(
        this.transitionLabel,
        this.name,
        this.downloadCache,
        this.progressListener,
        this.downloadConfig,
        this.distribution,
        this.distPackage,
        this.tempDirectory,
        this.destination,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableDownloadPackage} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableDownloadPackage
        && equalTo(0, (ImmutableDownloadPackage) another);
  }

  private boolean equalTo(int synthetic, ImmutableDownloadPackage another) {
    return transitionLabel.equals(another.transitionLabel)
        && name.equals(another.name)
        && downloadCache.equals(another.downloadCache)
        && progressListener.equals(another.progressListener)
        && downloadConfig.equals(another.downloadConfig)
        && distribution.equals(another.distribution)
        && distPackage.equals(another.distPackage)
        && tempDirectory.equals(another.tempDirectory)
        && destination.equals(another.destination)
        && downloadToPath.equals(another.downloadToPath);
  }

  /**
   * Computes a hash code from attributes: {@code transitionLabel}, {@code name}, {@code downloadCache}, {@code progressListener}, {@code downloadConfig}, {@code distribution}, {@code distPackage}, {@code tempDirectory}, {@code destination}, {@code downloadToPath}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + transitionLabel.hashCode();
    h += (h << 5) + name.hashCode();
    h += (h << 5) + downloadCache.hashCode();
    h += (h << 5) + progressListener.hashCode();
    h += (h << 5) + downloadConfig.hashCode();
    h += (h << 5) + distribution.hashCode();
    h += (h << 5) + distPackage.hashCode();
    h += (h << 5) + tempDirectory.hashCode();
    h += (h << 5) + destination.hashCode();
    h += (h << 5) + downloadToPath.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code DownloadPackage} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "DownloadPackage{"
        + "transitionLabel=" + transitionLabel
        + ", name=" + name
        + ", downloadCache=" + downloadCache
        + ", progressListener=" + progressListener
        + ", downloadConfig=" + downloadConfig
        + ", distribution=" + distribution
        + ", distPackage=" + distPackage
        + ", tempDirectory=" + tempDirectory
        + ", destination=" + destination
        + ", downloadToPath=" + downloadToPath
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link DownloadPackage} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable DownloadPackage instance
   */
  public static ImmutableDownloadPackage copyOf(DownloadPackage instance) {
    if (instance instanceof ImmutableDownloadPackage) {
      return (ImmutableDownloadPackage) instance;
    }
    return ImmutableDownloadPackage.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableDownloadPackage ImmutableDownloadPackage}.
   * <pre>
   * ImmutableDownloadPackage.builder()
   *    .transitionLabel(String) // optional {@link DownloadPackage#transitionLabel() transitionLabel}
   *    .name(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.types.Name&amp;gt;) // optional {@link DownloadPackage#name() name}
   *    .downloadCache(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.store.DownloadCache&amp;gt;) // optional {@link DownloadPackage#downloadCache() downloadCache}
   *    .progressListener(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.io.progress.ProgressListener&amp;gt;) // optional {@link DownloadPackage#progressListener() progressListener}
   *    .downloadConfig(de.flapdoodle.embed.process.config.DownloadConfig) // optional {@link DownloadPackage#downloadConfig() downloadConfig}
   *    .distribution(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.distribution.Distribution&amp;gt;) // optional {@link DownloadPackage#distribution() distribution}
   *    .distPackage(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.config.store.Package&amp;gt;) // optional {@link DownloadPackage#distPackage() distPackage}
   *    .tempDirectory(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.io.directories.TempDir&amp;gt;) // optional {@link DownloadPackage#tempDirectory() tempDirectory}
   *    .destination(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.types.Archive&amp;gt;) // optional {@link DownloadPackage#destination() destination}
   *    .downloadToPath(de.flapdoodle.embed.process.net.DownloadToPath) // optional {@link DownloadPackage#downloadToPath() downloadToPath}
   *    .build();
   * </pre>
   * @return A new ImmutableDownloadPackage builder
   */
  public static ImmutableDownloadPackage.Builder builder() {
    return new ImmutableDownloadPackage.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableDownloadPackage ImmutableDownloadPackage}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private String transitionLabel;
    private StateID<Name> name;
    private StateID<DownloadCache> downloadCache;
    private StateID<ProgressListener> progressListener;
    private DownloadConfig downloadConfig;
    private StateID<Distribution> distribution;
    private StateID<Package> distPackage;
    private StateID<TempDir> tempDirectory;
    private StateID<Archive> destination;
    private DownloadToPath downloadToPath;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.embed.process.transitions.DownloadPackage} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(DownloadPackage instance) {
      Objects.requireNonNull(instance, "instance");
      from((short) 0, (Object) instance);
      return this;
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.reverse.naming.HasLabel} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(HasLabel instance) {
      Objects.requireNonNull(instance, "instance");
      from((short) 0, (Object) instance);
      return this;
    }

    private void from(short _unused, Object object) {
      long bits = 0;
      if (object instanceof DownloadPackage) {
        DownloadPackage instance = (DownloadPackage) object;
        progressListener(instance.progressListener());
        tempDirectory(instance.tempDirectory());
        downloadConfig(instance.downloadConfig());
        downloadToPath(instance.downloadToPath());
        if ((bits & 0x1L) == 0) {
          transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
        destination(instance.destination());
        name(instance.name());
        downloadCache(instance.downloadCache());
        distPackage(instance.distPackage());
        distribution(instance.distribution());
      }
      if (object instanceof HasLabel) {
        HasLabel instance = (HasLabel) object;
        if ((bits & 0x1L) == 0) {
          transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
      }
    }

    /**
     * Initializes the value for the {@link DownloadPackage#transitionLabel() transitionLabel} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#transitionLabel() transitionLabel}.</em>
     * @param transitionLabel The value for transitionLabel 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder transitionLabel(String transitionLabel) {
      this.transitionLabel = Objects.requireNonNull(transitionLabel, "transitionLabel");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#name() name} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#name() name}.</em>
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder name(StateID<Name> name) {
      this.name = Objects.requireNonNull(name, "name");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#downloadCache() downloadCache} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#downloadCache() downloadCache}.</em>
     * @param downloadCache The value for downloadCache 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder downloadCache(StateID<DownloadCache> downloadCache) {
      this.downloadCache = Objects.requireNonNull(downloadCache, "downloadCache");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#progressListener() progressListener} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#progressListener() progressListener}.</em>
     * @param progressListener The value for progressListener 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder progressListener(StateID<ProgressListener> progressListener) {
      this.progressListener = Objects.requireNonNull(progressListener, "progressListener");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#downloadConfig() downloadConfig} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#downloadConfig() downloadConfig}.</em>
     * @param downloadConfig The value for downloadConfig 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder downloadConfig(DownloadConfig downloadConfig) {
      this.downloadConfig = Objects.requireNonNull(downloadConfig, "downloadConfig");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#distribution() distribution} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#distribution() distribution}.</em>
     * @param distribution The value for distribution 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder distribution(StateID<Distribution> distribution) {
      this.distribution = Objects.requireNonNull(distribution, "distribution");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#distPackage() distPackage} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#distPackage() distPackage}.</em>
     * @param distPackage The value for distPackage 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder distPackage(StateID<Package> distPackage) {
      this.distPackage = Objects.requireNonNull(distPackage, "distPackage");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#tempDirectory() tempDirectory} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#tempDirectory() tempDirectory}.</em>
     * @param tempDirectory The value for tempDirectory 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder tempDirectory(StateID<TempDir> tempDirectory) {
      this.tempDirectory = Objects.requireNonNull(tempDirectory, "tempDirectory");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#destination() destination} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#destination() destination}.</em>
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder destination(StateID<Archive> destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      return this;
    }

    /**
     * Initializes the value for the {@link DownloadPackage#downloadToPath() downloadToPath} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link DownloadPackage#downloadToPath() downloadToPath}.</em>
     * @param downloadToPath The value for downloadToPath 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder downloadToPath(DownloadToPath downloadToPath) {
      this.downloadToPath = Objects.requireNonNull(downloadToPath, "downloadToPath");
      return this;
    }

    /**
     * Builds a new {@link ImmutableDownloadPackage ImmutableDownloadPackage}.
     * @return An immutable instance of DownloadPackage
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableDownloadPackage build() {
      return new ImmutableDownloadPackage(this);
    }
  }
}
