/*
 * Decompiled with CFR 0.152.
 */
package ai.vespa.util.http.retry;

import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.logging.Logger;
import org.apache.http.annotation.Contract;
import org.apache.http.annotation.ThreadingBehavior;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.protocol.HttpContext;

@Contract(threading=ThreadingBehavior.IMMUTABLE)
public class DelayedHttpRequestRetryHandler
implements HttpRequestRetryHandler {
    private static final Logger log = Logger.getLogger(HttpRequestRetryHandler.class.getName());
    private final DelaySupplier delaySupplier;
    private final int maxRetries;
    private final RetryPredicate predicate;
    private final RetryConsumer retryConsumer;
    private final RetryFailedConsumer retryFailedConsumer;
    private final Sleeper sleeper;

    private DelayedHttpRequestRetryHandler(DelaySupplier delaySupplier, int maxRetries, RetryPredicate predicate, RetryConsumer retryConsumer, RetryFailedConsumer retryFailedConsumer, Sleeper sleeper) {
        this.delaySupplier = delaySupplier;
        this.maxRetries = maxRetries;
        this.predicate = predicate;
        this.retryConsumer = retryConsumer;
        this.retryFailedConsumer = retryFailedConsumer;
        this.sleeper = sleeper;
    }

    public boolean retryRequest(IOException exception, int executionCount, HttpContext ctx) {
        log.fine(() -> String.format("retryRequest(exception='%s', executionCount='%d', ctx='%s'", exception.getClass().getName(), executionCount, ctx));
        HttpClientContext clientCtx = HttpClientContext.adapt((HttpContext)ctx);
        if (!this.predicate.test(exception, clientCtx)) {
            log.fine(() -> String.format("Not retrying for '%s'", ctx));
            return false;
        }
        if (executionCount > this.maxRetries) {
            log.fine(() -> String.format("Max retries exceeded for '%s'", ctx));
            this.retryFailedConsumer.onRetryFailed(exception, executionCount, clientCtx);
            return false;
        }
        Duration delay = this.delaySupplier.getDelay(executionCount);
        log.fine(() -> String.format("Retrying after %s for '%s'", delay, ctx));
        this.retryConsumer.onRetry(exception, delay, executionCount, clientCtx);
        this.sleeper.sleep(delay);
        return true;
    }

    @FunctionalInterface
    private static interface DelaySupplier {
        public Duration getDelay(int var1);
    }

    static interface Sleeper {
        public void sleep(Duration var1);
    }

    public static class Builder {
        private final DelaySupplier delaySupplier;
        private final int maxRetries;
        private RetryPredicate predicate = (ioException, ctx) -> true;
        private RetryConsumer retryConsumer = (exception, delay, count, ctx) -> {};
        private RetryFailedConsumer retryFailedConsumer = (exception, count, ctx) -> {};
        private Sleeper sleeper = new DefaultSleeper();

        private Builder(DelaySupplier delaySupplier, int maxRetries) {
            this.delaySupplier = delaySupplier;
            this.maxRetries = maxRetries;
        }

        public static Builder withFixedDelay(Duration delay, int maxRetries) {
            return new Builder(executionCount -> delay, maxRetries);
        }

        public static Builder withExponentialBackoff(Duration startDelay, Duration maxDelay, int maxRetries) {
            return new Builder(executionCount -> {
                Duration nextDelay = startDelay;
                for (int i = 1; i < executionCount; ++i) {
                    nextDelay = nextDelay.multipliedBy(2L);
                }
                return maxDelay.compareTo(nextDelay) > 0 ? nextDelay : maxDelay;
            }, maxRetries);
        }

        public Builder retryForExceptions(List<Class<? extends IOException>> exceptionTypes) {
            this.predicate = (ioException, ctx) -> exceptionTypes.stream().anyMatch(type -> type.isInstance(ioException));
            return this;
        }

        public Builder retryForExceptions(Predicate<IOException> predicate) {
            this.predicate = (ioException, ctx) -> predicate.test((IOException)ioException);
            return this;
        }

        public Builder retryFor(RetryPredicate predicate) {
            this.predicate = predicate;
            return this;
        }

        public Builder onRetry(RetryConsumer consumer) {
            this.retryConsumer = consumer;
            return this;
        }

        public Builder onRetryFailed(RetryFailedConsumer consumer) {
            this.retryFailedConsumer = consumer;
            return this;
        }

        Builder withSleeper(Sleeper sleeper) {
            this.sleeper = sleeper;
            return this;
        }

        public DelayedHttpRequestRetryHandler build() {
            return new DelayedHttpRequestRetryHandler(this.delaySupplier, this.maxRetries, this.predicate, this.retryConsumer, this.retryFailedConsumer, this.sleeper);
        }

        private static class DefaultSleeper
        implements Sleeper {
            private DefaultSleeper() {
            }

            @Override
            public void sleep(Duration duration) {
                try {
                    Thread.sleep(duration.toMillis());
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    @FunctionalInterface
    public static interface RetryPredicate
    extends BiPredicate<IOException, HttpClientContext> {
    }

    @FunctionalInterface
    public static interface RetryFailedConsumer {
        public void onRetryFailed(IOException var1, int var2, HttpClientContext var3);
    }

    @FunctionalInterface
    public static interface RetryConsumer {
        public void onRetry(IOException var1, Duration var2, int var3, HttpClientContext var4);
    }
}

