/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.filedistribution;

import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.FileReference;
import com.yahoo.jrt.Int32Value;
import com.yahoo.jrt.Request;
import com.yahoo.jrt.StringValue;
import com.yahoo.jrt.Value;
import com.yahoo.vespa.config.Connection;
import com.yahoo.vespa.config.ConnectionPool;
import com.yahoo.vespa.filedistribution.FileReceiver;
import com.yahoo.vespa.filedistribution.FileReferenceDownload;
import java.io.File;
import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class FileReferenceDownloader {
    private static final Logger log = Logger.getLogger(FileReferenceDownloader.class.getName());
    private final ExecutorService downloadExecutor = Executors.newFixedThreadPool(Math.max(8, Runtime.getRuntime().availableProcessors()), (ThreadFactory)new DaemonThreadFactory("filereference downloader"));
    private final ConnectionPool connectionPool;
    private final Downloads downloads = new Downloads();
    private final DownloadStatuses downloadStatuses = new DownloadStatuses();
    private final Duration downloadTimeout;
    private final Duration sleepBetweenRetries;
    private final Duration rpcTimeout;

    FileReferenceDownloader(File downloadDirectory, File tmpDirectory, ConnectionPool connectionPool, Duration timeout, Duration sleepBetweenRetries) {
        this.connectionPool = connectionPool;
        this.downloadTimeout = timeout;
        this.sleepBetweenRetries = sleepBetweenRetries;
        new FileReceiver(connectionPool.getSupervisor(), this, downloadDirectory, tmpDirectory);
        String timeoutString = System.getenv("VESPA_CONFIGPROXY_FILEDOWNLOAD_RPC_TIMEOUT");
        this.rpcTimeout = Duration.ofSeconds(timeoutString == null ? 30L : (long)Integer.parseInt(timeoutString));
    }

    private void startDownload(FileReferenceDownload fileReferenceDownload) {
        FileReference fileReference = fileReferenceDownload.fileReference();
        Instant end = Instant.now().plus(this.downloadTimeout);
        boolean downloadStarted = false;
        int retryCount = 0;
        do {
            try {
                if (this.startDownloadRpc(fileReferenceDownload, retryCount)) {
                    downloadStarted = true;
                    continue;
                }
                ++retryCount;
                Thread.sleep(this.sleepBetweenRetries.toMillis());
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        } while (Instant.now().isBefore(end) && !downloadStarted);
        if (!downloadStarted) {
            fileReferenceDownload.future().completeExceptionally(new RuntimeException("Failed getting file reference '" + fileReference.value() + "'"));
            this.downloads.remove(fileReference);
        }
    }

    Future<Optional<File>> download(FileReferenceDownload fileReferenceDownload) {
        FileReference fileReference = fileReferenceDownload.fileReference();
        Optional<FileReferenceDownload> inProgress = this.downloads.get(fileReference);
        if (inProgress.isPresent()) {
            return inProgress.get().future();
        }
        log.log(Level.FINE, () -> "Will download file reference '" + fileReference.value() + "' with timeout " + this.downloadTimeout);
        this.downloads.add(fileReferenceDownload);
        this.downloadStatuses.add(fileReference);
        this.downloadExecutor.submit(() -> this.startDownload(fileReferenceDownload));
        return fileReferenceDownload.future();
    }

    void completedDownloading(FileReference fileReference, File file) {
        Optional<FileReferenceDownload> download = this.downloads.get(fileReference);
        if (download.isPresent()) {
            this.downloadStatuses.get(fileReference).ifPresent(DownloadStatus::finished);
            this.downloads.remove(fileReference);
            download.get().future().complete(Optional.of(file));
        } else {
            log.log(Level.FINE, () -> "Received '" + fileReference + "', which was not requested. Can be ignored if happening during upgrades/restarts");
        }
    }

    void failedDownloading(FileReference fileReference) {
        this.downloadStatuses.get(fileReference).ifPresent(d -> d.setProgress(0.0));
        this.downloads.remove(fileReference);
    }

    private boolean startDownloadRpc(FileReferenceDownload fileReferenceDownload, int retryCount) {
        Level logLevel;
        Connection connection = this.connectionPool.getCurrent();
        Request request = new Request("filedistribution.serveFile");
        String fileReference = fileReferenceDownload.fileReference().value();
        request.parameters().add((Value)new StringValue(fileReference));
        request.parameters().add((Value)new Int32Value(fileReferenceDownload.downloadFromOtherSourceIfNotFound() ? 0 : 1));
        connection.invokeSync(request, (double)this.rpcTimeout.getSeconds());
        Level level = logLevel = retryCount > 50 ? Level.INFO : Level.FINE;
        if (this.validateResponse(request)) {
            log.log(Level.FINE, () -> "Request callback, OK. Req: " + request + "\nSpec: " + connection + ", retry count " + retryCount);
            if (request.returnValues().get(0).asInt32() == 0) {
                log.log(Level.FINE, () -> "Found file reference '" + fileReference + "' available at " + connection.getAddress());
                return true;
            }
            log.log(logLevel, "File reference '" + fileReference + "' not found at " + connection.getAddress());
            this.connectionPool.setNewCurrentConnection();
            return false;
        }
        log.log(logLevel, () -> "Request failed. Req: " + request + "\nSpec: " + connection.getAddress() + ", error code: " + request.errorCode() + ", will use another spec for next request, retry count " + retryCount + ", rpc timeout " + this.rpcTimeout.getSeconds());
        this.connectionPool.setError(connection, request.errorCode());
        return false;
    }

    boolean isDownloading(FileReference fileReference) {
        return this.downloads.get(fileReference).isPresent();
    }

    private boolean validateResponse(Request request) {
        if (request.isError()) {
            return false;
        }
        if (request.returnValues().size() == 0) {
            return false;
        }
        if (!request.checkReturnTypes("is")) {
            log.log(Level.WARNING, "Invalid return types for response: " + request.errorMessage());
            return false;
        }
        return true;
    }

    double downloadStatus(String file) {
        double status = 0.0;
        Optional<DownloadStatus> downloadStatus = this.downloadStatuses.get(new FileReference(file));
        if (downloadStatus.isPresent()) {
            status = downloadStatus.get().progress();
        }
        return status;
    }

    void setDownloadStatus(FileReference fileReference, double completeness) {
        Optional<DownloadStatus> downloadStatus = this.downloadStatuses.get(fileReference);
        if (downloadStatus.isPresent()) {
            downloadStatus.get().setProgress(completeness);
        } else {
            this.downloadStatuses.add(fileReference, completeness);
        }
    }

    Map<FileReference, Double> downloadStatus() {
        return this.downloadStatuses.all().values().stream().collect(Collectors.toMap(DownloadStatus::fileReference, DownloadStatus::progress));
    }

    public ConnectionPool connectionPool() {
        return this.connectionPool;
    }

    public void close() {
        try {
            this.downloadExecutor.awaitTermination(1L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.interrupted();
        }
    }

    private static class DownloadStatuses {
        private static final int maxEntries = 100;
        private final Map<FileReference, DownloadStatus> downloadStatus = new ConcurrentHashMap<FileReference, DownloadStatus>();

        private DownloadStatuses() {
        }

        void add(FileReference fileReference) {
            this.add(fileReference, 0.0);
        }

        void add(FileReference fileReference, double progress) {
            DownloadStatus ds = new DownloadStatus(fileReference);
            ds.setProgress(progress);
            this.downloadStatus.put(fileReference, ds);
            if (this.downloadStatus.size() > 100) {
                Map.Entry oldest = Collections.min(this.downloadStatus.entrySet(), Comparator.comparing(e -> ((DownloadStatus)e.getValue()).created));
                this.downloadStatus.remove(oldest.getKey());
            }
        }

        Optional<DownloadStatus> get(FileReference fileReference) {
            return Optional.ofNullable(this.downloadStatus.get(fileReference));
        }

        Map<FileReference, DownloadStatus> all() {
            return Map.copyOf(this.downloadStatus);
        }
    }

    private static class DownloadStatus {
        private final FileReference fileReference;
        private double progress;
        private final Instant created;

        DownloadStatus(FileReference fileReference) {
            this.fileReference = fileReference;
            this.progress = 0.0;
            this.created = Instant.now();
        }

        public FileReference fileReference() {
            return this.fileReference;
        }

        public double progress() {
            return this.progress;
        }

        public void setProgress(double progress) {
            this.progress = progress;
        }

        public void finished() {
            this.setProgress(1.0);
        }

        public Instant created() {
            return this.created;
        }
    }

    private static class Downloads {
        private final Map<FileReference, FileReferenceDownload> downloads = new ConcurrentHashMap<FileReference, FileReferenceDownload>();

        private Downloads() {
        }

        void add(FileReferenceDownload fileReferenceDownload) {
            this.downloads.put(fileReferenceDownload.fileReference(), fileReferenceDownload);
        }

        void remove(FileReference fileReference) {
            this.downloads.remove(fileReference);
        }

        Optional<FileReferenceDownload> get(FileReference fileReference) {
            return Optional.ofNullable(this.downloads.get(fileReference));
        }
    }
}

