/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.search.dispatch.searchcluster;

import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.yahoo.container.handler.VipStatus;
import com.yahoo.net.HostName;
import com.yahoo.prelude.Pong;
import com.yahoo.search.cluster.ClusterMonitor;
import com.yahoo.search.cluster.NodeManager;
import com.yahoo.search.dispatch.searchcluster.Group;
import com.yahoo.search.dispatch.searchcluster.Node;
import com.yahoo.search.dispatch.searchcluster.PingFactory;
import com.yahoo.search.result.ErrorMessage;
import com.yahoo.vespa.config.search.DispatchConfig;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class SearchCluster
implements NodeManager<Node> {
    private static final Logger log = Logger.getLogger(SearchCluster.class.getName());
    private final DispatchConfig dispatchConfig;
    private final int size;
    private final String clusterId;
    private final ImmutableMap<Integer, Group> groups;
    private final ImmutableMultimap<String, Node> nodesByHost;
    private final ImmutableList<Group> orderedGroups;
    private final ClusterMonitor<Node> clusterMonitor;
    private final VipStatus vipStatus;
    private PingFactory pingFactory;
    private long nextLogTime = 0L;
    private final Optional<Node> localCorpusDispatchTarget;

    public SearchCluster(String clusterId, DispatchConfig dispatchConfig, int containerClusterSize, VipStatus vipStatus) {
        this.clusterId = clusterId;
        this.dispatchConfig = dispatchConfig;
        this.vipStatus = vipStatus;
        ImmutableList<Node> nodes = SearchCluster.toNodes(dispatchConfig);
        this.size = nodes.size();
        ImmutableMap.Builder groupsBuilder = new ImmutableMap.Builder();
        for (Map.Entry<Integer, List<Node>> group : nodes.stream().collect(Collectors.groupingBy(Node::group)).entrySet()) {
            Group g = new Group(group.getKey(), group.getValue());
            groupsBuilder.put((Object)group.getKey(), (Object)g);
        }
        this.groups = groupsBuilder.build();
        LinkedHashMap groupIntroductionOrder = new LinkedHashMap();
        nodes.forEach(node -> groupIntroductionOrder.put(node.group(), (Group)this.groups.get((Object)node.group())));
        this.orderedGroups = ImmutableList.builder().addAll(groupIntroductionOrder.values()).build();
        ImmutableMultimap.Builder nodesByHostBuilder = new ImmutableMultimap.Builder();
        for (Node node2 : nodes) {
            nodesByHostBuilder.put((Object)node2.hostname(), (Object)node2);
        }
        this.nodesByHost = nodesByHostBuilder.build();
        this.localCorpusDispatchTarget = SearchCluster.findLocalCorpusDispatchTarget(HostName.getLocalhost(), this.size, containerClusterSize, this.nodesByHost, this.groups);
        this.clusterMonitor = new ClusterMonitor<Node>(this);
    }

    public void shutDown() {
        this.clusterMonitor.shutdown();
    }

    public void startClusterMonitoring(PingFactory pingFactory) {
        this.pingFactory = pingFactory;
        for (Group group : this.orderedGroups) {
            for (Node node : group.nodes()) {
                this.clusterMonitor.add(node, true);
            }
        }
    }

    ClusterMonitor<Node> clusterMonitor() {
        return this.clusterMonitor;
    }

    private static Optional<Node> findLocalCorpusDispatchTarget(String selfHostname, int searchClusterSize, int containerClusterSize, ImmutableMultimap<String, Node> nodesByHost, ImmutableMap<Integer, Group> groups) {
        ImmutableCollection localSearchNodes = nodesByHost.get((Object)selfHostname);
        if (localSearchNodes.size() != 1) {
            return Optional.empty();
        }
        Node localSearchNode = (Node)localSearchNodes.iterator().next();
        Group localSearchGroup = (Group)groups.get((Object)localSearchNode.group());
        if (localSearchGroup.nodes().size() != 1) {
            return Optional.empty();
        }
        if (containerClusterSize < searchClusterSize) {
            return Optional.empty();
        }
        return Optional.of(localSearchNode);
    }

    private static ImmutableList<Node> toNodes(DispatchConfig dispatchConfig) {
        ImmutableList.Builder nodesBuilder = new ImmutableList.Builder();
        for (DispatchConfig.Node node : dispatchConfig.node()) {
            nodesBuilder.add((Object)new Node(node.key(), node.host(), node.group()));
        }
        return nodesBuilder.build();
    }

    public DispatchConfig dispatchConfig() {
        return this.dispatchConfig;
    }

    public int size() {
        return this.size;
    }

    public ImmutableMap<Integer, Group> groups() {
        return this.groups;
    }

    public ImmutableList<Group> orderedGroups() {
        return this.orderedGroups;
    }

    public Optional<Group> group(int n) {
        if (this.orderedGroups.size() > n) {
            return Optional.of((Group)this.orderedGroups.get(n));
        }
        return Optional.empty();
    }

    public int groupSize() {
        if (this.groups.size() == 0) {
            return this.size();
        }
        return this.size() / this.groups.size();
    }

    public int groupsWithSufficientCoverage() {
        int covered = 0;
        for (Group g : this.orderedGroups) {
            if (!g.hasSufficientCoverage()) continue;
            ++covered;
        }
        return covered;
    }

    public Optional<Node> localCorpusDispatchTarget() {
        if (this.localCorpusDispatchTarget.isEmpty()) {
            return Optional.empty();
        }
        Group localSearchGroup = (Group)this.groups.get((Object)this.localCorpusDispatchTarget.get().group());
        if (!localSearchGroup.hasSufficientCoverage()) {
            return Optional.empty();
        }
        if (this.localCorpusDispatchTarget.get().isWorking() == Boolean.FALSE) {
            return Optional.empty();
        }
        return this.localCorpusDispatchTarget;
    }

    private void updateWorkingState(Node node, boolean isWorking) {
        node.setWorking(isWorking);
        this.updateVipStatusOnNodeChange(node, isWorking);
    }

    @Override
    public void working(Node node) {
        this.updateWorkingState(node, true);
    }

    @Override
    public void failed(Node node) {
        this.updateWorkingState(node, false);
    }

    private void updateSufficientCoverage(Group group, boolean sufficientCoverage) {
        if (sufficientCoverage == group.hasSufficientCoverage()) {
            return;
        }
        group.setHasSufficientCoverage(sufficientCoverage);
        this.updateVipStatusOnCoverageChange(group, sufficientCoverage);
    }

    private void updateVipStatusOnNodeChange(Node node, boolean nodeIsWorking) {
        if (this.localCorpusDispatchTarget.isEmpty()) {
            if (this.hasInformationAboutAllNodes()) {
                this.setInRotationOnlyIf(this.hasWorkingNodes());
            }
        } else if (this.usesLocalCorpusIn(node)) {
            this.setInRotationOnlyIf(nodeIsWorking);
        }
    }

    private void updateVipStatusOnCoverageChange(Group group, boolean sufficientCoverage) {
        if (!this.localCorpusDispatchTarget.isEmpty() && this.usesLocalCorpusIn(group)) {
            this.setInRotationOnlyIf(sufficientCoverage);
        }
    }

    private void setInRotationOnlyIf(boolean inRotation) {
        if (inRotation) {
            this.vipStatus.addToRotation(this.clusterId);
        } else {
            this.vipStatus.removeFromRotation(this.clusterId);
        }
    }

    private boolean hasInformationAboutAllNodes() {
        return this.nodesByHost.values().stream().allMatch(node -> node.isWorking() != null);
    }

    private boolean hasWorkingNodes() {
        return this.nodesByHost.values().stream().anyMatch(node -> node.isWorking() != Boolean.FALSE);
    }

    private boolean usesLocalCorpusIn(Node node) {
        return this.localCorpusDispatchTarget.isPresent() && this.localCorpusDispatchTarget.get().equals(node);
    }

    private boolean usesLocalCorpusIn(Group group) {
        return this.localCorpusDispatchTarget.isPresent() && this.localCorpusDispatchTarget.get().group() == group.id();
    }

    @Override
    public void ping(Node node, Executor executor) {
        if (this.pingFactory == null) {
            return;
        }
        FutureTask<Pong> futurePong = new FutureTask<Pong>(this.pingFactory.createPinger(node, this.clusterMonitor));
        executor.execute(futurePong);
        Pong pong = this.getPong(futurePong, node);
        futurePong.cancel(true);
        if (pong.badResponse()) {
            this.clusterMonitor.failed(node, pong.error().get());
        } else {
            if (pong.activeDocuments().isPresent()) {
                node.setActiveDocuments(pong.activeDocuments().get());
            }
            this.clusterMonitor.responded(node);
        }
    }

    private void pingIterationCompletedSingleGroup() {
        Group group = (Group)this.groups.values().iterator().next();
        group.aggregateActiveDocuments();
        this.updateSufficientCoverage(group, true);
        boolean fullCoverage = this.isGroupCoverageSufficient(group.workingNodes(), group.nodes().size(), group.getActiveDocuments(), group.getActiveDocuments());
        this.trackGroupCoverageChanges(0, group, fullCoverage, group.getActiveDocuments());
    }

    private void pingIterationCompletedMultipleGroups() {
        int numGroups = this.orderedGroups.size();
        long[] activeDocumentsInGroup = new long[numGroups];
        long sumOfActiveDocuments = 0L;
        for (int i = 0; i < numGroups; ++i) {
            Group group = (Group)this.orderedGroups.get(i);
            group.aggregateActiveDocuments();
            activeDocumentsInGroup[i] = group.getActiveDocuments();
            sumOfActiveDocuments += activeDocumentsInGroup[i];
        }
        boolean anyGroupsSufficientCoverage = false;
        for (int i = 0; i < numGroups; ++i) {
            Group group = (Group)this.orderedGroups.get(i);
            long activeDocuments = activeDocumentsInGroup[i];
            long averageDocumentsInOtherGroups = (sumOfActiveDocuments - activeDocuments) / (long)(numGroups - 1);
            boolean sufficientCoverage = this.isGroupCoverageSufficient(group.workingNodes(), group.nodes().size(), activeDocuments, averageDocumentsInOtherGroups);
            anyGroupsSufficientCoverage = anyGroupsSufficientCoverage || sufficientCoverage;
            this.updateSufficientCoverage(group, sufficientCoverage);
            this.trackGroupCoverageChanges(i, group, sufficientCoverage, averageDocumentsInOtherGroups);
        }
    }

    @Override
    public void pingIterationCompleted() {
        int numGroups = this.orderedGroups.size();
        if (numGroups == 1) {
            this.pingIterationCompletedSingleGroup();
        } else {
            this.pingIterationCompletedMultipleGroups();
        }
    }

    private boolean isGroupCoverageSufficient(int workingNodes, int nodesInGroup, long activeDocuments, long averageDocumentsInOtherGroups) {
        boolean sufficientCoverage = true;
        if (averageDocumentsInOtherGroups > 0L) {
            double coverage = 100.0 * (double)activeDocuments / (double)averageDocumentsInOtherGroups;
            boolean bl = sufficientCoverage = coverage >= this.dispatchConfig.minActivedocsPercentage();
        }
        if (sufficientCoverage) {
            sufficientCoverage = this.isGroupNodeCoverageSufficient(workingNodes, nodesInGroup);
        }
        return sufficientCoverage;
    }

    private boolean isGroupNodeCoverageSufficient(int workingNodes, int nodesInGroup) {
        int nodesAllowedDown = this.dispatchConfig.maxNodesDownPerGroup() + (int)((double)nodesInGroup * (100.0 - this.dispatchConfig.minGroupCoverage()) / 100.0);
        return workingNodes + nodesAllowedDown >= nodesInGroup;
    }

    private Pong getPong(FutureTask<Pong> futurePong, Node node) {
        try {
            return futurePong.get(this.clusterMonitor.getConfiguration().getFailLimit(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            log.log(Level.WARNING, "Exception pinging " + node, e);
            return new Pong(ErrorMessage.createUnspecifiedError("Ping was interrupted: " + node));
        }
        catch (ExecutionException e) {
            log.log(Level.WARNING, "Exception pinging " + node, e);
            return new Pong(ErrorMessage.createUnspecifiedError("Execution was interrupted: " + node));
        }
        catch (TimeoutException e) {
            return new Pong(ErrorMessage.createNoAnswerWhenPingingNode("Ping thread timed out"));
        }
    }

    public boolean isPartialGroupCoverageSufficient(OptionalInt knownGroupId, List<Node> nodes) {
        if (this.orderedGroups.size() == 1) {
            boolean sufficient = nodes.size() >= this.groupSize() - this.dispatchConfig.maxNodesDownPerGroup();
            return sufficient;
        }
        if (knownGroupId.isEmpty()) {
            return false;
        }
        int groupId = knownGroupId.getAsInt();
        Group group = (Group)this.groups.get((Object)groupId);
        if (group == null) {
            return false;
        }
        int nodesInGroup = group.nodes().size();
        long sumOfActiveDocuments = 0L;
        int otherGroups = 0;
        for (Group g : this.orderedGroups) {
            if (g.id() == groupId) continue;
            sumOfActiveDocuments += g.getActiveDocuments();
            ++otherGroups;
        }
        long activeDocuments = 0L;
        for (Node n : nodes) {
            activeDocuments += n.getActiveDocuments();
        }
        long averageDocumentsInOtherGroups = sumOfActiveDocuments / (long)otherGroups;
        return this.isGroupCoverageSufficient(nodes.size(), nodesInGroup, activeDocuments, averageDocumentsInOtherGroups);
    }

    private void trackGroupCoverageChanges(int index, Group group, boolean fullCoverage, long averageDocuments) {
        boolean changed = group.isFullCoverageStatusChanged(fullCoverage);
        if (changed || !fullCoverage && System.currentTimeMillis() > this.nextLogTime) {
            this.nextLogTime = System.currentTimeMillis() + 30000L;
            int requiredNodes = this.groupSize() - this.dispatchConfig.maxNodesDownPerGroup();
            if (fullCoverage) {
                log.info(() -> String.format("Group %d is now good again (%d/%d active docs, coverage %d/%d)", index, group.getActiveDocuments(), averageDocuments, group.workingNodes(), this.groupSize()));
            } else {
                StringBuilder missing = new StringBuilder();
                for (Node node : group.nodes()) {
                    if (node.isWorking() == Boolean.TRUE) continue;
                    missing.append('\n').append(node.toString());
                }
                log.warning(() -> String.format("Coverage of group %d is only %d/%d (requires %d) (%d/%d active docs) Failed nodes are:%s", index, group.workingNodes(), this.groupSize(), requiredNodes, group.getActiveDocuments(), averageDocuments, missing.toString()));
            }
        }
    }
}

