/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.jdisc.http.server.jetty;

import com.yahoo.container.logging.ConnectionLog;
import com.yahoo.container.logging.ConnectionLogEntry;
import com.yahoo.io.HexDump;
import com.yahoo.jdisc.http.ServerConfig;
import com.yahoo.jdisc.http.server.jetty.SimpleConcurrentIdentityHashMap;
import com.yahoo.jdisc.http.server.jetty.SslHandshakeFailure;
import java.net.InetSocketAddress;
import java.security.cert.X509Certificate;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.ExtendedSSLSession;
import javax.net.ssl.SNIHostName;
import javax.net.ssl.SNIServerName;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import org.eclipse.jetty.alpn.server.ALPNServerConnection;
import org.eclipse.jetty.http2.server.HTTP2ServerConnection;
import org.eclipse.jetty.io.Connection;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.SocketChannelEndPoint;
import org.eclipse.jetty.io.ssl.SslConnection;
import org.eclipse.jetty.io.ssl.SslHandshakeListener;
import org.eclipse.jetty.server.HttpChannel;
import org.eclipse.jetty.server.HttpConnection;
import org.eclipse.jetty.server.ProxyConnectionFactory;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.util.component.AbstractLifeCycle;

class JettyConnectionLogger
extends AbstractLifeCycle
implements Connection.Listener,
HttpChannel.Listener,
SslHandshakeListener {
    static final String CONNECTION_ID_REQUEST_ATTRIBUTE = "jdisc.request.connection.id";
    private static final Logger log = Logger.getLogger(JettyConnectionLogger.class.getName());
    private final SimpleConcurrentIdentityHashMap<SocketChannelEndPoint, ConnectionInfo> connectionInfos = new SimpleConcurrentIdentityHashMap();
    private final SimpleConcurrentIdentityHashMap<SocketChannelEndPoint, SSLEngine> sslEngines = new SimpleConcurrentIdentityHashMap();
    private final SimpleConcurrentIdentityHashMap<SSLEngine, ConnectionInfo> sslToConnectionInfo = new SimpleConcurrentIdentityHashMap();
    private final boolean enabled;
    private final ConnectionLog connectionLog;

    JettyConnectionLogger(ServerConfig.ConnectionLog config, ConnectionLog connectionLog) {
        this.enabled = config.enabled();
        this.connectionLog = connectionLog;
        log.log(Level.FINE, () -> "Jetty connection logger is " + (config.enabled() ? "enabled" : "disabled"));
    }

    protected void doStop() {
        this.handleListenerInvocation("AbstractLifeCycle", "doStop", "", List.of(), () -> log.log(Level.FINE, () -> "Jetty connection logger is stopped"));
    }

    protected void doStart() {
        this.handleListenerInvocation("AbstractLifeCycle", "doStart", "", List.of(), () -> log.log(Level.FINE, () -> "Jetty connection logger is started"));
    }

    public void onOpened(Connection connection) {
        this.handleListenerInvocation("Connection.Listener", "onOpened", "%h", List.of(connection), () -> {
            SSLEngine sslEngine;
            SocketChannelEndPoint endpoint = JettyConnectionLogger.findUnderlyingSocketEndpoint(connection.getEndPoint());
            ConnectionInfo info = this.connectionInfos.computeIfAbsent(endpoint, ConnectionInfo::from);
            String connectionClassName = connection.getClass().getSimpleName();
            if (connection instanceof SslConnection) {
                sslEngine = ((SslConnection)connection).getSSLEngine();
                this.addReferenceToSslEngine(endpoint, info, sslEngine);
            } else if (connection instanceof ALPNServerConnection) {
                sslEngine = ((ALPNServerConnection)connection).getSSLEngine();
                this.addReferenceToSslEngine(endpoint, info, sslEngine);
            } else if (connection instanceof HttpConnection) {
                info.setHttpProtocol("HTTP/1.1");
            } else if (connection instanceof HTTP2ServerConnection) {
                info.setHttpProtocol("HTTP/2.0");
            } else if (connectionClassName.endsWith("ProxyProtocolV1Connection")) {
                info.setProxyProtocolVersion("v1");
            } else if (connectionClassName.endsWith("ProxyProtocolV2Connection")) {
                info.setProxyProtocolVersion("v2");
            }
            if (connection.getEndPoint() instanceof ProxyConnectionFactory.ProxyEndPoint) {
                InetSocketAddress remoteAddress = connection.getEndPoint().getRemoteAddress();
                info.setRemoteAddress(remoteAddress);
            }
        });
    }

    private void addReferenceToSslEngine(SocketChannelEndPoint endpoint, ConnectionInfo info, SSLEngine sslEngine) {
        if (sslEngine != null) {
            this.sslEngines.put(endpoint, sslEngine).ifPresent(this.sslToConnectionInfo::remove);
            this.sslToConnectionInfo.put(sslEngine, info);
        }
    }

    public void onClosed(Connection connection) {
        this.handleListenerInvocation("Connection.Listener", "onClosed", "%h", List.of(connection), () -> {
            SocketChannelEndPoint endpoint = JettyConnectionLogger.findUnderlyingSocketEndpoint(connection.getEndPoint());
            ConnectionInfo info = this.connectionInfos.get(endpoint).orElse(null);
            if (info == null) {
                return;
            }
            if (connection instanceof HttpConnection) {
                info.setHttpBytes(connection.getBytesIn(), connection.getBytesOut());
            }
            if (!endpoint.isOpen()) {
                info.setClosedAt(System.currentTimeMillis());
                this.connectionLog.log(info.toLogEntry());
                this.connectionInfos.remove(endpoint);
                this.sslEngines.remove(endpoint).ifPresent(this.sslToConnectionInfo::remove);
            }
        });
    }

    public void onRequestBegin(Request request) {
        this.handleListenerInvocation("HttpChannel.Listener", "onRequestBegin", "%h", List.of(request), () -> {
            SocketChannelEndPoint endpoint = JettyConnectionLogger.findUnderlyingSocketEndpoint(request.getHttpChannel().getEndPoint());
            ConnectionInfo info = this.connectionInfos.get(endpoint).get();
            info.incrementRequests();
            request.setAttribute(CONNECTION_ID_REQUEST_ATTRIBUTE, (Object)info.uuid());
        });
    }

    public void onResponseBegin(Request request) {
        this.handleListenerInvocation("HttpChannel.Listener", "onResponseBegin", "%h", List.of(request), () -> {
            SocketChannelEndPoint endpoint = JettyConnectionLogger.findUnderlyingSocketEndpoint(request.getHttpChannel().getEndPoint());
            ConnectionInfo info = this.connectionInfos.get(endpoint).orElse(null);
            if (info == null) {
                return;
            }
            info.incrementResponses();
        });
    }

    public void handshakeSucceeded(SslHandshakeListener.Event event) {
        SSLEngine sslEngine = event.getSSLEngine();
        this.handleListenerInvocation("SslHandshakeListener", "handshakeSucceeded", "sslEngine=%h", List.of(sslEngine), () -> {
            ConnectionInfo info = this.sslToConnectionInfo.get(sslEngine).orElse(null);
            if (info == null) {
                return;
            }
            info.setSslSessionDetails(sslEngine.getSession());
        });
    }

    public void handshakeFailed(SslHandshakeListener.Event event, Throwable failure) {
        SSLEngine sslEngine = event.getSSLEngine();
        this.handleListenerInvocation("SslHandshakeListener", "handshakeFailed", "sslEngine=%h,failure=%s", List.of(sslEngine, failure), () -> {
            log.log(Level.FINE, failure, failure::toString);
            ConnectionInfo info = this.sslToConnectionInfo.get(sslEngine).orElse(null);
            if (info == null) {
                return;
            }
            info.setSslHandshakeFailure((SSLHandshakeException)failure);
        });
    }

    private void handleListenerInvocation(String listenerType, String methodName, String methodArgumentsFormat, List<Object> methodArguments, ListenerHandler handler) {
        if (!this.enabled) {
            return;
        }
        try {
            log.log(Level.FINE, () -> String.format(listenerType + "." + methodName + "(" + methodArgumentsFormat + ")", methodArguments.toArray()));
            handler.run();
        }
        catch (Exception e) {
            log.log(Level.WARNING, String.format("Exception in %s.%s listener: %s", listenerType, methodName, e.getMessage()), e);
        }
    }

    private static SocketChannelEndPoint findUnderlyingSocketEndpoint(EndPoint endpoint) {
        if (endpoint instanceof SocketChannelEndPoint) {
            return (SocketChannelEndPoint)endpoint;
        }
        if (endpoint instanceof SslConnection.DecryptedEndPoint) {
            SslConnection.DecryptedEndPoint decryptedEndpoint = (SslConnection.DecryptedEndPoint)endpoint;
            return JettyConnectionLogger.findUnderlyingSocketEndpoint(decryptedEndpoint.getSslConnection().getEndPoint());
        }
        if (endpoint instanceof ProxyConnectionFactory.ProxyEndPoint) {
            ProxyConnectionFactory.ProxyEndPoint proxyEndpoint = (ProxyConnectionFactory.ProxyEndPoint)endpoint;
            return JettyConnectionLogger.findUnderlyingSocketEndpoint(proxyEndpoint.unwrap());
        }
        throw new IllegalArgumentException("Unknown connection endpoint type: " + endpoint.getClass().getName());
    }

    private static class ConnectionInfo {
        private final UUID uuid;
        private final long createdAt;
        private final InetSocketAddress localAddress;
        private final InetSocketAddress peerAddress;
        private long closedAt = 0L;
        private long httpBytesReceived = 0L;
        private long httpBytesSent = 0L;
        private long requests = 0L;
        private long responses = 0L;
        private InetSocketAddress remoteAddress;
        private byte[] sslSessionId;
        private String sslProtocol;
        private String sslCipherSuite;
        private String sslPeerSubject;
        private Date sslPeerNotBefore;
        private Date sslPeerNotAfter;
        private List<SNIServerName> sslSniServerNames;
        private SSLHandshakeException sslHandshakeException;
        private String proxyProtocolVersion;
        private String httpProtocol;

        private ConnectionInfo(UUID uuid, long createdAt, InetSocketAddress localAddress, InetSocketAddress peerAddress) {
            this.uuid = uuid;
            this.createdAt = createdAt;
            this.localAddress = localAddress;
            this.peerAddress = peerAddress;
        }

        static ConnectionInfo from(SocketChannelEndPoint endpoint) {
            return new ConnectionInfo(UUID.randomUUID(), endpoint.getCreatedTimeStamp(), endpoint.getLocalAddress(), endpoint.getRemoteAddress());
        }

        synchronized UUID uuid() {
            return this.uuid;
        }

        synchronized ConnectionInfo setClosedAt(long closedAt) {
            this.closedAt = closedAt;
            return this;
        }

        synchronized ConnectionInfo setHttpBytes(long received, long sent) {
            this.httpBytesReceived = received;
            this.httpBytesSent = sent;
            return this;
        }

        synchronized ConnectionInfo incrementRequests() {
            ++this.requests;
            return this;
        }

        synchronized ConnectionInfo incrementResponses() {
            ++this.responses;
            return this;
        }

        synchronized ConnectionInfo setRemoteAddress(InetSocketAddress remoteAddress) {
            this.remoteAddress = remoteAddress;
            return this;
        }

        synchronized ConnectionInfo setSslSessionDetails(SSLSession session) {
            this.sslCipherSuite = session.getCipherSuite();
            this.sslProtocol = session.getProtocol();
            this.sslSessionId = session.getId();
            if (session instanceof ExtendedSSLSession) {
                ExtendedSSLSession extendedSession = (ExtendedSSLSession)session;
                this.sslSniServerNames = extendedSession.getRequestedServerNames();
            }
            try {
                this.sslPeerSubject = session.getPeerPrincipal().getName();
                X509Certificate peerCertificate = (X509Certificate)session.getPeerCertificates()[0];
                this.sslPeerNotBefore = peerCertificate.getNotBefore();
                this.sslPeerNotAfter = peerCertificate.getNotAfter();
            }
            catch (SSLPeerUnverifiedException sSLPeerUnverifiedException) {
                // empty catch block
            }
            return this;
        }

        synchronized ConnectionInfo setSslHandshakeFailure(SSLHandshakeException exception) {
            this.sslHandshakeException = exception;
            return this;
        }

        synchronized ConnectionInfo setHttpProtocol(String protocol) {
            this.httpProtocol = protocol;
            return this;
        }

        synchronized ConnectionInfo setProxyProtocolVersion(String version) {
            this.proxyProtocolVersion = version;
            return this;
        }

        synchronized ConnectionLogEntry toLogEntry() {
            ConnectionLogEntry.Builder builder = ConnectionLogEntry.builder(this.uuid, Instant.ofEpochMilli(this.createdAt));
            if (this.closedAt > 0L) {
                builder.withDuration((double)(this.closedAt - this.createdAt) / 1000.0);
            }
            if (this.httpBytesReceived > 0L) {
                builder.withHttpBytesReceived(this.httpBytesReceived);
            }
            if (this.httpBytesSent > 0L) {
                builder.withHttpBytesSent(this.httpBytesSent);
            }
            if (this.requests > 0L) {
                builder.withRequests(this.requests);
            }
            if (this.responses > 0L) {
                builder.withResponses(this.responses);
            }
            if (this.peerAddress != null) {
                builder.withPeerAddress(this.peerAddress.getHostString()).withPeerPort(this.peerAddress.getPort());
            }
            if (this.localAddress != null) {
                builder.withLocalAddress(this.localAddress.getHostString()).withLocalPort(this.localAddress.getPort());
            }
            if (this.remoteAddress != null) {
                builder.withRemoteAddress(this.remoteAddress.getHostString()).withRemotePort(this.remoteAddress.getPort());
            }
            if (this.sslProtocol != null && this.sslCipherSuite != null && this.sslSessionId != null) {
                builder.withSslProtocol(this.sslProtocol).withSslCipherSuite(this.sslCipherSuite).withSslSessionId(HexDump.toHexString((byte[])this.sslSessionId));
            }
            if (this.sslSniServerNames != null) {
                this.sslSniServerNames.stream().filter(name -> name instanceof SNIHostName && name.getType() == 0).map(name -> ((SNIHostName)name).getAsciiName()).findAny().ifPresent(builder::withSslSniServerName);
            }
            if (this.sslPeerSubject != null && this.sslPeerNotAfter != null && this.sslPeerNotBefore != null) {
                builder.withSslPeerSubject(this.sslPeerSubject).withSslPeerNotAfter(this.sslPeerNotAfter.toInstant()).withSslPeerNotBefore(this.sslPeerNotBefore.toInstant());
            }
            if (this.sslHandshakeException != null) {
                ArrayList<ConnectionLogEntry.SslHandshakeFailure.ExceptionEntry> exceptionChain = new ArrayList<ConnectionLogEntry.SslHandshakeFailure.ExceptionEntry>();
                for (Throwable cause = this.sslHandshakeException; cause != null; cause = cause.getCause()) {
                    exceptionChain.add(new ConnectionLogEntry.SslHandshakeFailure.ExceptionEntry(cause.getClass().getName(), cause.getMessage()));
                }
                String type = SslHandshakeFailure.fromSslHandshakeException(this.sslHandshakeException).map(SslHandshakeFailure::failureType).orElse("UNKNOWN");
                builder.withSslHandshakeFailure(new ConnectionLogEntry.SslHandshakeFailure(type, exceptionChain));
            }
            if (this.httpProtocol != null) {
                builder.withHttpProtocol(this.httpProtocol);
            }
            if (this.proxyProtocolVersion != null) {
                builder.withProxyProtocolVersion(this.proxyProtocolVersion);
            }
            return builder.build();
        }
    }

    @FunctionalInterface
    private static interface ListenerHandler {
        public void run() throws Exception;
    }
}

