/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.filedistribution;

import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.component.AbstractComponent;
import com.yahoo.component.annotation.Inject;
import com.yahoo.concurrent.Lock;
import com.yahoo.concurrent.Locks;
import com.yahoo.config.FileReference;
import com.yahoo.io.IOUtils;
import com.yahoo.text.Utf8;
import com.yahoo.vespa.defaults.Defaults;
import com.yahoo.yolean.Exceptions;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Clock;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.jpountz.xxhash.XXHash64;
import net.jpountz.xxhash.XXHashFactory;

public class FileDirectory
extends AbstractComponent {
    private static final Logger log = Logger.getLogger(FileDirectory.class.getName());
    private final Locks<FileReference> locks = new Locks(1L, TimeUnit.MINUTES);
    private final File root;

    @Inject
    public FileDirectory(ConfigserverConfig configserverConfig) {
        this(new File(Defaults.getDefaults().underVespaHome(configserverConfig.fileReferencesDir())));
    }

    public FileDirectory(File rootDir) {
        this.root = rootDir;
        try {
            this.ensureRootExist();
        }
        catch (IllegalArgumentException e) {
            log.log(Level.WARNING, "Failed creating directory in constructor, will retry on demand : " + e.getMessage());
        }
    }

    private void ensureRootExist() {
        if (!this.root.exists()) {
            if (!this.root.mkdir()) {
                throw new IllegalArgumentException("Failed creating root dir '" + this.root.getAbsolutePath() + "'.");
            }
        } else if (!this.root.isDirectory()) {
            throw new IllegalArgumentException("'" + this.root.getAbsolutePath() + "' is not a directory");
        }
    }

    String getPath(FileReference ref) {
        return this.root.getAbsolutePath() + "/" + ref.value();
    }

    public Optional<File> getFile(FileReference reference) {
        this.ensureRootExist();
        File dir = new File(this.getPath(reference));
        if (!dir.exists()) {
            log.log(Level.FINEST, "File reference '" + reference.value() + "' ('" + dir.getAbsolutePath() + "') does not exist.");
            return Optional.empty();
        }
        if (!dir.isDirectory()) {
            log.log(Level.INFO, "File reference '" + reference.value() + "' ('" + dir.getAbsolutePath() + ")' is not a directory.");
            return Optional.empty();
        }
        File[] files = dir.listFiles(new Filter());
        if (files == null || files.length == 0) {
            log.log(Level.INFO, "File reference '" + reference.value() + "' ('" + dir.getAbsolutePath() + "') does not contain any files");
            return Optional.empty();
        }
        return Optional.of(files[0]);
    }

    public File getRoot() {
        return this.root;
    }

    private Long computeHash(File file) throws IOException {
        XXHash64 hasher = XXHashFactory.fastestInstance().hash64();
        if (file.isDirectory()) {
            return Files.walk(file.toPath(), 100, new FileVisitOption[0]).map(path -> {
                try {
                    log.log(Level.FINEST, () -> "Calculating hash for '" + String.valueOf(path) + "'");
                    return this.hash(path.toFile(), hasher);
                }
                catch (IOException e) {
                    log.log(Level.WARNING, "Failed getting hash from '" + String.valueOf(path) + "'");
                    return 0;
                }
            }).mapToLong(rec$ -> ((Number)rec$).longValue()).sum();
        }
        return this.hash(file, hasher);
    }

    private long hash(File file, XXHash64 hasher) throws IOException {
        byte[] wholeFile = file.isDirectory() ? new byte[]{} : IOUtils.readFileBytes((File)file);
        return hasher.hash(ByteBuffer.wrap(wholeFile), hasher.hash(ByteBuffer.wrap(Utf8.toBytes((String)file.getName())), 0L));
    }

    public FileReference addFile(File source) throws IOException {
        Long hash = this.computeHash(source);
        FileReference fileReference = this.fileReferenceFromHash(hash);
        try (Lock lock = this.locks.lock((Object)fileReference);){
            FileReference fileReference2 = this.addFile(source, fileReference, hash);
            return fileReference2;
        }
    }

    public boolean delete(FileReference fileReference, Function<FileReference, Boolean> isInUse, Function<File, Boolean> isOld) {
        try (Lock lock = this.locks.lock((Object)fileReference);){
            if (isInUse.apply(fileReference).booleanValue()) {
                log.log(Level.FINE, "Unable to delete file reference '" + fileReference.value() + "' since it is still in use");
            } else if (!isOld.apply(new File(this.getRoot(), fileReference.value())).booleanValue()) {
                log.log(Level.FINE, "Unable to delete file reference '" + fileReference.value() + "' since it is recently used");
            } else {
                this.deleteDirRecursively(this.destinationDir(fileReference));
                log.log(Level.FINE, "Deleted file reference '" + fileReference.value() + "'");
                boolean bl = true;
                return bl;
            }
            boolean bl = false;
            return bl;
        }
    }

    private void deleteDirRecursively(File dir) {
        log.log(Level.FINEST, "Will delete dir " + String.valueOf(dir));
        if (!IOUtils.recursiveDeleteDir((File)dir)) {
            log.log(Level.INFO, "Failed to delete " + String.valueOf(dir));
        }
    }

    private boolean shouldAddFile(File source, Long hashOfFileToBeAdded) throws IOException {
        FileReference fileReference = this.fileReferenceFromHash(hashOfFileToBeAdded);
        File destinationDir = this.destinationDir(fileReference);
        if (!destinationDir.exists()) {
            return true;
        }
        File existingFile = destinationDir.toPath().resolve(source.getName()).toFile();
        if (!existingFile.exists() || !this.computeHash(existingFile).equals(hashOfFileToBeAdded)) {
            log.log(Level.WARNING, "Directory for file reference '" + fileReference.value() + "' has content that does not match its hash, deleting everything in " + destinationDir.getAbsolutePath());
            this.deleteDirRecursively(destinationDir);
            return true;
        }
        destinationDir.setLastModified(Clock.systemUTC().instant().toEpochMilli());
        log.log(Level.FINE, "Directory for file reference '" + fileReference.value() + "' already exists and has all content");
        return false;
    }

    private File destinationDir(FileReference fileReference) {
        return new File(this.root, fileReference.value());
    }

    private FileReference fileReferenceFromHash(Long hash) {
        return new FileReference(Long.toHexString(hash));
    }

    private FileReference addFile(File source, FileReference reference, Long hash) throws IOException {
        if (!this.shouldAddFile(source, hash)) {
            return reference;
        }
        this.ensureRootExist();
        Path tempDestinationDir = (Path)Exceptions.uncheck(() -> Files.createTempDirectory(this.root.toPath(), "writing", new FileAttribute[0]));
        try {
            this.logfileInfo(source);
            File tempDestination = new File(tempDestinationDir.toFile(), source.getName());
            log.log(Level.FINE, () -> "Copying " + source.getAbsolutePath() + " to " + tempDestination.getAbsolutePath());
            if (source.isDirectory()) {
                IOUtils.copyDirectory((File)source, (File)tempDestination, (int)-1);
            } else {
                FileDirectory.copyFile(source, tempDestination);
            }
            Path destinationDir = this.destinationDir(reference).toPath();
            log.log(Level.FINE, () -> "Moving " + String.valueOf(tempDestinationDir) + " to " + String.valueOf(destinationDir));
            Files.move(tempDestinationDir, destinationDir, new CopyOption[0]);
            FileReference fileReference = reference;
            return fileReference;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        finally {
            IOUtils.recursiveDeleteDir((File)tempDestinationDir.toFile());
        }
    }

    private void logfileInfo(File file) throws IOException {
        BasicFileAttributes basicFileAttributes = Files.readAttributes(file.toPath(), BasicFileAttributes.class, new LinkOption[0]);
        log.log(Level.FINE, () -> "Adding file " + file.getAbsolutePath() + " (created " + String.valueOf(basicFileAttributes.creationTime()) + ", modified " + String.valueOf(basicFileAttributes.lastModifiedTime()) + ", size " + basicFileAttributes.size() + ")");
    }

    private static void copyFile(File source, File dest) throws IOException {
        try (FileChannel sourceChannel = new FileInputStream(source).getChannel();
             FileChannel destChannel = new FileOutputStream(dest).getChannel();){
            destChannel.transferFrom(sourceChannel, 0L, sourceChannel.size());
        }
    }

    public String toString() {
        return "root dir: " + this.root.getAbsolutePath();
    }

    private static class Filter
    implements FilenameFilter {
        private Filter() {
        }

        @Override
        public boolean accept(File dir, String name) {
            return !".".equals(name) && !"..".equals(name);
        }
    }
}

