/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.rpc.security;

import com.yahoo.cloud.config.SentinelConfig;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.FileReference;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.config.provision.security.NodeIdentifier;
import com.yahoo.config.provision.security.NodeIdentifierException;
import com.yahoo.config.provision.security.NodeIdentity;
import com.yahoo.jrt.Request;
import com.yahoo.security.tls.ConnectionAuthContext;
import com.yahoo.security.tls.MixedMode;
import com.yahoo.security.tls.TransportSecurityUtils;
import com.yahoo.vespa.config.ConfigKey;
import com.yahoo.vespa.config.protocol.JRTServerConfigRequestV3;
import com.yahoo.vespa.config.server.RequestHandler;
import com.yahoo.vespa.config.server.host.HostRegistry;
import com.yahoo.vespa.config.server.rpc.RequestHandlerProvider;
import com.yahoo.vespa.config.server.rpc.security.AuthorizationException;
import com.yahoo.vespa.config.server.rpc.security.GlobalConfigAuthorizationPolicy;
import com.yahoo.vespa.config.server.rpc.security.RpcAuthorizer;
import com.yahoo.yolean.Exceptions;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.function.BiConsumer;
import java.util.logging.Level;
import java.util.logging.Logger;

public class MultiTenantRpcAuthorizer
implements RpcAuthorizer {
    private static final Logger log = Logger.getLogger(MultiTenantRpcAuthorizer.class.getName());
    private final NodeIdentifier nodeIdentifier;
    private final HostRegistry hostRegistry;
    private final RequestHandlerProvider handlerProvider;
    private final Executor executor;

    public MultiTenantRpcAuthorizer(NodeIdentifier nodeIdentifier, HostRegistry hostRegistry, RequestHandlerProvider handlerProvider, int threadPoolSize) {
        this(nodeIdentifier, hostRegistry, handlerProvider, Executors.newFixedThreadPool(threadPoolSize, (ThreadFactory)new DaemonThreadFactory("multi-tenant-rpc-authorizer-")));
    }

    MultiTenantRpcAuthorizer(NodeIdentifier nodeIdentifier, HostRegistry hostRegistry, RequestHandlerProvider handlerProvider, Executor executor) {
        this.nodeIdentifier = nodeIdentifier;
        this.hostRegistry = hostRegistry;
        this.handlerProvider = handlerProvider;
        this.executor = executor;
    }

    @Override
    public CompletableFuture<Void> authorizeConfigRequest(Request request) {
        return this.doAsyncAuthorization(request, this::doConfigRequestAuthorization);
    }

    @Override
    public CompletableFuture<Void> authorizeFileRequest(Request request) {
        return this.doAsyncAuthorization(request, this::doFileRequestAuthorization);
    }

    private CompletableFuture<Void> doAsyncAuthorization(Request request, BiConsumer<Request, NodeIdentity> authorizer) {
        return CompletableFuture.runAsync(() -> {
            try {
                this.getPeerIdentity(request).ifPresent(peerIdentity -> authorizer.accept(request, (NodeIdentity)peerIdentity));
                log.log(Level.FINE, () -> String.format("Authorization succeeded for request '%s' from '%s'", request.methodName(), request.target().toString()));
            }
            catch (Throwable t) {
                this.handleAuthorizationFailure(request, t);
            }
        }, this.executor);
    }

    private void doConfigRequestAuthorization(Request request, NodeIdentity peerIdentity) {
        switch (peerIdentity.nodeType()) {
            case config: {
                return;
            }
            case proxy: 
            case tenant: 
            case host: {
                JRTServerConfigRequestV3 configRequest = JRTServerConfigRequestV3.createFromRequest((Request)request);
                ConfigKey configKey = configRequest.getConfigKey();
                if (MultiTenantRpcAuthorizer.isConfigKeyForGlobalConfig(configKey)) {
                    GlobalConfigAuthorizationPolicy.verifyAccessAllowed(configKey, peerIdentity.nodeType());
                    return;
                }
                String hostname = configRequest.getClientHostName();
                ApplicationId applicationId = this.hostRegistry.getApplicationId(hostname);
                if (applicationId == null) {
                    if (MultiTenantRpcAuthorizer.isConfigKeyForSentinelConfig(configKey)) {
                        return;
                    }
                    throw new AuthorizationException(AuthorizationException.Type.SILENT, String.format("Host '%s' not found in host registry for [%s]", hostname, configKey));
                }
                RequestHandler tenantHandler = this.getTenantHandler(applicationId.tenant());
                ApplicationId resolvedApplication = tenantHandler.resolveApplicationId(hostname);
                ApplicationId peerOwner = MultiTenantRpcAuthorizer.applicationId(peerIdentity);
                if (peerOwner.equals((Object)resolvedApplication)) {
                    return;
                }
                throw new AuthorizationException(String.format("Peer is not allowed to access config owned by %s. Peer is owned by %s", resolvedApplication.toShortString(), peerOwner.toShortString()));
            }
        }
        throw new AuthorizationException(String.format("'%s' nodes are not allowed to access config", peerIdentity.nodeType()));
    }

    private void doFileRequestAuthorization(Request request, NodeIdentity peerIdentity) {
        switch (peerIdentity.nodeType()) {
            case config: {
                return;
            }
            case proxy: 
            case tenant: 
            case host: {
                ApplicationId peerOwner = MultiTenantRpcAuthorizer.applicationId(peerIdentity);
                FileReference requestedFile = new FileReference(request.parameters().get(0).asString());
                RequestHandler tenantHandler = this.getTenantHandler(peerOwner.tenant());
                Set<FileReference> filesOwnedByApplication = tenantHandler.listFileReferences(peerOwner);
                if (filesOwnedByApplication.contains(requestedFile)) {
                    return;
                }
                throw new AuthorizationException(String.format("Peer is not allowed to access file reference %s. Peer is owned by %s. File references owned by this application: %s", requestedFile.value(), peerOwner.toShortString(), filesOwnedByApplication));
            }
        }
        throw new AuthorizationException(String.format("'%s' nodes are not allowed to access files", peerIdentity.nodeType()));
    }

    private void handleAuthorizationFailure(Request request, Throwable throwable) {
        boolean isAuthorizationException = throwable instanceof AuthorizationException;
        String errorMessage = String.format("For request '%s' from '%s': %s", request.methodName(), request.target().toString(), throwable.getMessage());
        if (!isAuthorizationException || ((AuthorizationException)throwable).type() != AuthorizationException.Type.SILENT) {
            log.log(Level.INFO, errorMessage);
        }
        log.log(Level.FINE, throwable, throwable::getMessage);
        JrtErrorCode error = isAuthorizationException ? JrtErrorCode.UNAUTHORIZED : JrtErrorCode.AUTHORIZATION_FAILED;
        request.setError(error.code, errorMessage);
        request.returnRequest();
        throw Exceptions.throwUnchecked((Throwable)throwable);
    }

    private Optional<NodeIdentity> getPeerIdentity(Request request) {
        ConnectionAuthContext authCtx = request.target().connectionAuthContext();
        if (authCtx.peerCertificate().isEmpty()) {
            if (TransportSecurityUtils.getInsecureMixedMode() == MixedMode.DISABLED) {
                throw new IllegalStateException("Security context missing");
            }
            return Optional.empty();
        }
        List certChain = authCtx.peerCertificateChain();
        if (certChain.isEmpty()) {
            throw new IllegalStateException("Client authentication is not enforced!");
        }
        try {
            NodeIdentity identity = this.nodeIdentifier.identifyNode(certChain);
            log.log(Level.FINE, () -> String.format("Client '%s' identified as %s", request.target().toString(), identity.toString()));
            return Optional.of(identity);
        }
        catch (NodeIdentifierException e) {
            throw new AuthorizationException("Failed to identify peer: " + e.getMessage(), e);
        }
    }

    private static boolean isConfigKeyForGlobalConfig(ConfigKey<?> configKey) {
        return "*".equals(configKey.getConfigId());
    }

    private static boolean isConfigKeyForSentinelConfig(ConfigKey<?> configKey) {
        return SentinelConfig.getDefName().equals(configKey.getName()) && SentinelConfig.getDefNamespace().equals(configKey.getNamespace());
    }

    private static ApplicationId applicationId(NodeIdentity peerIdentity) {
        return (ApplicationId)peerIdentity.applicationId().orElseThrow(() -> new AuthorizationException("Peer node is not associated with an application: " + peerIdentity.toString()));
    }

    private RequestHandler getTenantHandler(TenantName tenantName) {
        return this.handlerProvider.getRequestHandler(tenantName).orElseThrow(() -> new AuthorizationException(String.format("No handler exists for tenant '%s'", tenantName.value())));
    }

    private static enum JrtErrorCode {
        UNAUTHORIZED(1),
        AUTHORIZATION_FAILED(2);

        final int code;

        private JrtErrorCode(int errorOffset) {
            this.code = 131072 + errorOffset;
        }
    }
}

