/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.session;

import com.google.common.collect.HashMultiset;
import com.yahoo.concurrent.ThreadFactoryFactory;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.log.LogLevel;
import com.yahoo.path.Path;
import com.yahoo.vespa.config.server.ReloadHandler;
import com.yahoo.vespa.config.server.application.TenantApplications;
import com.yahoo.vespa.config.server.monitoring.MetricUpdater;
import com.yahoo.vespa.config.server.session.RemoteSession;
import com.yahoo.vespa.config.server.session.RemoteSessionFactory;
import com.yahoo.vespa.config.server.session.RemoteSessionStateWatcher;
import com.yahoo.vespa.config.server.session.Session;
import com.yahoo.vespa.config.server.session.SessionRepo;
import com.yahoo.vespa.config.server.tenant.TenantRepository;
import com.yahoo.vespa.curator.Curator;
import com.yahoo.vespa.flags.BooleanFlag;
import com.yahoo.vespa.flags.FlagSource;
import com.yahoo.vespa.flags.Flags;
import com.yahoo.vespa.flags.InMemoryFlagSource;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;

public class RemoteSessionRepo
extends SessionRepo<RemoteSession> {
    private static final Logger log = Logger.getLogger(RemoteSessionRepo.class.getName());
    private static final ExecutorService pathChildrenExecutor = Executors.newCachedThreadPool(ThreadFactoryFactory.getDaemonThreadFactory((String)RemoteSessionRepo.class.getName()));
    private final Curator curator;
    private final Path sessionsPath;
    private final RemoteSessionFactory remoteSessionFactory;
    private final Map<Long, RemoteSessionStateWatcher> sessionStateWatchers = new HashMap<Long, RemoteSessionStateWatcher>();
    private final ReloadHandler reloadHandler;
    private final TenantName tenantName;
    private final MetricUpdater metrics;
    private final FlagSource flagSource;
    private final Curator.DirectoryCache directoryCache;
    private final TenantApplications applicationRepo;

    public RemoteSessionRepo(Curator curator, RemoteSessionFactory remoteSessionFactory, ReloadHandler reloadHandler, TenantName tenantName, TenantApplications applicationRepo, MetricUpdater metricUpdater, FlagSource flagSource) {
        this.curator = curator;
        this.sessionsPath = TenantRepository.getSessionsPath(tenantName);
        this.applicationRepo = applicationRepo;
        this.remoteSessionFactory = remoteSessionFactory;
        this.reloadHandler = reloadHandler;
        this.tenantName = tenantName;
        this.metrics = metricUpdater;
        this.flagSource = flagSource;
        this.initializeSessions();
        this.directoryCache = curator.createDirectoryCache(this.sessionsPath.getAbsolute(), false, false, pathChildrenExecutor);
        this.directoryCache.addListener(this::childEvent);
        this.directoryCache.start();
    }

    public RemoteSessionRepo(TenantName tenantName) {
        this.curator = null;
        this.remoteSessionFactory = null;
        this.reloadHandler = null;
        this.tenantName = tenantName;
        this.sessionsPath = TenantRepository.getSessionsPath(tenantName);
        this.metrics = null;
        this.directoryCache = null;
        this.applicationRepo = null;
        this.flagSource = new InMemoryFlagSource();
    }

    public List<Long> getSessions() {
        return this.getSessionList(this.curator.getChildren(this.sessionsPath));
    }

    public int deleteExpiredSessions(Duration expiryTime) {
        int deleted = 0;
        for (long sessionId : this.getSessions()) {
            RemoteSession session = (RemoteSession)this.getSession(sessionId);
            Instant created = Instant.ofEpochSecond(session.getCreateTime());
            if (!this.sessionHasExpired(created, expiryTime)) continue;
            log.log(LogLevel.INFO, "Remote session " + sessionId + " for " + this.tenantName + " has expired, deleting it");
            session.delete();
            ++deleted;
        }
        return deleted;
    }

    private boolean sessionHasExpired(Instant created, Duration expiryTime) {
        return created.plus(expiryTime).isBefore(Instant.now());
    }

    private List<Long> getSessionListFromDirectoryCache(List<ChildData> children) {
        return this.getSessionList(children.stream().map(child -> Path.fromString((String)child.getPath()).getName()).collect(Collectors.toList()));
    }

    private List<Long> getSessionList(List<String> children) {
        return children.stream().map(Long::parseLong).collect(Collectors.toList());
    }

    private void initializeSessions() throws NumberFormatException {
        this.getSessions().forEach(this::sessionAdded);
    }

    private synchronized void sessionsChanged() throws NumberFormatException {
        List<Long> sessions = this.getSessionListFromDirectoryCache(this.directoryCache.getCurrentData());
        this.checkForRemovedSessions(sessions);
        this.checkForAddedSessions(sessions);
    }

    private void checkForRemovedSessions(List<Long> sessions) {
        for (RemoteSession session : this.listSessions()) {
            if (sessions.contains(session.getSessionId())) continue;
            this.sessionRemoved(session.getSessionId());
        }
    }

    private void checkForAddedSessions(List<Long> sessions) {
        for (Long sessionId : sessions) {
            if (this.getSession(sessionId) != null) continue;
            this.sessionAdded(sessionId);
        }
    }

    private void sessionAdded(long sessionId) {
        try {
            log.log((Level)LogLevel.DEBUG, () -> "Adding session to RemoteSessionRepo: " + sessionId);
            RemoteSession session = this.remoteSessionFactory.createSession(sessionId);
            Path sessionPath = this.sessionsPath.append(String.valueOf(sessionId));
            Curator.FileCache fileCache = this.curator.createFileCache(sessionPath.append("/sessionState").getAbsolute(), false);
            fileCache.addListener(this::nodeChanged);
            this.loadSessionIfActive(session);
            this.sessionStateWatchers.put(sessionId, new RemoteSessionStateWatcher(fileCache, this.reloadHandler, session, this.metrics));
            this.addSession(session);
            this.metrics.incAddedSessions();
        }
        catch (Exception e) {
            if (((BooleanFlag)Flags.CONFIG_SERVER_FAIL_IF_ACTIVE_SESSION_CANNOT_BE_LOADED.bindTo(this.flagSource)).value()) {
                throw e;
            }
            log.log(Level.WARNING, "Failed loading session " + sessionId + ": No config for this session can be served", e);
        }
    }

    private void sessionRemoved(long sessionId) {
        RemoteSessionStateWatcher watcher = this.sessionStateWatchers.remove(sessionId);
        if (watcher != null) {
            watcher.close();
        }
        this.removeSession(sessionId);
        this.metrics.incRemovedSessions();
    }

    private void loadSessionIfActive(RemoteSession session) {
        for (ApplicationId applicationId : this.applicationRepo.activeApplications()) {
            if (this.applicationRepo.requireActiveSessionOf(applicationId) != session.getSessionId()) continue;
            log.log((Level)LogLevel.DEBUG, () -> "Found active application for session " + session.getSessionId() + " , loading it");
            this.reloadHandler.reloadConfig(session.ensureApplicationLoaded());
            log.log(LogLevel.INFO, session.logPre() + "Application activated successfully: " + applicationId);
            return;
        }
    }

    public synchronized void close() {
        try {
            if (this.directoryCache != null) {
                this.directoryCache.close();
            }
        }
        catch (Exception e) {
            log.log(LogLevel.WARNING, "Exception when closing path cache", e);
        }
        finally {
            this.checkForRemovedSessions(new ArrayList<Long>());
        }
    }

    private void nodeChanged() {
        HashMultiset sessionMetrics = HashMultiset.create();
        for (RemoteSession session : this.listSessions()) {
            sessionMetrics.add((Object)session.getStatus());
        }
        this.metrics.setNewSessions(sessionMetrics.count((Object)Session.Status.NEW));
        this.metrics.setPreparedSessions(sessionMetrics.count((Object)Session.Status.PREPARE));
        this.metrics.setActivatedSessions(sessionMetrics.count((Object)Session.Status.ACTIVATE));
        this.metrics.setDeactivatedSessions(sessionMetrics.count((Object)Session.Status.DEACTIVATE));
    }

    private void childEvent(CuratorFramework ignored, PathChildrenCacheEvent event) {
        log.log((Level)LogLevel.DEBUG, () -> "Got child event: " + event);
        switch (event.getType()) {
            case CHILD_ADDED: {
                this.sessionsChanged();
                this.synchronizeOnNew(this.getSessionListFromDirectoryCache(Collections.singletonList(event.getData())));
                break;
            }
            case CHILD_REMOVED: {
                this.sessionsChanged();
                break;
            }
            case CONNECTION_RECONNECTED: {
                this.sessionsChanged();
            }
        }
    }

    private void synchronizeOnNew(List<Long> sessionList) {
        for (long sessionId : sessionList) {
            RemoteSession session = (RemoteSession)this.getSession(sessionId);
            if (session == null) continue;
            log.log((Level)LogLevel.DEBUG, () -> session.logPre() + "Confirming upload for session " + sessionId);
            session.confirmUpload();
        }
    }
}

