/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server;

import com.google.inject.Inject;
import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.component.AbstractComponent;
import com.yahoo.concurrent.DaemonThreadFactory;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.Deployment;
import com.yahoo.config.provision.TransientException;
import com.yahoo.container.handler.VipStatus;
import com.yahoo.container.jdisc.state.StateMonitor;
import com.yahoo.vespa.config.server.ApplicationRepository;
import com.yahoo.vespa.config.server.rpc.RpcServer;
import com.yahoo.vespa.config.server.version.VersionState;
import com.yahoo.yolean.Exceptions;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ConfigServerBootstrap
extends AbstractComponent
implements Runnable {
    private static final Logger log = Logger.getLogger(ConfigServerBootstrap.class.getName());
    private final ApplicationRepository applicationRepository;
    private final RpcServer server;
    private final VersionState versionState;
    private final StateMonitor stateMonitor;
    private final VipStatus vipStatus;
    private final ConfigserverConfig configserverConfig;
    private final Duration maxDurationOfRedeployment;
    private final Duration sleepTimeWhenRedeployingFails;
    private final RedeployingApplicationsFails exitIfRedeployingApplicationsFails;
    private final ExecutorService rpcServerExecutor;

    @Inject
    public ConfigServerBootstrap(ApplicationRepository applicationRepository, RpcServer server, VersionState versionState, StateMonitor stateMonitor, VipStatus vipStatus) {
        this(applicationRepository, server, versionState, stateMonitor, vipStatus, Mode.BOOTSTRAP_IN_CONSTRUCTOR, RedeployingApplicationsFails.EXIT_JVM, applicationRepository.configserverConfig().hostedVespa() ? VipStatusMode.VIP_STATUS_FILE : VipStatusMode.VIP_STATUS_PROGRAMMATICALLY);
    }

    ConfigServerBootstrap(ApplicationRepository applicationRepository, RpcServer server, VersionState versionState, StateMonitor stateMonitor, VipStatus vipStatus, VipStatusMode vipStatusMode) {
        this(applicationRepository, server, versionState, stateMonitor, vipStatus, Mode.FOR_TESTING_NO_BOOTSTRAP_OF_APPS, RedeployingApplicationsFails.CONTINUE, vipStatusMode);
    }

    private ConfigServerBootstrap(ApplicationRepository applicationRepository, RpcServer server, VersionState versionState, StateMonitor stateMonitor, VipStatus vipStatus, Mode mode, RedeployingApplicationsFails exitIfRedeployingApplicationsFails, VipStatusMode vipStatusMode) {
        this.applicationRepository = applicationRepository;
        this.server = server;
        this.versionState = versionState;
        this.stateMonitor = stateMonitor;
        this.vipStatus = vipStatus;
        this.configserverConfig = applicationRepository.configserverConfig();
        this.maxDurationOfRedeployment = Duration.ofSeconds(this.configserverConfig.maxDurationOfBootstrap());
        this.sleepTimeWhenRedeployingFails = Duration.ofSeconds(this.configserverConfig.sleepTimeWhenRedeployingFails());
        this.exitIfRedeployingApplicationsFails = exitIfRedeployingApplicationsFails;
        this.rpcServerExecutor = Executors.newSingleThreadExecutor((ThreadFactory)new DaemonThreadFactory("config server RPC server"));
        log.log(Level.FINE, () -> "Bootstrap mode: " + mode + ", VIP status mode: " + vipStatusMode);
        this.initializing(vipStatusMode);
        switch (mode) {
            case BOOTSTRAP_IN_CONSTRUCTOR: {
                this.start();
                break;
            }
            case FOR_TESTING_NO_BOOTSTRAP_OF_APPS: {
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown bootstrap mode " + mode + ", legal values: " + Arrays.toString((Object[])Mode.values()));
            }
        }
    }

    public void deconstruct() {
        log.log(Level.INFO, "Stopping config server");
        this.down();
        this.server.stop();
        log.log(Level.FINE, "RPC server stopped");
        this.rpcServerExecutor.shutdown();
    }

    @Override
    public void run() {
        this.start();
        do {
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException e) {
                log.log(Level.SEVERE, "Got interrupted", e);
                break;
            }
        } while (this.server.isRunning());
        this.down();
    }

    public void start() {
        this.startRpcServerWithFileDistribution();
        if (this.versionState.isUpgraded()) {
            log.log(Level.INFO, "Config server upgrading from " + this.versionState.storedVersion() + " to " + this.versionState.currentVersion() + ". Redeploying all applications");
            try {
                if (!this.redeployAllApplications()) {
                    this.redeployingApplicationsFailed();
                    return;
                }
                this.versionState.saveNewVersion();
                log.log(Level.INFO, "All applications redeployed successfully");
            }
            catch (Exception e) {
                log.log(Level.SEVERE, "Redeployment of applications failed", e);
                this.redeployingApplicationsFailed();
                return;
            }
        }
        this.applicationRepository.bootstrappingDone();
        this.allowConfigRpcRequests(this.server);
        this.up();
    }

    StateMonitor.Status status() {
        return this.stateMonitor.status();
    }

    private void up() {
        this.vipStatus.setInRotation(Boolean.valueOf(true));
    }

    private void down() {
        this.vipStatus.setInRotation(Boolean.valueOf(false));
    }

    private void initializing(VipStatusMode vipStatusMode) {
        this.stateMonitor.status(StateMonitor.Status.initializing);
        if (vipStatusMode == VipStatusMode.VIP_STATUS_PROGRAMMATICALLY) {
            this.vipStatus.setInRotation(Boolean.valueOf(false));
        }
    }

    private void startRpcServerWithFileDistribution() {
        this.rpcServerExecutor.execute(this.server);
        Instant end = Instant.now().plus(Duration.ofSeconds(10L));
        while (!this.server.isRunning() && Instant.now().isBefore(end)) {
            try {
                Thread.sleep(10L);
            }
            catch (InterruptedException e) {
                log.log(Level.SEVERE, "Got interrupted", e);
                break;
            }
        }
        if (!this.server.isRunning()) {
            throw new RuntimeException("RPC server not started in 10 seconds");
        }
    }

    private void allowConfigRpcRequests(RpcServer rpcServer) {
        log.log(Level.INFO, "Allowing RPC config requests");
        rpcServer.setUpGetConfigHandlers();
    }

    private void redeployingApplicationsFailed() {
        if (this.exitIfRedeployingApplicationsFails == RedeployingApplicationsFails.EXIT_JVM) {
            System.exit(1);
        }
    }

    private boolean redeployAllApplications() throws InterruptedException {
        Instant end = Instant.now().plus(this.maxDurationOfRedeployment);
        List<ApplicationId> applicationsToRedeploy = this.applicationRepository.listApplications();
        Collections.shuffle(applicationsToRedeploy);
        long failCount = 0L;
        do {
            Duration sleepTime;
            if ((applicationsToRedeploy = this.redeployApplications(applicationsToRedeploy)).isEmpty() || this.sleepTimeWhenRedeployingFails.isZero()) continue;
            if ((sleepTime = this.sleepTimeWhenRedeployingFails.multipliedBy(++failCount)).compareTo(Duration.ofMinutes(10L)) > 0) {
                sleepTime = Duration.ofMinutes(10L);
            }
            log.log(Level.INFO, "Redeployment of " + applicationsToRedeploy + " not finished, will retry in " + sleepTime);
            Thread.sleep(sleepTime.toMillis());
        } while (!applicationsToRedeploy.isEmpty() && Instant.now().isBefore(end));
        if (!applicationsToRedeploy.isEmpty()) {
            log.log(Level.SEVERE, "Redeploying applications not finished after " + this.maxDurationOfRedeployment + ", exiting, applications that failed redeployment: " + applicationsToRedeploy);
            return false;
        }
        return true;
    }

    private List<ApplicationId> redeployApplications(List<ApplicationId> applicationIds) throws InterruptedException {
        ExecutorService executor = Executors.newFixedThreadPool(this.configserverConfig.numRedeploymentThreads(), (ThreadFactory)new DaemonThreadFactory("redeploy-apps-"));
        HashMap deployments = new HashMap();
        log.log(Level.INFO, () -> "Redeploying " + applicationIds.size() + " apps: " + applicationIds);
        applicationIds.forEach(appId -> deployments.put((ApplicationId)appId, executor.submit(() -> {
            log.log(Level.INFO, () -> "Starting redeployment of " + appId);
            this.applicationRepository.deployFromLocalActive((ApplicationId)appId, true).ifPresent(Deployment::activate);
            log.log(Level.INFO, () -> appId + " redeployed");
        })));
        List<ApplicationId> failedDeployments = this.checkDeployments(deployments);
        executor.shutdown();
        executor.awaitTermination(365L, TimeUnit.DAYS);
        return failedDeployments;
    }

    private List<ApplicationId> checkDeployments(Map<ApplicationId, Future<?>> deployments) {
        int applicationCount = deployments.size();
        LinkedHashSet<ApplicationId> failedDeployments = new LinkedHashSet<ApplicationId>();
        LinkedHashSet<ApplicationId> finishedDeployments = new LinkedHashSet<ApplicationId>();
        Instant lastLogged = Instant.EPOCH;
        do {
            deployments.forEach((applicationId, future) -> {
                if (finishedDeployments.contains(applicationId) || failedDeployments.contains(applicationId)) {
                    return;
                }
                DeploymentStatus status = this.getDeploymentStatus((ApplicationId)applicationId, (Future<?>)future);
                switch (status) {
                    case done: {
                        finishedDeployments.add((ApplicationId)applicationId);
                        break;
                    }
                    case inProgress: {
                        break;
                    }
                    case failed: {
                        failedDeployments.add((ApplicationId)applicationId);
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("Unknown deployment status " + status);
                    }
                }
            });
            if (Duration.between(lastLogged, Instant.now()).minus(Duration.ofSeconds(10L)).isNegative()) continue;
            this.logProgress(applicationCount, failedDeployments, finishedDeployments);
            lastLogged = Instant.now();
        } while (failedDeployments.size() + finishedDeployments.size() < applicationCount);
        this.logProgress(applicationCount, failedDeployments, finishedDeployments);
        return new ArrayList<ApplicationId>(failedDeployments);
    }

    private void logProgress(int applicationCount, Set<ApplicationId> failedDeployments, Set<ApplicationId> finishedDeployments) {
        log.log(Level.INFO, () -> finishedDeployments.size() + " of " + applicationCount + " apps redeployed (" + failedDeployments.size() + " failed)");
    }

    private DeploymentStatus getDeploymentStatus(ApplicationId applicationId, Future<?> future) {
        try {
            future.get(1L, TimeUnit.MILLISECONDS);
            return DeploymentStatus.done;
        }
        catch (InterruptedException | ExecutionException e) {
            if (e.getCause() instanceof TransientException) {
                log.log(Level.INFO, "Redeploying " + applicationId + " failed with transient error, will retry after bootstrap: " + Exceptions.toMessageString((Throwable)e));
            } else {
                log.log(Level.WARNING, "Redeploying " + applicationId + " failed, will retry", e);
            }
            return DeploymentStatus.failed;
        }
        catch (TimeoutException e) {
            return DeploymentStatus.inProgress;
        }
    }

    private static enum DeploymentStatus {
        inProgress,
        done,
        failed;

    }

    static enum VipStatusMode {
        VIP_STATUS_FILE,
        VIP_STATUS_PROGRAMMATICALLY;

    }

    static enum RedeployingApplicationsFails {
        EXIT_JVM,
        CONTINUE;

    }

    static enum Mode {
        BOOTSTRAP_IN_CONSTRUCTOR,
        FOR_TESTING_NO_BOOTSTRAP_OF_APPS;

    }
}

