/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.session;

import com.google.common.collect.HashMultiset;
import com.yahoo.config.FileReference;
import com.yahoo.config.application.api.ApplicationPackage;
import com.yahoo.config.application.api.DeployLogger;
import com.yahoo.config.model.application.provider.DeployData;
import com.yahoo.config.model.application.provider.FilesApplicationPackage;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.io.IOUtils;
import com.yahoo.path.Path;
import com.yahoo.transaction.AbstractTransaction;
import com.yahoo.transaction.NestedTransaction;
import com.yahoo.transaction.Transaction;
import com.yahoo.vespa.config.server.GlobalComponentRegistry;
import com.yahoo.vespa.config.server.ReloadHandler;
import com.yahoo.vespa.config.server.TimeoutBudget;
import com.yahoo.vespa.config.server.application.ApplicationSet;
import com.yahoo.vespa.config.server.application.TenantApplications;
import com.yahoo.vespa.config.server.configchange.ConfigChangeActions;
import com.yahoo.vespa.config.server.deploy.TenantFileSystemDirs;
import com.yahoo.vespa.config.server.filedistribution.FileDirectory;
import com.yahoo.vespa.config.server.monitoring.MetricUpdater;
import com.yahoo.vespa.config.server.monitoring.Metrics;
import com.yahoo.vespa.config.server.session.LocalSession;
import com.yahoo.vespa.config.server.session.LocalSessionStateWatcher;
import com.yahoo.vespa.config.server.session.PrepareParams;
import com.yahoo.vespa.config.server.session.RemoteSession;
import com.yahoo.vespa.config.server.session.RemoteSessionStateWatcher;
import com.yahoo.vespa.config.server.session.Session;
import com.yahoo.vespa.config.server.session.SessionCache;
import com.yahoo.vespa.config.server.session.SessionPreparer;
import com.yahoo.vespa.config.server.session.SessionZooKeeperClient;
import com.yahoo.vespa.config.server.tenant.TenantRepository;
import com.yahoo.vespa.config.server.zookeeper.SessionCounter;
import com.yahoo.vespa.curator.Curator;
import com.yahoo.vespa.defaults.Defaults;
import com.yahoo.vespa.flags.BooleanFlag;
import com.yahoo.vespa.flags.FlagSource;
import com.yahoo.vespa.flags.Flags;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;

public class SessionRepository {
    private static final Logger log = Logger.getLogger(SessionRepository.class.getName());
    private static final FilenameFilter sessionApplicationsFilter = (dir, name) -> name.matches("\\d+");
    private static final long nonExistingActiveSession = 0L;
    private final SessionCache<LocalSession> localSessionCache = new SessionCache();
    private final SessionCache<RemoteSession> remoteSessionCache = new SessionCache();
    private final Map<Long, LocalSessionStateWatcher> localSessionStateWatchers = new HashMap<Long, LocalSessionStateWatcher>();
    private final Map<Long, RemoteSessionStateWatcher> remoteSessionStateWatchers = new HashMap<Long, RemoteSessionStateWatcher>();
    private final Duration sessionLifetime;
    private final Clock clock;
    private final Curator curator;
    private final Executor zkWatcherExecutor;
    private final TenantFileSystemDirs tenantFileSystemDirs;
    private final BooleanFlag distributeApplicationPackage;
    private final ReloadHandler reloadHandler;
    private final MetricUpdater metrics;
    private final Curator.DirectoryCache directoryCache;
    private final TenantApplications applicationRepo;
    private final SessionPreparer sessionPreparer;
    private final Path sessionsPath;
    private final TenantName tenantName;
    private final GlobalComponentRegistry componentRegistry;

    public SessionRepository(TenantName tenantName, GlobalComponentRegistry componentRegistry, TenantApplications applicationRepo, ReloadHandler reloadHandler, FlagSource flagSource, SessionPreparer sessionPreparer) {
        this.tenantName = tenantName;
        this.componentRegistry = componentRegistry;
        this.sessionsPath = TenantRepository.getSessionsPath(tenantName);
        this.clock = componentRegistry.getClock();
        this.curator = componentRegistry.getCurator();
        this.sessionLifetime = Duration.ofSeconds(componentRegistry.getConfigserverConfig().sessionLifetime());
        this.zkWatcherExecutor = command -> componentRegistry.getZkWatcherExecutor().execute((Object)tenantName, command);
        this.tenantFileSystemDirs = new TenantFileSystemDirs(componentRegistry.getConfigServerDB(), tenantName);
        this.applicationRepo = applicationRepo;
        this.sessionPreparer = sessionPreparer;
        this.distributeApplicationPackage = (BooleanFlag)Flags.CONFIGSERVER_DISTRIBUTE_APPLICATION_PACKAGE.bindTo(flagSource);
        this.reloadHandler = reloadHandler;
        this.metrics = componentRegistry.getMetrics().getOrCreateMetricUpdater(Metrics.createDimensions(tenantName));
        this.loadLocalSessions();
        this.initializeRemoteSessions();
        this.directoryCache = this.curator.createDirectoryCache(this.sessionsPath.getAbsolute(), false, false, componentRegistry.getZkCacheExecutor());
        this.directoryCache.addListener(this::childEvent);
        this.directoryCache.start();
    }

    public synchronized void addSession(LocalSession session) {
        this.localSessionCache.addSession(session);
        Path sessionsPath = TenantRepository.getSessionsPath(session.getTenantName());
        long sessionId = session.getSessionId();
        Curator.FileCache fileCache = this.curator.createFileCache(sessionsPath.append(String.valueOf(sessionId)).append("/sessionState").getAbsolute(), false);
        this.localSessionStateWatchers.put(sessionId, new LocalSessionStateWatcher(fileCache, session, this, this.zkWatcherExecutor));
    }

    public LocalSession getLocalSession(long sessionId) {
        return this.localSessionCache.getSession(sessionId);
    }

    public List<LocalSession> getLocalSessions() {
        return this.localSessionCache.getSessions();
    }

    private void loadLocalSessions() {
        File[] sessions = this.tenantFileSystemDirs.sessionsPath().listFiles(sessionApplicationsFilter);
        if (sessions == null) {
            return;
        }
        for (File session : sessions) {
            try {
                this.addSession(this.createSessionFromId(Long.parseLong(session.getName())));
            }
            catch (IllegalArgumentException e) {
                log.log(Level.WARNING, "Could not load session '" + session.getAbsolutePath() + "':" + e.getMessage() + ", skipping it.");
            }
        }
    }

    public ConfigChangeActions prepareLocalSession(LocalSession session, DeployLogger logger, PrepareParams params, Optional<ApplicationSet> currentActiveApplicationSet, Path tenantPath, Instant now) {
        this.applicationRepo.createApplication(params.getApplicationId());
        logger.log(Level.FINE, "Created application " + params.getApplicationId());
        long sessionId = session.getSessionId();
        SessionZooKeeperClient sessionZooKeeperClient = this.createSessionZooKeeperClient(sessionId);
        Curator.CompletionWaiter waiter = sessionZooKeeperClient.createPrepareWaiter();
        ConfigChangeActions actions = this.sessionPreparer.prepare(this.applicationRepo.getHostValidator(), logger, params, currentActiveApplicationSet, tenantPath, now, this.getSessionAppDir(sessionId), session.getApplicationPackage(), sessionZooKeeperClient);
        session.setPrepared();
        waiter.awaitCompletion(params.getTimeoutBudget().timeLeft());
        return actions;
    }

    public void deleteExpiredSessions(Map<ApplicationId, Long> activeSessions) {
        log.log(Level.FINE, "Purging old sessions");
        try {
            for (LocalSession candidate : this.localSessionCache.getSessions()) {
                ApplicationId applicationId;
                Long activeSession;
                Instant createTime = candidate.getCreateTime();
                log.log(Level.FINE, "Candidate session for deletion: " + candidate.getSessionId() + ", created: " + createTime);
                if (this.hasExpired(candidate) && !this.isActiveSession(candidate)) {
                    this.deleteLocalSession(candidate);
                    continue;
                }
                if (!createTime.plus(Duration.ofDays(1L)).isBefore(this.clock.instant()) || (activeSession = activeSessions.get(applicationId = candidate.getApplicationId())) != null && activeSession.longValue() == candidate.getSessionId()) continue;
                this.deleteLocalSession(candidate);
                log.log(Level.INFO, "Deleted inactive session " + candidate.getSessionId() + " created " + createTime + " for '" + applicationId + "'");
            }
        }
        catch (Throwable e) {
            log.log(Level.WARNING, "Error when purging old sessions ", e);
        }
        log.log(Level.FINE, "Done purging old sessions");
    }

    private boolean hasExpired(LocalSession candidate) {
        return candidate.getCreateTime().plus(this.sessionLifetime).isBefore(this.clock.instant());
    }

    private boolean isActiveSession(LocalSession candidate) {
        return candidate.getStatus() == Session.Status.ACTIVATE;
    }

    public void deleteLocalSession(LocalSession session) {
        long sessionId = session.getSessionId();
        log.log(Level.FINE, "Deleting local session " + sessionId);
        LocalSessionStateWatcher watcher = this.localSessionStateWatchers.remove(sessionId);
        if (watcher != null) {
            watcher.close();
        }
        this.localSessionCache.removeSession(sessionId);
        NestedTransaction transaction = new NestedTransaction();
        this.deleteLocalSession(session, transaction);
        transaction.commit();
    }

    public void deleteLocalSession(LocalSession session, NestedTransaction transaction) {
        long sessionId = session.getSessionId();
        SessionZooKeeperClient sessionZooKeeperClient = this.createSessionZooKeeperClient(sessionId);
        transaction.add((Transaction)sessionZooKeeperClient.deleteTransaction(), new Class[]{FileTransaction.class});
        transaction.add((Transaction)FileTransaction.from(FileOperations.delete(this.getSessionAppDir(sessionId).getAbsolutePath())), new Class[0]);
    }

    public void close() {
        this.deleteAllSessions();
        this.tenantFileSystemDirs.delete();
        try {
            if (this.directoryCache != null) {
                this.directoryCache.close();
            }
        }
        catch (Exception e) {
            log.log(Level.WARNING, "Exception when closing path cache", e);
        }
        finally {
            this.checkForRemovedSessions(new ArrayList<Long>());
        }
    }

    private void deleteAllSessions() {
        ArrayList<LocalSession> sessions = new ArrayList<LocalSession>(this.localSessionCache.getSessions());
        for (LocalSession session : sessions) {
            this.deleteLocalSession(session);
        }
    }

    public RemoteSession getRemoteSession(long sessionId) {
        return this.remoteSessionCache.getSession(sessionId);
    }

    public List<Long> getRemoteSessions() {
        return this.getSessionList(this.curator.getChildren(this.sessionsPath));
    }

    public void addRemoteSession(RemoteSession session) {
        this.remoteSessionCache.addSession(session);
        this.metrics.incAddedSessions();
    }

    public int deleteExpiredRemoteSessions(Clock clock, Duration expiryTime) {
        int deleted = 0;
        for (long sessionId : this.getRemoteSessions()) {
            RemoteSession session = this.remoteSessionCache.getSession(sessionId);
            if (session == null || session.getStatus() == Session.Status.ACTIVATE || !this.sessionHasExpired(session.getCreateTime(), expiryTime, clock)) continue;
            log.log(Level.INFO, "Remote session " + sessionId + " for " + this.tenantName + " has expired, deleting it");
            session.delete();
            ++deleted;
        }
        return deleted;
    }

    private boolean sessionHasExpired(Instant created, Duration expiryTime, Clock clock) {
        return created.plus(expiryTime).isBefore(clock.instant());
    }

    private List<Long> getSessionListFromDirectoryCache(List<ChildData> children) {
        return this.getSessionList(children.stream().map(child -> Path.fromString((String)child.getPath()).getName()).collect(Collectors.toList()));
    }

    private List<Long> getSessionList(List<String> children) {
        return children.stream().map(Long::parseLong).collect(Collectors.toList());
    }

    private void initializeRemoteSessions() throws NumberFormatException {
        this.getRemoteSessions().forEach(this::sessionAdded);
    }

    private synchronized void sessionsChanged() throws NumberFormatException {
        List<Long> sessions = this.getSessionListFromDirectoryCache(this.directoryCache.getCurrentData());
        this.checkForRemovedSessions(sessions);
        this.checkForAddedSessions(sessions);
    }

    private void checkForRemovedSessions(List<Long> sessions) {
        for (RemoteSession session : this.remoteSessionCache.getSessions()) {
            if (sessions.contains(session.getSessionId())) continue;
            this.sessionRemoved(session.getSessionId());
        }
    }

    private void checkForAddedSessions(List<Long> sessions) {
        for (Long sessionId : sessions) {
            if (this.remoteSessionCache.getSession(sessionId) != null) continue;
            this.sessionAdded(sessionId);
        }
    }

    public void sessionAdded(long sessionId) {
        log.log(Level.FINE, () -> "Adding session to SessionRepository: " + sessionId);
        RemoteSession session = this.createRemoteSession(sessionId);
        Path sessionPath = this.sessionsPath.append(String.valueOf(sessionId));
        Curator.FileCache fileCache = this.curator.createFileCache(sessionPath.append("/sessionState").getAbsolute(), false);
        fileCache.addListener(this::nodeChanged);
        this.loadSessionIfActive(session);
        this.addRemoteSession(session);
        this.remoteSessionStateWatchers.put(sessionId, new RemoteSessionStateWatcher(fileCache, this.reloadHandler, session, this.metrics, this.zkWatcherExecutor));
        if (this.distributeApplicationPackage.value()) {
            Optional<LocalSession> localSession = this.createLocalSessionUsingDistributedApplicationPackage(sessionId);
            localSession.ifPresent(this::addSession);
        }
    }

    private void sessionRemoved(long sessionId) {
        RemoteSessionStateWatcher watcher = this.remoteSessionStateWatchers.remove(sessionId);
        if (watcher != null) {
            watcher.close();
        }
        this.remoteSessionCache.removeSession(sessionId);
        this.metrics.incRemovedSessions();
    }

    private void loadSessionIfActive(RemoteSession session) {
        for (ApplicationId applicationId : this.applicationRepo.activeApplications()) {
            if (this.applicationRepo.requireActiveSessionOf(applicationId) != session.getSessionId()) continue;
            log.log(Level.FINE, () -> "Found active application for session " + session.getSessionId() + " , loading it");
            this.reloadHandler.reloadConfig(session.ensureApplicationLoaded());
            log.log(Level.INFO, session.logPre() + "Application activated successfully: " + applicationId + " (generation " + session.getSessionId() + ")");
            return;
        }
    }

    private void nodeChanged() {
        this.zkWatcherExecutor.execute(() -> {
            HashMultiset sessionMetrics = HashMultiset.create();
            for (RemoteSession session : this.remoteSessionCache.getSessions()) {
                sessionMetrics.add((Object)session.getStatus());
            }
            this.metrics.setNewSessions(sessionMetrics.count((Object)Session.Status.NEW));
            this.metrics.setPreparedSessions(sessionMetrics.count((Object)Session.Status.PREPARE));
            this.metrics.setActivatedSessions(sessionMetrics.count((Object)Session.Status.ACTIVATE));
            this.metrics.setDeactivatedSessions(sessionMetrics.count((Object)Session.Status.DEACTIVATE));
        });
    }

    private void childEvent(CuratorFramework ignored, PathChildrenCacheEvent event) {
        this.zkWatcherExecutor.execute(() -> {
            log.log(Level.FINE, () -> "Got child event: " + event);
            switch (event.getType()) {
                case CHILD_ADDED: {
                    this.sessionsChanged();
                    this.synchronizeOnNew(this.getSessionListFromDirectoryCache(Collections.singletonList(event.getData())));
                    break;
                }
                case CHILD_REMOVED: 
                case CONNECTION_RECONNECTED: {
                    this.sessionsChanged();
                }
            }
        });
    }

    private void synchronizeOnNew(List<Long> sessionList) {
        for (long sessionId : sessionList) {
            RemoteSession session = this.remoteSessionCache.getSession(sessionId);
            if (session == null) continue;
            log.log(Level.FINE, () -> session.logPre() + "Confirming upload for session " + sessionId);
            session.confirmUpload();
        }
    }

    public LocalSession createSession(File applicationDirectory, ApplicationId applicationId, TimeoutBudget timeoutBudget, Optional<Long> activeSessionId) {
        return this.create(applicationDirectory, applicationId, activeSessionId, false, timeoutBudget);
    }

    public RemoteSession createRemoteSession(long sessionId) {
        SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
        return new RemoteSession(this.tenantName, sessionId, this.componentRegistry, sessionZKClient);
    }

    private void ensureSessionPathDoesNotExist(long sessionId) {
        Path sessionPath = this.getSessionPath(sessionId);
        if (this.componentRegistry.getConfigCurator().exists(sessionPath.getAbsolute())) {
            throw new IllegalArgumentException("Path " + sessionPath.getAbsolute() + " already exists in ZooKeeper");
        }
    }

    private ApplicationPackage createApplication(File userDir, File configApplicationDir, ApplicationId applicationId, long sessionId, Optional<Long> currentlyActiveSessionId, boolean internalRedeploy) {
        long deployTimestamp = System.currentTimeMillis();
        String user = System.getenv("USER");
        if (user == null) {
            user = "unknown";
        }
        DeployData deployData = new DeployData(user, userDir.getAbsolutePath(), applicationId, Long.valueOf(deployTimestamp), internalRedeploy, Long.valueOf(sessionId), currentlyActiveSessionId.orElse(0L).longValue());
        return FilesApplicationPackage.fromFileWithDeployData((File)configApplicationDir, (DeployData)deployData);
    }

    private LocalSession createSessionFromApplication(ApplicationPackage applicationPackage, long sessionId, TimeoutBudget timeoutBudget, Clock clock) {
        log.log(Level.FINE, TenantRepository.logPre(this.tenantName) + "Creating session " + sessionId + " in ZooKeeper");
        SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
        sessionZKClient.createNewSession(clock.instant());
        Curator.CompletionWaiter waiter = sessionZKClient.getUploadWaiter();
        LocalSession session = new LocalSession(this.tenantName, sessionId, applicationPackage, sessionZKClient, this.applicationRepo);
        waiter.awaitCompletion(timeoutBudget.timeLeft());
        return session;
    }

    public LocalSession createSessionFromExisting(Session existingSession, DeployLogger logger, boolean internalRedeploy, TimeoutBudget timeoutBudget) {
        File existingApp = this.getSessionAppDir(existingSession.getSessionId());
        ApplicationId existingApplicationId = existingSession.getApplicationId();
        Optional<Long> activeSessionId = this.getActiveSessionId(existingApplicationId);
        logger.log(Level.FINE, "Create new session for application id '" + existingApplicationId + "' from existing active session " + activeSessionId);
        LocalSession session = this.create(existingApp, existingApplicationId, activeSessionId, internalRedeploy, timeoutBudget);
        session.setApplicationId(existingApplicationId);
        if (this.distributeApplicationPackage.value() && existingSession.getApplicationPackageReference() != null) {
            session.setApplicationPackageReference(existingSession.getApplicationPackageReference());
        }
        session.setVespaVersion(existingSession.getVespaVersion());
        session.setDockerImageRepository(existingSession.getDockerImageRepository());
        session.setAthenzDomain(existingSession.getAthenzDomain());
        return session;
    }

    private LocalSession create(File applicationFile, ApplicationId applicationId, Optional<Long> currentlyActiveSessionId, boolean internalRedeploy, TimeoutBudget timeoutBudget) {
        long sessionId = this.getNextSessionId();
        try {
            this.ensureSessionPathDoesNotExist(sessionId);
            ApplicationPackage app = this.createApplicationPackage(applicationFile, applicationId, sessionId, currentlyActiveSessionId, internalRedeploy);
            return this.createSessionFromApplication(app, sessionId, timeoutBudget, this.clock);
        }
        catch (Exception e) {
            throw new RuntimeException("Error creating session " + sessionId, e);
        }
    }

    private LocalSession createLocalSession(File applicationFile, ApplicationId applicationId, long sessionId, Optional<Long> currentlyActiveSessionId) {
        try {
            ApplicationPackage applicationPackage = this.createApplicationPackage(applicationFile, applicationId, sessionId, currentlyActiveSessionId, false);
            SessionZooKeeperClient sessionZooKeeperClient = this.createSessionZooKeeperClient(sessionId);
            return new LocalSession(this.tenantName, sessionId, applicationPackage, sessionZooKeeperClient, this.applicationRepo);
        }
        catch (Exception e) {
            throw new RuntimeException("Error creating session " + sessionId, e);
        }
    }

    private ApplicationPackage createApplicationPackage(File applicationFile, ApplicationId applicationId, long sessionId, Optional<Long> currentlyActiveSessionId, boolean internalRedeploy) throws IOException {
        File userApplicationDir = this.getSessionAppDir(sessionId);
        IOUtils.copyDirectory((File)applicationFile, (File)userApplicationDir);
        ApplicationPackage applicationPackage = this.createApplication(applicationFile, userApplicationDir, applicationId, sessionId, currentlyActiveSessionId, internalRedeploy);
        applicationPackage.writeMetaData();
        return applicationPackage;
    }

    LocalSession createSessionFromId(long sessionId) {
        File sessionDir = this.getAndValidateExistingSessionAppDir(sessionId);
        FilesApplicationPackage applicationPackage = FilesApplicationPackage.fromFile((File)sessionDir);
        SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
        return new LocalSession(this.tenantName, sessionId, (ApplicationPackage)applicationPackage, sessionZKClient, this.applicationRepo);
    }

    Optional<LocalSession> createLocalSessionUsingDistributedApplicationPackage(long sessionId) {
        if (this.applicationRepo.hasLocalSession(sessionId)) {
            log.log(Level.FINE, "Local session for session id " + sessionId + " already exists");
            return Optional.of(this.createSessionFromId(sessionId));
        }
        log.log(Level.INFO, "Creating local session for session id " + sessionId);
        SessionZooKeeperClient sessionZKClient = this.createSessionZooKeeperClient(sessionId);
        FileReference fileReference = sessionZKClient.readApplicationPackageReference();
        log.log(Level.FINE, "File reference for session id " + sessionId + ": " + fileReference);
        if (fileReference != null) {
            File sessionDir;
            File rootDir = new File(Defaults.getDefaults().underVespaHome(this.componentRegistry.getConfigserverConfig().fileReferencesDir()));
            FileDirectory fileDirectory = new FileDirectory(rootDir);
            try {
                sessionDir = fileDirectory.getFile(fileReference);
            }
            catch (IllegalArgumentException e) {
                log.log(Level.INFO, "File reference for session id " + sessionId + ": " + fileReference + " not found in " + fileDirectory);
                return Optional.empty();
            }
            ApplicationId applicationId = sessionZKClient.readApplicationId();
            return Optional.of(this.createLocalSession(sessionDir, applicationId, sessionId, this.getActiveSessionId(applicationId)));
        }
        return Optional.empty();
    }

    private Optional<Long> getActiveSessionId(ApplicationId applicationId) {
        List<ApplicationId> applicationIds = this.applicationRepo.activeApplications();
        return applicationIds.contains(applicationId) ? Optional.of(this.applicationRepo.requireActiveSessionOf(applicationId)) : Optional.empty();
    }

    private long getNextSessionId() {
        return new SessionCounter(this.componentRegistry.getConfigCurator(), this.tenantName).nextSessionId();
    }

    private Path getSessionPath(long sessionId) {
        return this.sessionsPath.append(String.valueOf(sessionId));
    }

    private SessionZooKeeperClient createSessionZooKeeperClient(long sessionId) {
        String serverId = this.componentRegistry.getConfigserverConfig().serverId();
        Optional nodeFlavors = this.componentRegistry.getZone().nodeFlavors();
        Path sessionPath = this.getSessionPath(sessionId);
        return new SessionZooKeeperClient(this.curator, this.componentRegistry.getConfigCurator(), sessionPath, serverId, nodeFlavors);
    }

    private File getAndValidateExistingSessionAppDir(long sessionId) {
        File appDir = this.getSessionAppDir(sessionId);
        if (!appDir.exists() || !appDir.isDirectory()) {
            throw new IllegalArgumentException("Unable to find correct application directory for session " + sessionId);
        }
        return appDir;
    }

    private File getSessionAppDir(long sessionId) {
        return new TenantFileSystemDirs(this.componentRegistry.getConfigServerDB(), this.tenantName).getUserApplicationDir(sessionId);
    }

    public String toString() {
        return this.getLocalSessions().toString();
    }

    private static class DeleteOperation
    implements FileOperation {
        private final String pathToDelete;

        DeleteOperation(String pathToDelete) {
            this.pathToDelete = pathToDelete;
        }

        @Override
        public void commit() {
            IOUtils.recursiveDeleteDir((File)new File(this.pathToDelete));
        }
    }

    private static interface FileOperation
    extends Transaction.Operation {
        public void commit();
    }

    private static class FileOperations {
        private FileOperations() {
        }

        public static DeleteOperation delete(String pathToDelete) {
            return new DeleteOperation(pathToDelete);
        }
    }

    private static class FileTransaction
    extends AbstractTransaction {
        private FileTransaction() {
        }

        public static FileTransaction from(FileOperation operation) {
            FileTransaction transaction = new FileTransaction();
            transaction.add(operation);
            return transaction;
        }

        public void prepare() {
        }

        public void commit() {
            for (Transaction.Operation operation : this.operations()) {
                ((FileOperation)operation).commit();
            }
        }
    }
}

