/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.session;

import com.google.common.collect.HashMultiset;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.path.Path;
import com.yahoo.transaction.NestedTransaction;
import com.yahoo.vespa.config.server.GlobalComponentRegistry;
import com.yahoo.vespa.config.server.ReloadHandler;
import com.yahoo.vespa.config.server.application.TenantApplications;
import com.yahoo.vespa.config.server.deploy.TenantFileSystemDirs;
import com.yahoo.vespa.config.server.monitoring.MetricUpdater;
import com.yahoo.vespa.config.server.monitoring.Metrics;
import com.yahoo.vespa.config.server.session.LocalSession;
import com.yahoo.vespa.config.server.session.LocalSessionStateWatcher;
import com.yahoo.vespa.config.server.session.RemoteSession;
import com.yahoo.vespa.config.server.session.RemoteSessionStateWatcher;
import com.yahoo.vespa.config.server.session.Session;
import com.yahoo.vespa.config.server.session.SessionCache;
import com.yahoo.vespa.config.server.session.SessionFactory;
import com.yahoo.vespa.config.server.tenant.TenantRepository;
import com.yahoo.vespa.curator.Curator;
import com.yahoo.vespa.flags.BooleanFlag;
import com.yahoo.vespa.flags.FlagSource;
import com.yahoo.vespa.flags.Flags;
import java.io.File;
import java.io.FilenameFilter;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;

public class SessionRepository {
    private static final Logger log = Logger.getLogger(SessionRepository.class.getName());
    private static final FilenameFilter sessionApplicationsFilter = (dir, name) -> name.matches("\\d+");
    private final SessionCache<LocalSession> localSessionCache;
    private final SessionCache<RemoteSession> remoteSessionCache = new SessionCache();
    private final Map<Long, LocalSessionStateWatcher> localSessionStateWatchers = new HashMap<Long, LocalSessionStateWatcher>();
    private final Map<Long, RemoteSessionStateWatcher> remoteSessionStateWatchers = new HashMap<Long, RemoteSessionStateWatcher>();
    private final Duration sessionLifetime;
    private final Clock clock;
    private final Curator curator;
    private final Executor zkWatcherExecutor;
    private final TenantFileSystemDirs tenantFileSystemDirs;
    private final BooleanFlag distributeApplicationPackage;
    private final ReloadHandler reloadHandler;
    private final MetricUpdater metrics;
    private final Curator.DirectoryCache directoryCache;
    private final TenantApplications applicationRepo;
    private final Path sessionsPath;
    private final SessionFactory sessionFactory;
    private final TenantName tenantName;

    public SessionRepository(TenantName tenantName, GlobalComponentRegistry componentRegistry, SessionFactory sessionFactory, TenantApplications applicationRepo, ReloadHandler reloadHandler, FlagSource flagSource) {
        this.tenantName = tenantName;
        this.localSessionCache = new SessionCache();
        this.clock = componentRegistry.getClock();
        this.curator = componentRegistry.getCurator();
        this.sessionLifetime = Duration.ofSeconds(componentRegistry.getConfigserverConfig().sessionLifetime());
        this.zkWatcherExecutor = command -> componentRegistry.getZkWatcherExecutor().execute((Object)tenantName, command);
        this.tenantFileSystemDirs = new TenantFileSystemDirs(componentRegistry.getConfigServerDB(), tenantName);
        this.sessionFactory = sessionFactory;
        this.applicationRepo = applicationRepo;
        this.loadLocalSessions(sessionFactory);
        this.distributeApplicationPackage = (BooleanFlag)Flags.CONFIGSERVER_DISTRIBUTE_APPLICATION_PACKAGE.bindTo(flagSource);
        this.reloadHandler = reloadHandler;
        this.sessionsPath = TenantRepository.getSessionsPath(tenantName);
        this.metrics = componentRegistry.getMetrics().getOrCreateMetricUpdater(Metrics.createDimensions(tenantName));
        this.initializeRemoteSessions();
        this.directoryCache = this.curator.createDirectoryCache(this.sessionsPath.getAbsolute(), false, false, componentRegistry.getZkCacheExecutor());
        this.directoryCache.addListener(this::childEvent);
        this.directoryCache.start();
    }

    public synchronized void addSession(LocalSession session) {
        this.localSessionCache.addSession(session);
        Path sessionsPath = TenantRepository.getSessionsPath(session.getTenantName());
        long sessionId = session.getSessionId();
        Curator.FileCache fileCache = this.curator.createFileCache(sessionsPath.append(String.valueOf(sessionId)).append("/sessionState").getAbsolute(), false);
        this.localSessionStateWatchers.put(sessionId, new LocalSessionStateWatcher(fileCache, session, this, this.zkWatcherExecutor));
    }

    public LocalSession getSession(long sessionId) {
        return this.localSessionCache.getSession(sessionId);
    }

    public List<LocalSession> getSessions() {
        return this.localSessionCache.getSessions();
    }

    private void loadLocalSessions(SessionFactory sessionFactory) {
        File[] sessions = this.tenantFileSystemDirs.sessionsPath().listFiles(sessionApplicationsFilter);
        if (sessions == null) {
            return;
        }
        for (File session : sessions) {
            try {
                this.addSession(sessionFactory.createSessionFromId(Long.parseLong(session.getName())));
            }
            catch (IllegalArgumentException e) {
                log.log(Level.WARNING, "Could not load session '" + session.getAbsolutePath() + "':" + e.getMessage() + ", skipping it.");
            }
        }
    }

    public void deleteExpiredSessions(Map<ApplicationId, Long> activeSessions) {
        log.log(Level.FINE, "Purging old sessions");
        try {
            for (LocalSession candidate : this.localSessionCache.getSessions()) {
                ApplicationId applicationId;
                Long activeSession;
                Instant createTime = candidate.getCreateTime();
                log.log(Level.FINE, "Candidate session for deletion: " + candidate.getSessionId() + ", created: " + createTime);
                if (this.hasExpired(candidate) && !this.isActiveSession(candidate)) {
                    this.deleteSession(candidate);
                    continue;
                }
                if (!createTime.plus(Duration.ofDays(1L)).isBefore(this.clock.instant()) || (activeSession = activeSessions.get(applicationId = candidate.getApplicationId())) != null && activeSession.longValue() == candidate.getSessionId()) continue;
                this.deleteSession(candidate);
                log.log(Level.INFO, "Deleted inactive session " + candidate.getSessionId() + " created " + createTime + " for '" + applicationId + "'");
            }
        }
        catch (Throwable e) {
            log.log(Level.WARNING, "Error when purging old sessions ", e);
        }
        log.log(Level.FINE, "Done purging old sessions");
    }

    private boolean hasExpired(LocalSession candidate) {
        return candidate.getCreateTime().plus(this.sessionLifetime).isBefore(this.clock.instant());
    }

    private boolean isActiveSession(LocalSession candidate) {
        return candidate.getStatus() == Session.Status.ACTIVATE;
    }

    public void deleteSession(LocalSession session) {
        long sessionId = session.getSessionId();
        log.log(Level.FINE, "Deleting local session " + sessionId);
        LocalSessionStateWatcher watcher = this.localSessionStateWatchers.remove(sessionId);
        if (watcher != null) {
            watcher.close();
        }
        this.localSessionCache.removeSession(sessionId);
        NestedTransaction transaction = new NestedTransaction();
        session.delete(transaction);
        transaction.commit();
    }

    public void close() {
        this.deleteAllSessions();
        this.tenantFileSystemDirs.delete();
        try {
            if (this.directoryCache != null) {
                this.directoryCache.close();
            }
        }
        catch (Exception e) {
            log.log(Level.WARNING, "Exception when closing path cache", e);
        }
        finally {
            this.checkForRemovedSessions(new ArrayList<Long>());
        }
    }

    private void deleteAllSessions() {
        ArrayList<LocalSession> sessions = new ArrayList<LocalSession>(this.localSessionCache.getSessions());
        for (LocalSession session : sessions) {
            this.deleteSession(session);
        }
    }

    public RemoteSession getRemoteSession(long sessionId) {
        return this.remoteSessionCache.getSession(sessionId);
    }

    public List<Long> getRemoteSessions() {
        return this.getSessionList(this.curator.getChildren(this.sessionsPath));
    }

    public void addRemoteSession(RemoteSession session) {
        this.remoteSessionCache.addSession(session);
        this.metrics.incAddedSessions();
    }

    public int deleteExpiredRemoteSessions(Clock clock, Duration expiryTime) {
        int deleted = 0;
        for (long sessionId : this.getRemoteSessions()) {
            RemoteSession session = this.remoteSessionCache.getSession(sessionId);
            if (session == null || session.getStatus() == Session.Status.ACTIVATE || !this.sessionHasExpired(session.getCreateTime(), expiryTime, clock)) continue;
            log.log(Level.INFO, "Remote session " + sessionId + " for " + this.tenantName + " has expired, deleting it");
            session.delete();
            ++deleted;
        }
        return deleted;
    }

    private boolean sessionHasExpired(Instant created, Duration expiryTime, Clock clock) {
        return created.plus(expiryTime).isBefore(clock.instant());
    }

    private List<Long> getSessionListFromDirectoryCache(List<ChildData> children) {
        return this.getSessionList(children.stream().map(child -> Path.fromString((String)child.getPath()).getName()).collect(Collectors.toList()));
    }

    private List<Long> getSessionList(List<String> children) {
        return children.stream().map(Long::parseLong).collect(Collectors.toList());
    }

    private void initializeRemoteSessions() throws NumberFormatException {
        this.getRemoteSessions().forEach(this::sessionAdded);
    }

    private synchronized void sessionsChanged() throws NumberFormatException {
        List<Long> sessions = this.getSessionListFromDirectoryCache(this.directoryCache.getCurrentData());
        this.checkForRemovedSessions(sessions);
        this.checkForAddedSessions(sessions);
    }

    private void checkForRemovedSessions(List<Long> sessions) {
        for (RemoteSession session : this.remoteSessionCache.getSessions()) {
            if (sessions.contains(session.getSessionId())) continue;
            this.sessionRemoved(session.getSessionId());
        }
    }

    private void checkForAddedSessions(List<Long> sessions) {
        for (Long sessionId : sessions) {
            if (this.remoteSessionCache.getSession(sessionId) != null) continue;
            this.sessionAdded(sessionId);
        }
    }

    private void sessionAdded(long sessionId) {
        log.log(Level.FINE, () -> "Adding session to SessionRepository: " + sessionId);
        RemoteSession session = this.sessionFactory.createRemoteSession(sessionId);
        Path sessionPath = this.sessionsPath.append(String.valueOf(sessionId));
        Curator.FileCache fileCache = this.curator.createFileCache(sessionPath.append("/sessionState").getAbsolute(), false);
        fileCache.addListener(this::nodeChanged);
        this.loadSessionIfActive(session);
        this.addRemoteSession(session);
        this.remoteSessionStateWatchers.put(sessionId, new RemoteSessionStateWatcher(fileCache, this.reloadHandler, session, this.metrics, this.zkWatcherExecutor));
        if (this.distributeApplicationPackage.value()) {
            this.sessionFactory.createLocalSessionUsingDistributedApplicationPackage(sessionId);
        }
    }

    private void sessionRemoved(long sessionId) {
        RemoteSessionStateWatcher watcher = this.remoteSessionStateWatchers.remove(sessionId);
        if (watcher != null) {
            watcher.close();
        }
        this.remoteSessionCache.removeSession(sessionId);
        this.metrics.incRemovedSessions();
    }

    private void loadSessionIfActive(RemoteSession session) {
        for (ApplicationId applicationId : this.applicationRepo.activeApplications()) {
            if (this.applicationRepo.requireActiveSessionOf(applicationId) != session.getSessionId()) continue;
            log.log(Level.FINE, () -> "Found active application for session " + session.getSessionId() + " , loading it");
            this.reloadHandler.reloadConfig(session.ensureApplicationLoaded());
            log.log(Level.INFO, session.logPre() + "Application activated successfully: " + applicationId + " (generation " + session.getSessionId() + ")");
            return;
        }
    }

    private void nodeChanged() {
        this.zkWatcherExecutor.execute(() -> {
            HashMultiset sessionMetrics = HashMultiset.create();
            for (RemoteSession session : this.remoteSessionCache.getSessions()) {
                sessionMetrics.add((Object)session.getStatus());
            }
            this.metrics.setNewSessions(sessionMetrics.count((Object)Session.Status.NEW));
            this.metrics.setPreparedSessions(sessionMetrics.count((Object)Session.Status.PREPARE));
            this.metrics.setActivatedSessions(sessionMetrics.count((Object)Session.Status.ACTIVATE));
            this.metrics.setDeactivatedSessions(sessionMetrics.count((Object)Session.Status.DEACTIVATE));
        });
    }

    private void childEvent(CuratorFramework ignored, PathChildrenCacheEvent event) {
        this.zkWatcherExecutor.execute(() -> {
            log.log(Level.FINE, () -> "Got child event: " + event);
            switch (event.getType()) {
                case CHILD_ADDED: {
                    this.sessionsChanged();
                    this.synchronizeOnNew(this.getSessionListFromDirectoryCache(Collections.singletonList(event.getData())));
                    break;
                }
                case CHILD_REMOVED: {
                    this.sessionsChanged();
                    break;
                }
                case CONNECTION_RECONNECTED: {
                    this.sessionsChanged();
                }
            }
        });
    }

    private void synchronizeOnNew(List<Long> sessionList) {
        for (long sessionId : sessionList) {
            RemoteSession session = this.remoteSessionCache.getSession(sessionId);
            if (session == null) continue;
            log.log(Level.FINE, () -> session.logPre() + "Confirming upload for session " + sessionId);
            session.confirmUpload();
        }
    }

    public String toString() {
        return this.getSessions().toString();
    }
}

