/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.session;

import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.path.Path;
import com.yahoo.transaction.NestedTransaction;
import com.yahoo.vespa.config.server.GlobalComponentRegistry;
import com.yahoo.vespa.config.server.deploy.TenantFileSystemDirs;
import com.yahoo.vespa.config.server.session.LocalSession;
import com.yahoo.vespa.config.server.session.LocalSessionLoader;
import com.yahoo.vespa.config.server.session.LocalSessionStateWatcher;
import com.yahoo.vespa.config.server.session.Session;
import com.yahoo.vespa.config.server.session.SessionCache;
import com.yahoo.vespa.config.server.tenant.TenantRepository;
import com.yahoo.vespa.curator.Curator;
import com.yahoo.vespa.flags.Flags;
import com.yahoo.vespa.flags.LongFlag;
import java.io.File;
import java.io.FilenameFilter;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.logging.Level;
import java.util.logging.Logger;

public class LocalSessionRepo {
    private static final Logger log = Logger.getLogger(LocalSessionRepo.class.getName());
    private static final FilenameFilter sessionApplicationsFilter = (dir, name) -> name.matches("\\d+");
    private final SessionCache<LocalSession> sessionCache;
    private final Map<Long, LocalSessionStateWatcher> sessionStateWatchers = new HashMap<Long, LocalSessionStateWatcher>();
    private final Duration sessionLifetime;
    private final Clock clock;
    private final Curator curator;
    private final Executor zkWatcherExecutor;
    private final TenantFileSystemDirs tenantFileSystemDirs;
    private final LongFlag expiryTimeFlag;

    public LocalSessionRepo(TenantName tenantName, GlobalComponentRegistry componentRegistry, LocalSessionLoader loader) {
        this(tenantName, componentRegistry);
        this.loadSessions(loader);
    }

    public LocalSessionRepo(TenantName tenantName, GlobalComponentRegistry componentRegistry) {
        this.sessionCache = new SessionCache();
        this.clock = componentRegistry.getClock();
        this.curator = componentRegistry.getCurator();
        this.sessionLifetime = Duration.ofSeconds(componentRegistry.getConfigserverConfig().sessionLifetime());
        this.zkWatcherExecutor = command -> componentRegistry.getZkWatcherExecutor().execute((Object)tenantName, command);
        this.tenantFileSystemDirs = new TenantFileSystemDirs(componentRegistry.getConfigServerDB(), tenantName);
        this.expiryTimeFlag = (LongFlag)Flags.CONFIGSERVER_LOCAL_SESSIONS_EXPIRY_INTERVAL_IN_DAYS.bindTo(componentRegistry.getFlagSource());
    }

    public synchronized void addSession(LocalSession session) {
        this.sessionCache.addSession(session);
        Path sessionsPath = TenantRepository.getSessionsPath(session.getTenantName());
        long sessionId = session.getSessionId();
        Curator.FileCache fileCache = this.curator.createFileCache(sessionsPath.append(String.valueOf(sessionId)).append("/sessionState").getAbsolute(), false);
        this.sessionStateWatchers.put(sessionId, new LocalSessionStateWatcher(fileCache, session, this, this.zkWatcherExecutor));
    }

    public LocalSession getSession(long sessionId) {
        return this.sessionCache.getSession(sessionId);
    }

    public List<LocalSession> getSessions() {
        return this.sessionCache.getSessions();
    }

    private void loadSessions(LocalSessionLoader loader) {
        File[] sessions = this.tenantFileSystemDirs.sessionsPath().listFiles(sessionApplicationsFilter);
        if (sessions == null) {
            return;
        }
        for (File session : sessions) {
            try {
                this.addSession(loader.loadSession(Long.parseLong(session.getName())));
            }
            catch (IllegalArgumentException e) {
                log.log(Level.WARNING, "Could not load session '" + session.getAbsolutePath() + "':" + e.getMessage() + ", skipping it.");
            }
        }
    }

    public void deleteExpiredSessions(Map<ApplicationId, Long> activeSessions) {
        log.log(Level.FINE, "Purging old sessions");
        try {
            for (LocalSession candidate : this.sessionCache.getSessions()) {
                ApplicationId applicationId;
                Long activeSession;
                Instant createTime = candidate.getCreateTime();
                log.log(Level.FINE, "Candidate session for deletion: " + candidate.getSessionId() + ", created: " + createTime);
                if (this.hasExpired(candidate) && !this.isActiveSession(candidate)) {
                    this.deleteSession(candidate);
                    continue;
                }
                if (!createTime.plus(Duration.ofDays(this.expiryTimeFlag.value())).isBefore(this.clock.instant()) || (activeSession = activeSessions.get(applicationId = candidate.getApplicationId())) != null && activeSession.longValue() == candidate.getSessionId()) continue;
                this.deleteSession(candidate);
                log.log(Level.INFO, "Deleted inactive session " + candidate.getSessionId() + " created " + createTime + " for '" + applicationId + "'");
            }
        }
        catch (Throwable e) {
            log.log(Level.WARNING, "Error when purging old sessions ", e);
        }
        log.log(Level.FINE, "Done purging old sessions");
    }

    private boolean hasExpired(LocalSession candidate) {
        return candidate.getCreateTime().plus(this.sessionLifetime).isBefore(this.clock.instant());
    }

    private boolean isActiveSession(LocalSession candidate) {
        return candidate.getStatus() == Session.Status.ACTIVATE;
    }

    public void deleteSession(LocalSession session) {
        long sessionId = session.getSessionId();
        log.log(Level.FINE, "Deleting local session " + sessionId);
        LocalSessionStateWatcher watcher = this.sessionStateWatchers.remove(sessionId);
        if (watcher != null) {
            watcher.close();
        }
        this.sessionCache.removeSession(sessionId);
        NestedTransaction transaction = new NestedTransaction();
        session.delete(transaction);
        transaction.commit();
    }

    public void close() {
        this.deleteAllSessions();
        this.tenantFileSystemDirs.delete();
    }

    private void deleteAllSessions() {
        ArrayList<LocalSession> sessions = new ArrayList<LocalSession>(this.sessionCache.getSessions());
        for (LocalSession session : sessions) {
            this.deleteSession(session);
        }
    }
}

