/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.config.server.tenant;

import com.google.common.collect.ImmutableSet;
import com.google.inject.Inject;
import com.yahoo.cloud.config.ConfigserverConfig;
import com.yahoo.concurrent.ThreadFactoryFactory;
import com.yahoo.config.provision.ApplicationId;
import com.yahoo.config.provision.TenantName;
import com.yahoo.log.LogLevel;
import com.yahoo.path.Path;
import com.yahoo.vespa.config.server.GlobalComponentRegistry;
import com.yahoo.vespa.config.server.monitoring.MetricUpdater;
import com.yahoo.vespa.config.server.tenant.Tenant;
import com.yahoo.vespa.config.server.tenant.TenantBuilder;
import com.yahoo.vespa.config.server.tenant.TenantListener;
import com.yahoo.vespa.curator.Curator;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheListener;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.zookeeper.KeeperException;

public class Tenants
implements ConnectionStateListener,
PathChildrenCacheListener {
    public static final TenantName HOSTED_VESPA_TENANT = TenantName.from((String)"hosted-vespa");
    private static final TenantName DEFAULT_TENANT = TenantName.defaultName();
    private static final Path tenantsPath = Path.fromString((String)"/config/v2/tenants/");
    private static final Path vespaPath = Path.fromString((String)"/vespa");
    private static final Logger log = Logger.getLogger(Tenants.class.getName());
    private final Map<TenantName, Tenant> tenants = new LinkedHashMap<TenantName, Tenant>();
    private final GlobalComponentRegistry globalComponentRegistry;
    private final List<TenantListener> tenantListeners = Collections.synchronizedList(new ArrayList());
    private final Curator curator;
    private final MetricUpdater metricUpdater;
    private final ExecutorService pathChildrenExecutor = Executors.newFixedThreadPool(1, ThreadFactoryFactory.getThreadFactory((String)Tenants.class.getName()));
    private final Curator.DirectoryCache directoryCache;

    @Inject
    public Tenants(GlobalComponentRegistry globalComponentRegistry) throws Exception {
        this.globalComponentRegistry = globalComponentRegistry;
        this.curator = globalComponentRegistry.getCurator();
        this.metricUpdater = globalComponentRegistry.getMetrics().getOrCreateMetricUpdater(Collections.emptyMap());
        this.tenantListeners.add(globalComponentRegistry.getTenantListener());
        this.curator.framework().getConnectionStateListenable().addListener((Object)this);
        this.curator.create(tenantsPath);
        this.createSystemTenants(globalComponentRegistry.getConfigserverConfig());
        this.curator.create(vespaPath);
        this.directoryCache = this.curator.createDirectoryCache(tenantsPath.getAbsolute(), false, false, this.pathChildrenExecutor);
        this.directoryCache.start();
        this.directoryCache.addListener((PathChildrenCacheListener)this);
        this.createTenants();
        this.notifyTenantsLoaded();
    }

    public Tenants(GlobalComponentRegistry globalComponentRegistry, Collection<Tenant> tenants) {
        this.globalComponentRegistry = globalComponentRegistry;
        this.curator = globalComponentRegistry.getCurator();
        this.metricUpdater = globalComponentRegistry.getMetrics().getOrCreateMetricUpdater(Collections.emptyMap());
        this.tenantListeners.add(globalComponentRegistry.getTenantListener());
        this.curator.create(tenantsPath);
        this.createSystemTenants(globalComponentRegistry.getConfigserverConfig());
        this.createTenants();
        this.directoryCache = this.curator.createDirectoryCache(tenantsPath.getAbsolute(), false, false, this.pathChildrenExecutor);
        this.tenants.putAll(this.addTenants(tenants));
    }

    private void notifyTenantsLoaded() {
        for (TenantListener tenantListener : this.tenantListeners) {
            tenantListener.onTenantsLoaded();
        }
    }

    private LinkedHashMap<TenantName, Tenant> addTenants(Collection<Tenant> newTenants) {
        LinkedHashMap<TenantName, Tenant> sessionTenants = new LinkedHashMap<TenantName, Tenant>();
        for (Tenant t : newTenants) {
            sessionTenants.put(t.getName(), t);
        }
        log.log((Level)LogLevel.DEBUG, "Tenants at startup: " + sessionTenants);
        this.metricUpdater.setTenants(this.tenants.size());
        return sessionTenants;
    }

    public synchronized void addTenant(TenantName tenantName) throws Exception {
        this.writeTenantPath(tenantName);
        this.createTenant(tenantName);
    }

    private Set<TenantName> readTenantsFromZooKeeper() {
        LinkedHashSet<TenantName> tenants = new LinkedHashSet<TenantName>();
        for (String tenant : this.curator.getChildren(tenantsPath)) {
            tenants.add(TenantName.from((String)tenant));
        }
        return tenants;
    }

    synchronized void createTenants() {
        Set<TenantName> allTenants = this.readTenantsFromZooKeeper();
        log.log((Level)LogLevel.DEBUG, "Create tenants, tenants found in zookeeper: " + allTenants);
        this.checkForRemovedTenants(allTenants);
        this.checkForAddedTenants(allTenants);
        this.metricUpdater.setTenants(this.tenants.size());
    }

    private void checkForRemovedTenants(Set<TenantName> newTenants) {
        LinkedHashMap<TenantName, Tenant> current = new LinkedHashMap<TenantName, Tenant>(this.tenants);
        for (Map.Entry entry : current.entrySet()) {
            TenantName tenant = (TenantName)entry.getKey();
            if (newTenants.contains(tenant) || DEFAULT_TENANT.equals((Object)tenant)) continue;
            this.notifyRemovedTenant(tenant);
            ((Tenant)entry.getValue()).close();
            this.tenants.remove(tenant);
        }
    }

    private void checkForAddedTenants(Set<TenantName> newTenants) {
        ExecutorService executor = Executors.newFixedThreadPool(this.globalComponentRegistry.getConfigserverConfig().numParallelTenantLoaders());
        for (TenantName tenantName : newTenants) {
            if (this.tenants.containsKey(tenantName)) continue;
            executor.execute(() -> this.createTenant(tenantName));
        }
        executor.shutdown();
        try {
            executor.awaitTermination(365L, TimeUnit.DAYS);
        }
        catch (InterruptedException e) {
            throw new RuntimeException("Executor for creating tenants did not terminate within timeout");
        }
    }

    private void createTenant(TenantName tenantName) {
        if (this.tenants.containsKey(tenantName)) {
            return;
        }
        try {
            Tenant tenant = TenantBuilder.create(this.globalComponentRegistry, tenantName, Tenants.getTenantPath(tenantName)).build();
            this.notifyNewTenant(tenant);
            this.tenants.put(tenantName, tenant);
        }
        catch (Exception e) {
            log.log(LogLevel.WARNING, "Error loading tenant '" + tenantName + "', skipping.", e);
        }
    }

    public synchronized Tenant defaultTenant() {
        return this.tenants.get(DEFAULT_TENANT);
    }

    private void notifyNewTenant(Tenant tenant) {
        for (TenantListener listener : this.tenantListeners) {
            listener.onTenantCreate(tenant.getName(), tenant);
        }
    }

    private void notifyRemovedTenant(TenantName name) {
        for (TenantListener listener : this.tenantListeners) {
            listener.onTenantDelete(name);
        }
    }

    private synchronized void createSystemTenants(ConfigserverConfig configserverConfig) {
        ArrayList<TenantName> systemTenants = new ArrayList<TenantName>();
        systemTenants.add(DEFAULT_TENANT);
        if (configserverConfig.hostedVespa()) {
            systemTenants.add(HOSTED_VESPA_TENANT);
        }
        for (TenantName tenantName : systemTenants) {
            try {
                this.writeTenantPath(tenantName);
            }
            catch (RuntimeException e) {
                if (e.getCause().getClass() == KeeperException.NodeExistsException.class) continue;
                throw e;
            }
        }
    }

    private synchronized void writeTenantPath(TenantName name) {
        Path tenantPath = Tenants.getTenantPath(name);
        this.curator.createAtomically(new Path[]{tenantPath, tenantPath.append("sessions"), tenantPath.append("applications")});
    }

    public synchronized Tenants deleteTenant(TenantName name) {
        if (name.equals((Object)DEFAULT_TENANT)) {
            throw new IllegalArgumentException("Deleting 'default' tenant is not allowed");
        }
        Tenant tenant = this.tenants.get(name);
        tenant.delete();
        return this;
    }

    String tenantZkPath(TenantName tenant) {
        return Tenants.getTenantPath(tenant).getAbsolute();
    }

    public static String logPre(ApplicationId app) {
        if (DEFAULT_TENANT.equals((Object)app.tenant())) {
            return "";
        }
        StringBuilder ret = new StringBuilder().append(Tenants.logPre(app.tenant())).append("app:" + app.application().value()).append(":" + app.instance().value()).append(" ");
        return ret.toString();
    }

    public static String logPre(TenantName tenant) {
        if (DEFAULT_TENANT.equals((Object)tenant)) {
            return "";
        }
        StringBuilder ret = new StringBuilder().append("tenant:" + tenant.value()).append(" ");
        return ret.toString();
    }

    public void stateChanged(CuratorFramework framework, ConnectionState connectionState) {
        switch (connectionState) {
            case CONNECTED: {
                this.metricUpdater.incZKConnected();
                break;
            }
            case SUSPENDED: {
                this.metricUpdater.incZKSuspended();
                break;
            }
            case RECONNECTED: {
                this.metricUpdater.incZKReconnected();
                break;
            }
            case LOST: {
                this.metricUpdater.incZKConnectionLost();
                break;
            }
        }
    }

    public void childEvent(CuratorFramework framework, PathChildrenCacheEvent event) throws Exception {
        switch (event.getType()) {
            case CHILD_ADDED: 
            case CHILD_REMOVED: {
                this.createTenants();
            }
        }
    }

    public void close() {
        this.directoryCache.close();
        this.pathChildrenExecutor.shutdown();
    }

    public boolean checkThatTenantExists(TenantName tenant) {
        return this.tenants.containsKey(tenant);
    }

    public Tenant getTenant(TenantName tenantName) {
        return this.tenants.get(tenantName);
    }

    public Set<TenantName> getAllTenantNames() {
        return ImmutableSet.copyOf(this.tenants.keySet());
    }

    public Collection<Tenant> getAllTenants() {
        return ImmutableSet.copyOf(this.tenants.values());
    }

    public static Path getTenantPath(TenantName tenantName) {
        return tenantsPath.append(tenantName.value());
    }
}

