/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.vespa.clustercontroller.core;

import ai.vespa.metrics.StorageMetrics;
import com.yahoo.lang.MutableBoolean;
import com.yahoo.lang.SettableOptional;
import com.yahoo.vdslib.distribution.ConfiguredNode;
import com.yahoo.vdslib.distribution.Group;
import com.yahoo.vdslib.state.ClusterState;
import com.yahoo.vdslib.state.Node;
import com.yahoo.vdslib.state.NodeState;
import com.yahoo.vdslib.state.NodeType;
import com.yahoo.vdslib.state.State;
import com.yahoo.vespa.clustercontroller.core.ClusterInfo;
import com.yahoo.vespa.clustercontroller.core.ContentCluster;
import com.yahoo.vespa.clustercontroller.core.DistributorNodeInfo;
import com.yahoo.vespa.clustercontroller.core.HierarchicalGroupVisiting;
import com.yahoo.vespa.clustercontroller.core.NodeInfo;
import com.yahoo.vespa.clustercontroller.core.StorageNodeInfo;
import com.yahoo.vespa.clustercontroller.core.hostinfo.HostInfo;
import com.yahoo.vespa.clustercontroller.core.hostinfo.Metrics;
import com.yahoo.vespa.clustercontroller.core.hostinfo.StorageNode;
import com.yahoo.vespa.clustercontroller.utils.staterestapi.requests.SetUnitStateRequest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

public class NodeStateChangeChecker {
    private static final Logger log = Logger.getLogger(NodeStateChangeChecker.class.getName());
    private static final String BUCKETS_METRIC_NAME = StorageMetrics.VDS_DATASTORED_BUCKET_SPACE_BUCKETS_TOTAL.baseName();
    private static final String ENTRIES_METRIC_NAME = StorageMetrics.VDS_DATASTORED_BUCKET_SPACE_ENTRIES.baseName();
    private static final String DOCS_METRIC_NAME = StorageMetrics.VDS_DATASTORED_BUCKET_SPACE_DOCS.baseName();
    private static final Map<String, String> DEFAULT_SPACE_METRIC_DIMENSIONS = Map.of("bucketSpace", "default");
    private final int requiredRedundancy;
    private final HierarchicalGroupVisiting groupVisiting;
    private final ClusterInfo clusterInfo;
    private final boolean inMoratorium;
    private final int maxNumberOfGroupsAllowedToBeDown;

    public NodeStateChangeChecker(ContentCluster cluster, boolean inMoratorium) {
        this.requiredRedundancy = cluster.getDistribution().getRedundancy();
        this.groupVisiting = new HierarchicalGroupVisiting(cluster.getDistribution());
        this.clusterInfo = cluster.clusterInfo();
        this.inMoratorium = inMoratorium;
        this.maxNumberOfGroupsAllowedToBeDown = cluster.maxNumberOfGroupsAllowedToBeDown();
        if (!this.isGroupedSetup() && this.maxNumberOfGroupsAllowedToBeDown > 1) {
            throw new IllegalArgumentException("Cannot have both 1 group and maxNumberOfGroupsAllowedToBeDown > 1");
        }
    }

    public Result evaluateTransition(Node node, ClusterState clusterState, SetUnitStateRequest.Condition condition, NodeState oldWantedState, NodeState newWantedState) {
        if (condition == SetUnitStateRequest.Condition.FORCE) {
            return Result.allow();
        }
        if (this.inMoratorium) {
            return Result.disallow("Master cluster controller is bootstrapping and in moratorium");
        }
        if (condition != SetUnitStateRequest.Condition.SAFE) {
            return Result.disallow("Condition not implemented: " + condition.name());
        }
        if (node.getType() != NodeType.STORAGE) {
            return Result.disallow("Safe-set of node state is only supported for storage nodes! Requested node type: " + node.getType().toString());
        }
        StorageNodeInfo nodeInfo = this.clusterInfo.getStorageNodeInfo(node.getIndex());
        if (nodeInfo == null) {
            return Result.disallow("Unknown node " + node);
        }
        if (NodeStateChangeChecker.noChanges(oldWantedState, newWantedState)) {
            return Result.alreadySet();
        }
        return switch (newWantedState.getState()) {
            case State.UP -> this.canSetStateUp(nodeInfo, oldWantedState);
            case State.MAINTENANCE -> this.canSetStateMaintenanceTemporarily(nodeInfo, clusterState, newWantedState.getDescription());
            case State.DOWN -> this.canSetStateDownPermanently(nodeInfo, clusterState, newWantedState.getDescription());
            default -> Result.disallow("Destination node state unsupported in safe mode: " + newWantedState);
        };
    }

    private static boolean noChanges(NodeState oldWantedState, NodeState newWantedState) {
        return newWantedState.getState().equals((Object)oldWantedState.getState()) && Objects.equals(newWantedState.getDescription(), oldWantedState.getDescription());
    }

    private static NodeDataMetrics dataMetricsFromHostInfo(HostInfo hostInfo) {
        return new NodeDataMetrics(hostInfo.getMetrics().getValueAt(BUCKETS_METRIC_NAME, DEFAULT_SPACE_METRIC_DIMENSIONS), hostInfo.getMetrics().getValueAt(ENTRIES_METRIC_NAME, DEFAULT_SPACE_METRIC_DIMENSIONS), hostInfo.getMetrics().getValueAt(DOCS_METRIC_NAME, DEFAULT_SPACE_METRIC_DIMENSIONS));
    }

    private static Optional<Result> checkZeroEntriesStoredOnContentNode(NodeDataMetrics metrics, int nodeIndex) {
        if (metrics.entries.isEmpty() || metrics.entries.get().getLast() == null) {
            return Optional.empty();
        }
        if (metrics.docs.isEmpty() || metrics.docs.get().getLast() == null) {
            log.log(Level.WARNING, "Host info inconsistency: storage node %d reports entry count but not document count".formatted(nodeIndex));
            return Optional.of(Result.disallow("The storage node host info reports stored entry count, but not document count"));
        }
        long lastEntries = metrics.entries.get().getLast();
        long lastDocs = metrics.docs.get().getLast();
        if (lastEntries != 0L) {
            long buckets = metrics.buckets.map(Metrics.Value::getLast).orElse(-1L);
            long tombstones = lastEntries - lastDocs;
            return Optional.of(Result.disallow("The storage node stores %d documents and %d tombstones across %d buckets".formatted(lastDocs, tombstones, buckets)));
        }
        if (lastDocs != 0L) {
            log.log(Level.WARNING, "Host info inconsistency: storage node %d reports 0 entries, but %d documents".formatted(nodeIndex, lastDocs));
            return Optional.of(Result.disallow("The storage node reports 0 entries, but %d documents".formatted(lastDocs)));
        }
        return Optional.of(Result.allow());
    }

    private static Result checkLegacyZeroBucketsStoredOnContentNode(long lastBuckets) {
        if (lastBuckets != 0L) {
            return Result.disallow("The storage node manages %d buckets".formatted(lastBuckets));
        }
        return Result.allow();
    }

    private Result canSetStateDownPermanently(NodeInfo nodeInfo, ClusterState clusterState, String newDescription) {
        Result result = NodeStateChangeChecker.checkIfStateSetWithDifferentDescription(nodeInfo, newDescription);
        if (result.notAllowed()) {
            return result;
        }
        State reportedState = nodeInfo.getReportedState().getState();
        if (reportedState != State.UP) {
            return Result.disallow("Reported state (" + reportedState + ") is not UP, so no bucket data is available");
        }
        State currentState = clusterState.getNodeState(nodeInfo.getNode()).getState();
        if (currentState != State.RETIRED) {
            return Result.disallow("Only retired nodes are allowed to be set to DOWN in safe mode - is " + currentState);
        }
        HostInfo hostInfo = nodeInfo.getHostInfo();
        Integer hostInfoNodeVersion = hostInfo.getClusterStateVersionOrNull();
        int clusterControllerVersion = clusterState.getVersion();
        int nodeIndex = nodeInfo.getNodeIndex();
        if (hostInfoNodeVersion == null || hostInfoNodeVersion != clusterControllerVersion) {
            return Result.disallow("Cluster controller at version " + clusterControllerVersion + " got info for storage node " + nodeIndex + " at a different version " + hostInfoNodeVersion);
        }
        NodeDataMetrics metrics = NodeStateChangeChecker.dataMetricsFromHostInfo(hostInfo);
        if (metrics.buckets.isEmpty() || metrics.buckets.get().getLast() == null) {
            return Result.disallow("Missing last value of the " + BUCKETS_METRIC_NAME + " metric for storage node " + nodeIndex);
        }
        Optional<Result> entriesCheckResult = NodeStateChangeChecker.checkZeroEntriesStoredOnContentNode(metrics, nodeIndex);
        return entriesCheckResult.orElseGet(() -> NodeStateChangeChecker.checkLegacyZeroBucketsStoredOnContentNode(metrics.buckets.get().getLast()));
    }

    private Result canSetStateUp(NodeInfo nodeInfo, NodeState oldWantedState) {
        if (oldWantedState.getState() == State.UP) {
            return Result.alreadySet();
        }
        State reportedState = nodeInfo.getReportedState().getState();
        if (reportedState != State.UP) {
            return Result.disallow("Refuse to set wanted state to UP, since the reported state is not UP (" + reportedState + ")");
        }
        return Result.allow();
    }

    private Result canSetStateMaintenanceTemporarily(StorageNodeInfo nodeInfo, ClusterState clusterState, String newDescription) {
        Result result = NodeStateChangeChecker.checkIfStateSetWithDifferentDescription(nodeInfo, newDescription);
        if (result.notAllowed()) {
            return result;
        }
        if (this.isGroupedSetup()) {
            if (this.maxNumberOfGroupsAllowedToBeDown == -1) {
                result = this.checkIfAnotherNodeInAnotherGroupHasWantedState(nodeInfo);
                if (result.notAllowed()) {
                    return result;
                }
                if (this.anotherNodeInGroupAlreadyAllowed(nodeInfo, newDescription)) {
                    return Result.allow();
                }
            } else {
                Optional<Result> optionalResult = this.checkIfOtherNodesHaveWantedState(nodeInfo, newDescription, clusterState);
                if (optionalResult.isPresent()) {
                    return optionalResult.get();
                }
            }
        } else {
            result = this.otherNodeHasWantedState(nodeInfo);
            if (result.notAllowed()) {
                return result;
            }
        }
        if (NodeStateChangeChecker.nodeIsDown(clusterState, nodeInfo)) {
            log.log(Level.FINE, "node is DOWN, allow");
            return Result.allow();
        }
        result = this.checkIfNodesAreUpOrRetired(clusterState);
        if (result.notAllowed()) {
            log.log(Level.FINE, "nodesAreUpOrRetired: " + result);
            return result;
        }
        result = this.checkClusterStateAndRedundancy(nodeInfo.getNode(), clusterState.getVersion());
        if (result.notAllowed()) {
            log.log(Level.FINE, "checkDistributors: " + result);
            return result;
        }
        return Result.allow();
    }

    private boolean isGroupedSetup() {
        return this.groupVisiting.isHierarchical();
    }

    private static Result checkIfStateSetWithDifferentDescription(NodeInfo nodeInfo, String newDescription) {
        State oldWantedState = nodeInfo.getUserWantedState().getState();
        String oldDescription = nodeInfo.getUserWantedState().getDescription();
        if (oldWantedState != State.UP && !oldDescription.equals(newDescription)) {
            return Result.disallow("A conflicting wanted state is already set: " + oldWantedState + ": " + oldDescription);
        }
        return Result.allow();
    }

    private Result checkIfAnotherNodeInAnotherGroupHasWantedState(StorageNodeInfo nodeInfo) {
        SettableOptional anotherNodeHasWantedState = new SettableOptional();
        this.groupVisiting.visit(group -> {
            Result result;
            if (!NodeStateChangeChecker.groupContainsNode(group, nodeInfo.getNode()) && (result = this.otherNodeInGroupHasWantedState(group)).notAllowed()) {
                anotherNodeHasWantedState.set((Object)result);
                return false;
            }
            return true;
        });
        return anotherNodeHasWantedState.asOptional().orElseGet(Result::allow);
    }

    private Optional<Result> checkIfOtherNodesHaveWantedState(StorageNodeInfo nodeInfo, String newDescription, ClusterState clusterState) {
        Optional<Result> result;
        Node node = nodeInfo.getNode();
        Set<Integer> groupsWithNodesWantedStateNotUp = this.groupsWithUserWantedStateNotUp();
        if (groupsWithNodesWantedStateNotUp.size() == 0) {
            log.log(Level.FINE, "groupsWithNodesWantedStateNotUp=0");
            return Optional.empty();
        }
        Set<Integer> groupsWithSameStateAndDescription = this.groupsWithSameStateAndDescription(State.MAINTENANCE, newDescription);
        if (this.aGroupContainsNode(groupsWithSameStateAndDescription, node)) {
            log.log(Level.FINE, "Node is in group with same state and description, allow");
            return Optional.of(Result.allow());
        }
        if (groupsWithSameStateAndDescription.size() == 0) {
            return Optional.of(Result.disallow("Wanted state already set for another node in groups: " + this.sortSetIntoList(groupsWithNodesWantedStateNotUp)));
        }
        Set<Integer> retiredAndNotUpGroups = this.groupsWithNotRetiredAndNotUp(clusterState);
        int numberOfGroupsToConsider = retiredAndNotUpGroups.size();
        if (this.aGroupContainsNode(retiredAndNotUpGroups, node)) {
            numberOfGroupsToConsider = retiredAndNotUpGroups.size() - 1;
        }
        if ((result = this.checkRedundancy(retiredAndNotUpGroups, clusterState)).isPresent() && result.get().notAllowed()) {
            return result;
        }
        if (numberOfGroupsToConsider < this.maxNumberOfGroupsAllowedToBeDown) {
            log.log(Level.FINE, "Allow, retiredAndNotUpGroups=" + retiredAndNotUpGroups);
            return Optional.of(Result.allow());
        }
        return Optional.of(Result.disallow(String.format("At most %d groups can have wanted state: %s", this.maxNumberOfGroupsAllowedToBeDown, this.sortSetIntoList(retiredAndNotUpGroups))));
    }

    private Optional<Result> checkRedundancy(Set<Integer> retiredAndNotUpGroups, ClusterState clusterState) {
        HashSet<Integer> indexesToCheck = new HashSet<Integer>();
        retiredAndNotUpGroups.forEach(index -> this.getNodesInGroup((int)index).forEach(node -> indexesToCheck.add(node.index())));
        for (DistributorNodeInfo distributorNodeInfo : this.clusterInfo.getDistributorNodeInfos()) {
            Result r;
            if (clusterState.getNodeState(distributorNodeInfo.getNode()).getState() != State.UP || !(r = this.checkRedundancySeenFromDistributor(distributorNodeInfo, indexesToCheck)).notAllowed()) continue;
            return Optional.of(r);
        }
        return Optional.empty();
    }

    private static boolean nodeIsDown(ClusterState clusterState, NodeInfo nodeInfo) {
        return clusterState.getNodeState(nodeInfo.getNode()).getState() == State.DOWN;
    }

    private ArrayList<Integer> sortSetIntoList(Set<Integer> set) {
        ArrayList<Integer> sortedList = new ArrayList<Integer>(set);
        Collections.sort(sortedList);
        return sortedList;
    }

    private Result otherNodeInGroupHasWantedState(Group group) {
        for (ConfiguredNode configuredNode : group.getNodes()) {
            int index = configuredNode.index();
            StorageNodeInfo storageNodeInfo = this.clusterInfo.getStorageNodeInfo(index);
            if (storageNodeInfo == null) continue;
            State storageNodeWantedState = storageNodeInfo.getUserWantedState().getState();
            if (storageNodeWantedState != State.UP) {
                return Result.disallow("At most one group can have wanted state: Other storage node " + index + " in group " + group.getIndex() + " has wanted state " + storageNodeWantedState);
            }
            State distributorWantedState = this.clusterInfo.getDistributorNodeInfo(index).getUserWantedState().getState();
            if (distributorWantedState == State.UP) continue;
            return Result.disallow("At most one group can have wanted state: Other distributor " + index + " in group " + group.getIndex() + " has wanted state " + distributorWantedState);
        }
        return Result.allow();
    }

    private Result otherNodeHasWantedState(StorageNodeInfo nodeInfo) {
        for (ConfiguredNode configuredNode : this.clusterInfo.getConfiguredNodes().values()) {
            int index = configuredNode.index();
            if (index == nodeInfo.getNodeIndex()) continue;
            State storageNodeWantedState = this.clusterInfo.getStorageNodeInfo(index).getUserWantedState().getState();
            if (storageNodeWantedState != State.UP) {
                return Result.disallow("At most one node can have a wanted state when #groups = 1: Other storage node " + index + " has wanted state " + storageNodeWantedState);
            }
            State distributorWantedState = this.clusterInfo.getDistributorNodeInfo(index).getUserWantedState().getState();
            if (distributorWantedState == State.UP) continue;
            return Result.disallow("At most one node can have a wanted state when #groups = 1: Other distributor " + index + " has wanted state " + distributorWantedState);
        }
        return Result.allow();
    }

    private boolean anotherNodeInGroupAlreadyAllowed(StorageNodeInfo nodeInfo, String newDescription) {
        MutableBoolean alreadyAllowed = new MutableBoolean(false);
        this.groupVisiting.visit(group -> {
            if (!NodeStateChangeChecker.groupContainsNode(group, nodeInfo.getNode())) {
                return true;
            }
            alreadyAllowed.set(this.anotherNodeInGroupAlreadyAllowed(group, nodeInfo.getNode(), newDescription));
            return false;
        });
        return alreadyAllowed.get();
    }

    private boolean anotherNodeInGroupAlreadyAllowed(Group group, Node node, String newDescription) {
        return group.getNodes().stream().filter(configuredNode -> configuredNode.index() != node.getIndex()).map(configuredNode -> this.clusterInfo.getStorageNodeInfo(configuredNode.index())).filter(Objects::nonNull).map(NodeInfo::getUserWantedState).anyMatch(userWantedState -> userWantedState.getState() == State.MAINTENANCE && Objects.equals(userWantedState.getDescription(), newDescription));
    }

    private static boolean groupContainsNode(Group group, Node node) {
        for (ConfiguredNode configuredNode : group.getNodes()) {
            if (configuredNode.index() != node.getIndex()) continue;
            return true;
        }
        return false;
    }

    private boolean aGroupContainsNode(Collection<Integer> groupIndexes, Node node) {
        for (Group group : this.getGroupsWithIndexes(groupIndexes)) {
            if (!NodeStateChangeChecker.groupContainsNode(group, node)) continue;
            return true;
        }
        return false;
    }

    private List<Group> getGroupsWithIndexes(Collection<Integer> groupIndexes) {
        return this.clusterInfo.getStorageNodeInfos().stream().map(NodeInfo::getGroup).filter(group -> groupIndexes.contains(group.getIndex())).collect(Collectors.toList());
    }

    private Result checkIfNodesAreUpOrRetired(ClusterState clusterState) {
        for (NodeInfo nodeInfo : this.clusterInfo.getAllNodeInfos()) {
            State wantedState = nodeInfo.getUserWantedState().getState();
            if (wantedState != State.UP && wantedState != State.RETIRED) {
                return Result.disallow("Another " + nodeInfo.type() + " wants state " + wantedState.toString().toUpperCase() + ": " + nodeInfo.getNodeIndex());
            }
            State state = clusterState.getNodeState(nodeInfo.getNode()).getState();
            if (state == State.UP || state == State.RETIRED) continue;
            return Result.disallow("Another " + nodeInfo.type() + " has state " + state.toString().toUpperCase() + ": " + nodeInfo.getNodeIndex());
        }
        return Result.allow();
    }

    private Result checkRedundancy(DistributorNodeInfo distributorNodeInfo, Node node) {
        Integer minReplication = this.minReplication(distributorNodeInfo).get(node.getIndex());
        return this.verifyRedundancy(distributorNodeInfo, minReplication, node.getIndex());
    }

    private Result checkRedundancySeenFromDistributor(DistributorNodeInfo distributorNodeInfo, Set<Integer> indexesToCheck) {
        LinkedHashMap<Integer, Integer> replication = new LinkedHashMap<Integer, Integer>(this.minReplication(distributorNodeInfo));
        Integer minReplication = null;
        Integer minReplicationIndex = null;
        for (Map.Entry entry : replication.entrySet()) {
            Integer value = (Integer)entry.getValue();
            Integer nodeIndex = (Integer)entry.getKey();
            if (!indexesToCheck.contains(nodeIndex) || minReplication != null && (value == null || value >= minReplication) || (minReplication = value) == null) continue;
            minReplicationIndex = nodeIndex;
            if (minReplication >= this.requiredRedundancy) continue;
            break;
        }
        return this.verifyRedundancy(distributorNodeInfo, minReplication, minReplicationIndex);
    }

    private Result verifyRedundancy(DistributorNodeInfo distributorNodeInfo, Integer minReplication, Integer minReplicationIndex) {
        if (minReplication != null && minReplication < this.requiredRedundancy) {
            return Result.disallow("Distributor " + distributorNodeInfo.getNodeIndex() + " says storage node " + minReplicationIndex + " has buckets with redundancy as low as " + minReplication + ", but we require at least " + this.requiredRedundancy);
        }
        return Result.allow();
    }

    private Map<Integer, Integer> minReplication(DistributorNodeInfo distributorNodeInfo) {
        HashMap<Integer, Integer> replicationPerNodeIndex = new HashMap<Integer, Integer>();
        for (StorageNode storageNode : distributorNodeInfo.getHostInfo().getDistributor().getStorageNodes()) {
            Integer currentValue = (Integer)replicationPerNodeIndex.get(storageNode.getIndex());
            Integer minReplicationFactor = storageNode.getMinCurrentReplicationFactorOrNull();
            if (currentValue != null && (minReplicationFactor == null || minReplicationFactor >= currentValue)) continue;
            replicationPerNodeIndex.put(storageNode.getIndex(), minReplicationFactor);
        }
        return replicationPerNodeIndex;
    }

    private Result checkClusterStateAndRedundancy(Node node, int clusterStateVersion) {
        if (this.clusterInfo.getDistributorNodeInfos().isEmpty()) {
            return Result.disallow("Not aware of any distributors, probably not safe to upgrade?");
        }
        for (DistributorNodeInfo distributorNodeInfo : this.clusterInfo.getDistributorNodeInfos()) {
            Integer distributorClusterStateVersion = distributorNodeInfo.getHostInfo().getClusterStateVersionOrNull();
            if (distributorClusterStateVersion == null) {
                return Result.disallow("Distributor node " + distributorNodeInfo.getNodeIndex() + " has not reported any cluster state version yet.");
            }
            if (distributorClusterStateVersion != clusterStateVersion) {
                return Result.disallow("Distributor node " + distributorNodeInfo.getNodeIndex() + " does not report same version (" + distributorNodeInfo.getHostInfo().getClusterStateVersionOrNull() + ") as fleetcontroller (" + clusterStateVersion + ")");
            }
            Result storageNodesResult = this.checkRedundancy(distributorNodeInfo, node);
            if (!storageNodesResult.notAllowed()) continue;
            return storageNodesResult;
        }
        return Result.allow();
    }

    private Set<Integer> groupsWithUserWantedStateNotUp() {
        return this.clusterInfo.getAllNodeInfos().stream().filter(sni -> !State.UP.equals((Object)sni.getUserWantedState().getState())).map(NodeInfo::getGroup).filter(Objects::nonNull).filter(Group::isLeafGroup).map(Group::getIndex).collect(Collectors.toSet());
    }

    private Group groupForThisIndex(int groupIndex) {
        return this.clusterInfo.getAllNodeInfos().stream().map(NodeInfo::getGroup).filter(Objects::nonNull).filter(Group::isLeafGroup).filter(group -> group.getIndex() == groupIndex).findFirst().orElseThrow();
    }

    private Set<Integer> groupsWithSameStateAndDescription(State state, String newDescription) {
        return this.clusterInfo.getAllNodeInfos().stream().filter(nodeInfo -> {
            NodeState userWantedState = nodeInfo.getUserWantedState();
            return userWantedState.getState() == state && Objects.equals(userWantedState.getDescription(), newDescription);
        }).map(NodeInfo::getGroup).filter(Objects::nonNull).filter(Group::isLeafGroup).map(Group::getIndex).collect(Collectors.toSet());
    }

    private Set<Integer> groupsWithNotRetiredAndNotUp(ClusterState clusterState) {
        return this.clusterInfo.getAllNodeInfos().stream().filter(nodeInfo -> nodeInfo.getUserWantedState().getState() != State.RETIRED && nodeInfo.getUserWantedState().getState() != State.UP || clusterState.getNodeState(nodeInfo.getNode()).getState() != State.RETIRED && clusterState.getNodeState(nodeInfo.getNode()).getState() != State.UP).map(NodeInfo::getGroup).filter(Objects::nonNull).filter(Group::isLeafGroup).map(Group::getIndex).collect(Collectors.toSet());
    }

    private List<ConfiguredNode> getNodesInGroup(int groupIndex) {
        return this.groupForThisIndex(groupIndex).getNodes();
    }

    public static class Result {
        private final Action action;
        private final String reason;

        private Result(Action action, String reason) {
            this.action = action;
            this.reason = reason;
        }

        public static Result disallow(String reason) {
            return new Result(Action.DISALLOWED, reason);
        }

        public static Result allow() {
            return new Result(Action.ALLOWED, "Preconditions fulfilled and new state different");
        }

        public static Result alreadySet() {
            return new Result(Action.ALREADY_SET, "Basic preconditions fulfilled and new state is already effective");
        }

        public boolean allowed() {
            return this.action == Action.ALLOWED;
        }

        public boolean notAllowed() {
            return !this.allowed();
        }

        public boolean isAlreadySet() {
            return this.action == Action.ALREADY_SET;
        }

        public String reason() {
            return this.reason;
        }

        public String toString() {
            return "action " + this.action + ": " + this.reason;
        }

        public static enum Action {
            ALLOWED,
            ALREADY_SET,
            DISALLOWED;

        }
    }

    private record NodeDataMetrics(Optional<Metrics.Value> buckets, Optional<Metrics.Value> entries, Optional<Metrics.Value> docs) {
    }
}

