package com.xujl.task.group;

import com.xujl.task.Emitter;
import com.xujl.task.Task;
import com.xujl.task.delegate.GroupTaskDelegate;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

/**
 * 组任务执行回调任务
 */
public abstract class GroupResultTask extends Task<TaskConfig> {
    public final Object mLock = new Object();
    private Map<TaskConfig, GroupTask> mGroupTasks = new Hashtable<>();
    /**
     * 已完成的任务
     */
    private Map<TaskConfig, GroupTask> mCompleteGroupTasks = new Hashtable<>();
    private GroupTaskBuilder mBuilder;

    @Override
    public void run (Emitter<TaskConfig> emitter) throws Exception {
        super.run(emitter);
        while (!isDestroy()) {
            synchronized (mLock) {
                mLock.wait();
                //判断哪些任务执行完毕
                final Iterator<TaskConfig> iterator = mGroupTasks.keySet().iterator();
                while (iterator.hasNext()) {
                    final TaskConfig config = iterator.next();
                    GroupTask groupTask = mGroupTasks.get(config);
                    //判断单个任务的执行完成，并变更状态为完毕
                    if (config.getState() == GroupTaskState.STATE_EXECUTE) {
                        config.setState(GroupTaskState.STATE_COMPLETE);
                        mCompleteGroupTasks.put(config, groupTask);
                        iterator.remove();
                        emitter.objNext(mCompleteGroupTasks.get(config));
                        continue;
                    }
                    if (config.getState() == GroupTaskState.STATE_FAILED) {
                        if (mBuilder.getGroupStrategy() == GroupTaskStrategy.ALL_COMPLETE) {
                            //全部执行模式下，有失败任务，且单个任务没有配置其他模式时则继续等待
                            if (config.getStrategy() == GroupTaskStrategy.NONE) {
                                continue;
                            }
                            //单任务策略为退出模式时终止任务
                            if (config.getStrategy() == GroupTaskStrategy.ERROR_EXIT) {
                                cancelAllTask();
                                return;
                            }
                        }
                        //错误退出模式下，出现错误任务，直接取消所有任务
                        if (mBuilder.getGroupStrategy() == GroupTaskStrategy.ERROR_EXIT) {
                            cancelAllTask();
                            return;
                        }
                        mCompleteGroupTasks.put(config, groupTask);
                        iterator.remove();
                    }
                }
                //判断是否执行完毕所有任务
                if (mGroupTasks.size() == 0) {
                    if (mBuilder.getGroupStrategy() == GroupTaskStrategy.ERROR_EXIT && !isAllTaskComplete()) {
                        return;
                    }
                    HashMap<Object, Object> resultMap = new HashMap<>();
                    for (TaskConfig config : mCompleteGroupTasks.keySet()) {
                        resultMap.put(config.getResultTag(), config.getResultData());
                    }
                    emitter.objNext(resultMap, -1);
                    return;
                }
            }
        }
    }

    /**
     * 取消所有任务
     */
    private void cancelAllTask () {
        for (TaskConfig config : mGroupTasks.keySet()) {
            mGroupTasks.get(config).onDestroy();
        }
    }

    /**
     * 全部任务是否正常完成
     *
     * @return
     */
    private boolean isAllTaskComplete () {
        for (TaskConfig config : mCompleteGroupTasks.keySet()) {
            if (config.getState() == GroupTaskState.STATE_FAILED) {
                return false;
            }
        }
        return true;
    }

    @Override
    public void onObjNext (int code, Object obj) {
        super.onObjNext(code, obj);
        if (code == 0) {
            GroupTaskDelegate delegate = (GroupTaskDelegate) ((GroupTask) obj).getDelegate();
            oneTaskComplete(delegate.getGroupConfig().getResultTag(), delegate.getGroupConfig().getResultData());
        } else {
            allTaskComplete((HashMap<Object, Object>) obj);
        }
    }

    /**
     * 某个任务完成
     *
     * @param tag
     * @param result
     */
    public abstract void oneTaskComplete (Object tag, Object result);


    /**
     * 所有组任务完成
     *
     * @param resultMap <tag,data>
     */
    public abstract void allTaskComplete (HashMap<Object, Object> resultMap);

    public void setGroupTasks (Map<TaskConfig, GroupTask> groupTasks) {
        mGroupTasks = groupTasks;
    }

    public void setBuilder (GroupTaskBuilder builder) {
        mBuilder = builder;
    }

    public Object getLock () {
        return mLock;
    }
}
