package com.xujl.task.delegate;

import com.xujl.task.Emitter;
import com.xujl.task.IFinishNotify;
import com.xujl.task.ITaskFinish;
import com.xujl.task.RxLog;
import com.xujl.task.RxExecutor;
import com.xujl.task.RxLife;
import com.xujl.task.Task;
import com.xujl.task.consts.TaskRetry;

import java.util.Random;
import java.util.concurrent.Future;

/**
 * 抽象任务代理
 */
public class AbstractTaskDelegate<T> implements RxLife {
    protected static final String TAG = "Task";
    protected Emitter<T> mEmitter;
    protected Future<T> mFuture;
    protected boolean isDestroy;
    protected ITaskFinish mITaskFinish;
    protected IFinishNotify mFinishNotify;
    /**
     * 任务编号
     */
    protected long taskId;
    /**
     * 绑定生命周期时是否需要在
     * 生命周期结束后取消任务
     * 设置为true时（任务可能执行到一半就会结束
     * ，慎重设置！）
     */
    protected boolean needCancel;

    protected Task<T> mTask;
    /**
     * 失败重试最大次数
     */
    protected int retryMaxCount = TaskRetry.NONE;
    /**
     * 当前重试次数
     */
    protected int retryCount;


    public AbstractTaskDelegate (Task<T> task, boolean needCancel) {
        mTask = task;
        mEmitter = new Emitter<>(task);
        mTask.bindLife(this);
        this.needCancel = needCancel;
        taskId = System.currentTimeMillis() + new Random().nextInt(10000);
    }

    /**
     * 生命周期结束后，子线程任务依然会执行
     * 完毕，但不会再回调结果,如果需要及时停止
     * 子线程任务，请在子线程任务中根据方法isDestroy()
     * 加以判断
     */
    @Override
    public void onDestroy () {
        if (mFinishNotify != null) {
            mFinishNotify.onDestroy(mTask);
        }
        if (mEmitter == null) {
            return;
        }
        if (mFuture != null && needCancel) {
            mFuture.cancel(true);
        }
        recycle();
    }

    private void recycle () {
        isDestroy = true;
        mEmitter = null;
    }

    @Override
    public boolean isDestroy () {
        return isDestroy;
    }


    /**
     * 主线程回调
     *
     * @param e
     */
    public void onError (Exception e) {
        if (mITaskFinish != null) {
            mITaskFinish.onFinish(this);
        }
        onDestroy();
        if (retryMaxCount != TaskRetry.NONE) {
            if (retryMaxCount != TaskRetry.INFINITE && retryCount >= retryMaxCount) {
                RxLog.d(TAG, "超过最大重试次数！任务失败，结束");
                return;
            }
            retryCount++;
            RxLog.d(TAG, "任务失败，重试第" + retryCount + "次！");
            reset();
            RxExecutor.getInstance().executeTask(mTask);
        }
    }

    /**
     * 主线程回调
     */
    public void onFinished () {
        if (mITaskFinish != null) {
            mITaskFinish.onFinish(this);
        }
        onDestroy();
    }

    /**
     * 取消任务
     * 只能是停掉线程中有sleep,wait,join逻辑的线程，抛出一个InterruptException
     *
     * @param interrupt
     * @return
     */
    public boolean cancel (boolean interrupt) {
        isDestroy = true;
        if (mFuture == null) {
            return false;
        }
        return mFuture.cancel(interrupt);
    }

    public void reset () {
        mEmitter = new Emitter(mTask);
        mTask.bindLife(this);
        isDestroy = false;
    }

    public Future<T> getFuture () {
        return mFuture;
    }

    public void setFuture (Future<T> future) {
        mFuture = future;
    }

    public boolean isNeedCancel () {
        return needCancel;
    }

    public void setNeedCancel (boolean needCancel) {
        this.needCancel = needCancel;
    }

    @Override
    final public void setITaskFinish (ITaskFinish iTaskFinish) {
        mITaskFinish = iTaskFinish;
    }

    public Emitter getEmitter () {
        return mEmitter;
    }

    @Override
    public boolean equals (Object obj) {
        return super.equals(obj);
    }

    @Override
    public int hashCode () {
        return super.hashCode();
    }

    public long getTaskId () {
        return taskId;
    }

    public void setFinishNotify (IFinishNotify finishNotify) {
        mFinishNotify = finishNotify;
    }


    public int getRetryMaxCount () {
        return retryMaxCount;
    }

    public void setRetryMaxCount (int retryMaxCount) {
        this.retryMaxCount = retryMaxCount;
    }

    public boolean canRetry () {
        if (retryMaxCount == TaskRetry.INFINITE) {
            return true;
        }
        return retryMaxCount != TaskRetry.NONE && retryCount < retryMaxCount;
    }

    public int getRetryCount () {
        return retryCount;
    }
}
