package com.xujl.task;

import com.xujl.task.consts.TaskRetry;
import com.xujl.task.consts.TaskType;
import com.xujl.task.delegate.AbstractTaskDelegate;
import com.xujl.task.delegate.GroupTaskDelegate;
import com.xujl.task.delegate.NormalTaskDelegate;

import java.util.concurrent.Future;

public abstract class Task<T> implements RxLife {

    private AbstractTaskDelegate<T> mDelegate;
    /**
     * 任务类型
     */
    @TaskType.Enum
    private int mTaskType;

    public Task () {
        this(false, TaskType.TYPE_NORMAL);
    }

    public Task (boolean needCancel) {
        this(needCancel, TaskType.TYPE_NORMAL);
    }

    public Task (TaskBuilder builder) {
        this(builder.isNeedCancel(), builder.getTaskType(), builder.getRetryMaxCount());
    }

    public Task (boolean needCancel, @TaskType.Enum int type) {
        this(needCancel, type, TaskRetry.NONE);
    }

    public Task (boolean needCancel, @TaskType.Enum int type, int retryMaxCount) {
        mTaskType = type;
        if (type == TaskType.TYPE_NORMAL) {
            mDelegate = new NormalTaskDelegate<>(this, needCancel);
            mDelegate.setRetryMaxCount(retryMaxCount);
        } else if (type == TaskType.TYPE_GROUP) {
            mDelegate = new GroupTaskDelegate<>(this, needCancel);
            mDelegate.setRetryMaxCount(retryMaxCount);
        }
    }


    public void run (Emitter<T> emitter) throws Exception {

    }

    /**
     * 生命周期结束后，子线程任务依然会执行
     * 完毕，但不会再回调结果,如果需要及时停止
     * 子线程任务，请在子线程任务中根据方法isDestroy()
     * 加以判断
     */
    @Override
    public void onDestroy () {
        mDelegate.onDestroy();
    }


    @Override
    public boolean isDestroy () {
        return mDelegate.isDestroy();
    }

    /**
     * 仅仅执行ui任务，
     * 主线程回调
     */
    public void onlyRunUiTask () {

    }

    /**
     * 当前线程回调，取决于开启任务的线程
     *
     * @param rxLife
     */
    public void bindLife (RxLife rxLife) {

    }

    /**
     * 主线程回调
     *
     * @param data
     */
    public void onNext (T data) {

    }

    /**
     * @param code 识别码
     * @param obj
     */
    public void onObjNext (int code, Object obj) {

    }

    /**
     * 主线程回调
     *
     * @param e
     */
    public void onError (Exception e) {
        mDelegate.onError(e);
    }

    /**
     * 主线程回调
     */
    public void onFinished () {
        mDelegate.onFinished();
    }

    /**
     * 取消任务
     * 只能是停掉线程中有sleep,wait,join逻辑的线程，抛出一个InterruptException
     *
     * @param interrupt
     * @return
     */
    public boolean cancel (boolean interrupt) {
        return mDelegate.cancel(interrupt);
    }

    public void reset () {
        mDelegate.reset();
    }

    public Future<T> getFuture () {
        return mDelegate.getFuture();
    }

    public void setFuture (Future<T> future) {
        mDelegate.setFuture(future);
    }

    public boolean isNeedCancel () {
        return mDelegate.isNeedCancel();
    }

    public void setNeedCancel (boolean needCancel) {
        mDelegate.setNeedCancel(needCancel);
    }

    @Override
    final public void setITaskFinish (ITaskFinish iTaskFinish) {
        mDelegate.setITaskFinish(iTaskFinish);
    }

    Emitter getEmitter () {
        return mDelegate.getEmitter();
    }

    @Override
    public boolean equals (Object obj) {
        return super.equals(obj);
    }

    @Override
    public int hashCode () {
        return super.hashCode();
    }

    public long getTaskId () {
        return mDelegate.getTaskId();
    }

    void setFinishNotify (IFinishNotify finishNotify) {
        mDelegate.setFinishNotify(finishNotify);
    }

    public int getTaskType () {
        return mTaskType;
    }

    public AbstractTaskDelegate<T> getDelegate () {
        return mDelegate;
    }

    public int getRetryMaxCount () {
        return mDelegate.getRetryMaxCount();
    }

    public void setRetryMaxCount (int retryMaxCount) {
        mDelegate.setRetryMaxCount(retryMaxCount);
    }

    /**
     * 任务是否还可以重试
     *
     * @return
     */
    public boolean canRetry () {
        return mDelegate.canRetry();
    }

    /**
     * 获取当前重试次数
     * @return
     */
    public int getRetryCount () {
        return mDelegate.getRetryCount();
    }
}
