/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.webauthn4j.data;

import java.io.Serializable;
import java.util.Objects;

/**
 * The PublicKeyCredentialEntity describes a user account, or a WebAuthn Relying Party,
 * which a public key credential is associated with or scoped to, respectively.
 *
 * @see <a href="https://www.w3.org/TR/webauthn-1/#dictionary-pkcredentialentity">
 * §5.4.1. Public Key Entity Description (dictionary PublicKeyCredentialEntity)</a>
 */
public abstract class PublicKeyCredentialEntity implements Serializable {

    // ~ Instance fields
    // ================================================================================================

    private String name;
    private String icon;

    // ~ Constructor
    // ========================================================================================================

    public PublicKeyCredentialEntity(String name, String icon) {
        this.name = name;
        this.icon = icon;
    }

    public PublicKeyCredentialEntity(String name) {
        this.name = name;
        this.icon = null;
    }

    public PublicKeyCredentialEntity() {
    }

    public String getName() {
        return name;
    }

    public String getIcon() {
        return icon;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        PublicKeyCredentialEntity that = (PublicKeyCredentialEntity) o;
        return Objects.equals(name, that.name) &&
                Objects.equals(icon, that.icon);
    }

    @Override
    public int hashCode() {

        return Objects.hash(name, icon);
    }
}
