package com.voxeet.sdk.services.telemetry.device;

import android.os.Build;
import android.support.annotation.NonNull;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.voxeet.audio2.devices.MediaDevice;
import com.voxeet.sdk.services.telemetry.SdkEnvironment;
import com.voxeet.sdk.services.telemetry.codec.Codec;
import com.voxeet.sdk.services.telemetry.network.CarrierInfo;
import com.voxeet.sdk.services.telemetry.network.wifi.Wifi;
import com.voxeet.sdk.utils.Map;
import com.voxeet.sdk.utils.Opt;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.ConcurrentHashMap;

@JsonIgnoreProperties(ignoreUnknown = true)
public class DeviceInformation {

    public String name;

    public String os;

    public String timeZone;

    public String sdkPlatform;

    public String sdkVersion;

    public String uxkitVersion;

    public Hardware hardware;

    // OPTIONALS INFORMATION
    public HashMap<String, String> versions;
    public Device extended;

    private DeviceInformation(@NonNull HardwareInfo info) {
        name = info.deviceName();
        os = info.os();
        timeZone = info.tz();

        hardware = new Hardware(info.cpu(),
                "" + info.ram(),
                String.format(Locale.getDefault(), "%dx%d", info.width(), info.height())
        );
    }

    public DeviceInformation(@NonNull HardwareInfo info, boolean full, @NonNull ConcurrentHashMap<SdkEnvironment, String> environment, List<MediaDevice> mediaDevices) {
        this(info);
        hardware.codecs = Codec.list();
        if (full) {
            extended = new Device(Codec.list(), info, mediaDevices);

            versions = new HashMap<>();
            for (SdkEnvironment key : environment.keySet()) {
                versions.put(key.name, environment.get(key));
            }
        }

        Opt<ConcurrentHashMap<SdkEnvironment, String>> opt = Opt.of(environment);


        sdkVersion = opt.then(v -> v.get(SdkEnvironment.SDK)).or("");

        sdkPlatform = opt.then(v -> v.get(SdkEnvironment.CORDOVA))
                .or(opt.then(v -> v.get(SdkEnvironment.REACT_NATIVE))
                        .or(sdkVersion));

        uxkitVersion = opt.then(v -> v.get(SdkEnvironment.UXKIT)).orNull();
    }

    public DeviceInformation(@NonNull HardwareInfo info, boolean full, @NonNull ConcurrentHashMap<SdkEnvironment, String> environment, List<MediaDevice> mediaDevices, List<CarrierInfo> carriers, Wifi wifi) {
        this(info, full, environment, mediaDevices);

        if (null != extended) {
            extended.networks = new Networks(carriers, wifi);
        }
    }

    public class Networks {
        public List<CarrierInfo> carriers;
        public Wifi wifi;

        public Networks(List<CarrierInfo> carriers, Wifi wifi) {
            this.carriers = carriers;
            this.wifi = wifi;
        }
    }

    public class Hardware {
        public String cpu;

        public String ram;

        public String screen;

        public List<Codec> codecs;

        public Hardware(String cpu, String ram, String screen) {
            this.cpu = cpu;
            this.ram = ram;
            this.screen = screen;
        }
    }

    public class Device {
        public List<MediaDeviceJson> peripherals;
        public String model;
        public String manufacturer;
        public String os = "android";
        public long version;
        public String brand;
        public String board;
        public List<Codec> codecs;
        public Networks networks;
        public Screen screen;

        public Device() {
            version = Build.VERSION.SDK_INT;
            brand = Build.BRAND.toLowerCase();
            board = Build.BOARD;
            manufacturer = Build.MANUFACTURER;
            model = Build.MODEL;
        }

        public Device(List<Codec> codecs, HardwareInfo info, List<MediaDevice> mediaDevices) {
            this();
            this.codecs = codecs;
            screen = new Screen(info);

            this.peripherals = Map.map(mediaDevices, MediaDeviceJson::new);
        }

        public class MediaDeviceJson {
            public String name;
            public String type;

            public MediaDeviceJson(MediaDevice device) {
                name = device.id();
                type = device.deviceType().toString().toLowerCase();
            }
        }

        public class Screen {

            public long width;
            public long height;

            public Screen(HardwareInfo info) {
                width = info.width();
                height = info.height();
            }
        }
    }
}
