package com.voxeet.sdk.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.VoxeetSDK;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.PromiseSolver;
import com.voxeet.promise.solve.Solver;
import com.voxeet.promise.solve.ThenPromise;
import com.voxeet.sdk.events.error.PermissionRefusedEvent;
import com.voxeet.sdk.events.promises.PromisePermissionRefusedEventException;
import com.voxeet.sdk.events.v2.VideoStateEvent;
import com.voxeet.sdk.json.StartVideoResponse;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.media.camera.CameraContext;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.v2.ServerErrorOrigin;
import com.voxeet.sdk.network.endpoints.IRestApiVideo;
import com.voxeet.sdk.services.AbstractPromiseable;
import com.voxeet.sdk.services.ConferenceService;
import com.voxeet.sdk.services.MediaDeviceService;
import com.voxeet.sdk.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.services.conference.information.ConferenceInformationHolder;
import com.voxeet.sdk.services.conference.information.ConferenceParticipantType;
import com.voxeet.sdk.services.media.VideoState;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.utils.Opt;
import com.voxeet.sdk.utils.Validate;

import org.greenrobot.eventbus.EventBus;

import retrofit2.Call;

public class StartVideoPromise extends AbstractPromiseable<Boolean, IRestApiVideo> {
    private final static String TAG = StartVideoPromise.class.getSimpleName();
    private final boolean isDefaultFrontFacing;
    private final ConferenceInformationHolder conferenceInformationHolder;

    public StartVideoPromise(@NonNull ConferenceService parent,
                             @NonNull MediaDeviceService mediaDeviceService,
                             @NonNull IRestApiVideo provider,
                             @Nullable ConferenceInformation information,
                             @NonNull EventBus eventBus,
                             boolean isDefaultFrontFacing,
                             ConferenceInformationHolder conferenceInformationHolder) {
        super(parent, mediaDeviceService, provider, information, eventBus);

        this.isDefaultFrontFacing = isDefaultFrontFacing;
        this.conferenceInformationHolder = conferenceInformationHolder;
    }

    @NonNull
    @Override
    public Promise<Boolean> createPromise() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                final ConferenceInformation information = getInformation();

                MediaSDK media = getMedia();

                CameraContext provider = getMediaService().getCameraContext();
                provider.setDefaultCameraFront(isDefaultFrontFacing);

                log("starting video");

                if (null == media) {
                    Log.d(TAG, "startVideo: media is null");
                    Promise.reject(solver, new MediaEngineException("Media is null, invalid state"));
                    return;
                }

                if (!Validate.hasCameraPermissions(getParent().getContext())) {
                    PermissionRefusedEvent event = new PermissionRefusedEvent(PermissionRefusedEvent.Permission.CAMERA);
                    getEventBus().post(event);
                    Promise.reject(solver, new PromisePermissionRefusedEventException(event));
                    return;
                }

                if (null == information) {
                    Log.d(TAG, "startVideo: information is null");
                    Promise.reject(solver, new MediaEngineException("ConferenceInformation is null, invalid state"));
                    return;
                }

                if (!VideoState.STOPPED.equals(information.getVideoState())) {
                    log("starting video failed : maybe already in some...");
                    Promise.reject(solver, new IllegalStateException("can not start video, pending ?"));
                    return;
                }
                if (ConferenceParticipantType.LISTENER.equals(information.getConferenceParticipantType())) {
                    Log.d(TAG, "startVideo: you are a listener, you can't start the video");
                    solver.resolve(false);
                    return;
                }

                conferenceInformationHolder.setVideoState(information, VideoState.STARTING);
                getEventBus().post(new VideoStateEvent(information.getConference(), VideoState.STARTING));

                String defaultCameraName = isDefaultFrontFacing ? provider.getFrontCameraName() : provider.getBackCameraName();

                if (null == defaultCameraName) {
                    Promise.reject(solver, new MediaEngineException("DefaultCamera is null, invalid state"));
                    return;
                }

                information.setRequestedOwnVideo(true);

                media.startVideo(defaultCameraName);

                String id = Opt.of(getInformation()).then(ConferenceInformation::getConference).then(Conference::getId).or("");

                final Call<StartVideoResponse> startVideo = getApiRestCall().startVideo(id, VoxeetSDK.session().getParticipantId());
                HttpHelper.promise(startVideo, ServerErrorOrigin.START_VIDEO)
                        .then((ThenPromise<HttpHelper.HttpAnswer<StartVideoResponse>, Boolean>) answer -> {
                            StartVideoResponse object = answer.object;
                            //remove start video since it is not to be used
                            //media.startVideo(mDefaultCamera);
                            conferenceInformationHolder.setVideoState(information, VideoState.STARTING);

                            information.setRequestedOwnVideo(true);

                            return createVideoAnswer(object.participantId, object.description, object.candidates);
                        })
                        .then(result -> {
                            conferenceInformationHolder.setVideoState(information, VideoState.STARTED);
                            getEventBus().post(new VideoStateEvent(information.getConference(), VideoState.STARTED));
                            solver.resolve(result);
                        })
                        .error(error -> {
                            conferenceInformationHolder.setVideoState(information, VideoState.STOPPED);
                            getEventBus().post(new VideoStateEvent(information.getConference(), VideoState.STOPPED));
                            media.stopVideo();
                            information.setRequestedOwnVideo(false);
                            solver.reject(error);
                        });
                //explicitly keep the list of the different exception
                //this methods throws
            }
        });
    }
}
