package com.voxeet.sdk.services.builders;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.json.internal.MetadataHolder;
import com.voxeet.sdk.json.internal.ParamsHolder;
import com.voxeet.sdk.utils.Annotate;

/**
 * The ConferenceCreateOptions model contains information about the conference that will be created, such as the [conference alias](/documentation/sdk/reference/android/models/conferencecreateoptions#getconferencealias), [metadata](/documentation/sdk/reference/android/models/conferencecreateoptions#getmetadataholder), and [parameters](/documentation/sdk/reference/android/models/conferencecreateoptions#getparamsholder).
 */
@Annotate
public class ConferenceCreateOptions {
    private String conferenceAlias;
    private MetadataHolder metadataHolder;
    private ParamsHolder paramsHolder;

    private ConferenceCreateOptions() {

    }

    /**
     * Gets the alias for a new conference.
     * @return the valid value that informs about the alias. Otherwise, a null value.
     */
    @Nullable
    public String getConferenceAlias() {
        return conferenceAlias;
    }

    /**
     * Gets conference metadata.
     * @return the holder instance or a null value.
     */
    @Nullable
    public MetadataHolder getMetadataHolder() {
        return metadataHolder;
    }

    /**
     * Gets conference parameters.
     * @return the holder instance or a null value.
     */
    @Nullable
    public ParamsHolder getParamsHolder() {
        return paramsHolder;
    }

    /**
     * The ConferenceCreateOptions.Builder model is responsible for building conferences. It creates [a new builder](/documentation/sdk/reference/android/models/conferencecreateoptions.builder#builder) and [the ConferenceCreateOptions instance](/documentation/sdk/reference/android/models/conferencecreateoptions.builder#build). It also sets [conference alias](/documentation/sdk/reference/android/models/conferencecreateoptions.builder#setconferencealias), [metadata](/documentation/sdk/reference/android/models/conferencecreateoptions.builder#setmetadataholder), and [parameters](/documentation/sdk/reference/android/models/conferencecreateoptions.builder#setparamsholder).
     */
    @Annotate
    public static class Builder {
        private ConferenceCreateOptions conferenceJoinInformation;

        /**
         * Creates a new builder.
         */
        public Builder() {
            conferenceJoinInformation = new ConferenceCreateOptions();
        }

        /**
         * Sets the conference alias. It is recommended to use a specific and unique name for new conferences.
         *
         * @param conferenceAlias conference name, preferably non-null
         * @return the builder instance.
         */
        public Builder setConferenceAlias(@Nullable String conferenceAlias) {
            conferenceJoinInformation.conferenceAlias = conferenceAlias;
            return this;
        }

        /**
         * Adds metadata to the conference.
         * @param metadataHolder reference to the metadata
         * @return the builder instance.
         */
        public Builder setMetadataHolder(@Nullable MetadataHolder metadataHolder) {
            conferenceJoinInformation.metadataHolder = metadataHolder;
            return this;
        }

        /**
         * Adds parameters to the conference.
         *
         * @param paramsHolder reference to parameters
         * @return the builder instance.
         */
        public Builder setParamsHolder(@Nullable ParamsHolder paramsHolder) {
            conferenceJoinInformation.paramsHolder = paramsHolder;
            return this;
        }

        /**
         * Constructs the [ConferenceCreateOptions](/documentation/sdk/reference/android/models/conferencecreateoptions) instance for creating conferences.
         * @return the builder instance.
         */
        @NonNull
        public ConferenceCreateOptions build() {
            return conferenceJoinInformation;
        }

    }
}
