package com.voxeet.sdk.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.promise.Promise;
import com.voxeet.promise.solve.Solver;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.json.FileConverted;
import com.voxeet.sdk.json.FilePresentationStarted;
import com.voxeet.sdk.json.FilePresentationStopped;
import com.voxeet.sdk.json.FilePresentationUpdated;
import com.voxeet.sdk.models.v1.File;
import com.voxeet.sdk.models.v1.FilePresentationConverted;
import com.voxeet.sdk.models.v2.ServerErrorOrigin;
import com.voxeet.sdk.network.endpoints.IRestApiFilePresentation;
import com.voxeet.sdk.services.abstracts.AbstractPresentationService;
import com.voxeet.sdk.services.presentation.PresentationState;
import com.voxeet.sdk.services.presentation.file.FilePresentation;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Opt;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.UUID;

import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import retrofit2.Response;

/**
 * The FilePresentationService allows an application to present a file during the conference. The Voxeet service converts the participant provided file into multiple pages (as images) accessible through the [getImage](/documentation/sdk/reference/android/filepresentation#getimage) method.
 *
 * **Typical application workflow:**
 *
 * **1.** The application calls the [convertFile](/documentation/sdk/reference/android/filepresentation#convertfile) method to upload and convert a file.
 *
 * **2.** The application can display the converted file locally by retrieving individual images through the [getImage](/documentation/sdk/reference/android/filepresentation#getimage) method.
 *
 * **3.** The application then starts presenting the file by calling the [start](/documentation/sdk/reference/android/filepresentation#start) method.
 *
 * **4.** The application is responsible for coordinating the page flip between local and presented files. It needs to call the [update](/documentation/sdk/reference/android/filepresentation#update) method to inform the service to send the updated page number to the participants.
 *
 * **5.** The application calls the [stop](/documentation/sdk/reference/android/filepresentation#stop) method to end the file presentation.
 *
 * **6.** The application may choose to call the [getThumbnail](/documentation/sdk/reference/android/filepresentation#getthumbnail) method to obtain thumbnail images of the file to implement a carousel control for the presenting participant to flip pages locally. This way of flipping pages can be also used for standard size images by using the [getImage](/documentation/sdk/reference/android/filepresentation#getimage) method.
 *
 * The application receives information about changes during the file presentation through the [FilePresentation](/documentation/sdk/reference/android/filepresentation#filepresentation) event.
 */
@Annotate
public class FilePresentationService extends AbstractPresentationService<FilePresentation> {
    private static final String TAG = FilePresentationService.class.getSimpleName();

    private HashMap<String, Solver<FilePresentation>> mCacheSolvers;
    private HashMap<String, Solver<FilePresentation>> mCacheStartedSolvers;
    private HashMap<String, Solver<FilePresentation>> mCacheStoppedSolvers;
    private HashMap<String, Solver<FilePresentation>> mCacheUpdatedSolvers;

    /**
     * Instantiates a new FilePresentation service.
     *
     * @param instance the parent instance
     */
    @NoDocumentation
    public FilePresentationService(@NonNull SdkEnvironmentHolder instance) {
        super(instance);

        mCacheSolvers = new HashMap<>();
        mCacheStartedSolvers = new HashMap<>();
        mCacheStoppedSolvers = new HashMap<>();
        mCacheUpdatedSolvers = new HashMap<>();
        registerEventBus();
    }

    @NoDocumentation
    void onInternalServiceEvent(FileConverted event) {
        List<File> list = event.getFiles();

        for (File file : list) {
            String key = findSolverFor(file.getName());
            if (null != key) {
                onInternalServiceEvent(new FilePresentationConverted(
                        file.getName(),
                        file.getFileId(),
                        file.getSize(),
                        file.getNbImageConverted()));
            }
        }
    }

    @NoDocumentation
    void onInternalServiceEvent(FilePresentationConverted event) {
        FilePresentation information = getPresentationInformation(event.fileId);
        presentations.add(information);

        information.state = PresentationState.CONVERTED;
        information.page = 0;
        information.nbPage = event.nbImageConverted;
        getEventBus().post(event);
        tryUnlock(information.key, information, mCacheSolvers);
    }

    @NoDocumentation
    void onInternalServiceEvent(FilePresentationStarted event) {
        FilePresentation information = getPresentationInformation(event.fileId);
        presentations.add(information);

        information.state = PresentationState.STARTED;
        information.page = event.position;
        information.nbPage = event.imageCount;
        getEventBus().post(event);
        tryUnlock(information.key, information, mCacheSolvers);
    }

    @NoDocumentation
    void onInternalServiceEvent(FilePresentationUpdated event) {
        FilePresentation information = getPresentationInformation(event.fileId);
        presentations.add(information);

        information.state = PresentationState.SEEK;
        information.page = event.position;
        getEventBus().post(event);
        tryUnlock(information.key, information, mCacheSolvers);
    }

    @NoDocumentation
    void onInternalServiceEvent(FilePresentationStopped event) {
        FilePresentation information = getPresentationInformation(event.fileId);
        presentations.add(information);

        information.state = PresentationState.STOP;
        getEventBus().post(event);
        tryUnlock(information.key, information, mCacheSolvers);
    }

    /**
     * Displays the converted file locally by retrieving URLs of individual images.
     *
     * @param fileId     the ID of the remote file
     * @param pageNumber the requested page number
     * @return the formatted string.
     */
    @NonNull
    public String getImage(String fileId, int pageNumber) {
        return String.format(Locale.getDefault(), "%s/v1/files/%s/converted/%d?token=%s",
                getURLRoot(), fileId, pageNumber, getInternalJwtToken());
    }

    /**
     * Obtains URLs of thumbnail images of the file to implement a carousel control for the presenting participant to flip pages locally.
     *
     * @param fileId     the ID of the remote file
     * @param pageNumber the requested page number
     * @return the formatted string.
     */
    @NonNull
    public String getThumbnail(String fileId, int pageNumber) {
        return String.format(Locale.getDefault(), "%s/v1/files/%s/converted/%d/thumbnail?token=%s",
                getURLRoot(), fileId, pageNumber, getInternalJwtToken());
    }

    /**
     * Starts uploading and converting the local file and waits for its management on the server side.
     *
     * @param file valid file to uploa
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<FilePresentation> convertFile(@NonNull final java.io.File file) {
        return new Promise<>(solver -> {
            final String uuid = UUID.randomUUID().toString();

            String appended_name = uuid + file.getName();
            RequestBody requestFile = RequestBody.create(MediaType.parse("multipart/form-data"), file);

            // MultipartBody.Part is used to send also the actual file name
            MultipartBody.Part body = MultipartBody.Part.createFormData("file", appended_name, requestFile);


            // finally, execute the request
            HttpHelper.promise(getService(IRestApiFilePresentation.class).convertFile(requestFile, body), ServerErrorOrigin.CONVERT_FILE)
                    .then(result -> {
                        Response<ResponseBody> response = Opt.of(result).then(c -> c.response).orNull();
                        //if response is successful will wait for conversion result, so we put into cache...
                        if (Opt.of(response).then(Response::isSuccessful).or(false)) {
                            mCacheSolvers.put(uuid, solver);
                        } else {
                            solver.reject(HttpException.throwResponse(response));
                        }
                    }).error(solver::reject);
        });
    }

    /**
     * Starts presenting the converted file.
     *
     * @param body valid descriptor of the file
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<FilePresentation> start(@NonNull final FilePresentationConverted body) {
        return start(body, 0);
    }


    /**
     * Starts presenting the specific page of the converted file.
     *
     * @param body     valid descriptor of the file
     * @param position the starting position
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<FilePresentation> start(@NonNull final FilePresentationConverted body,
                                           final int position) {
        return new Promise<>(solver -> {
            IRestApiFilePresentation.FilePresentationId body_sent = new IRestApiFilePresentation.FilePresentationId(body.fileId, body.name, position, body.nbImageConverted);

            consumeInternalCall(solver, body.fileId, mCacheStartedSolvers,
                    internalCall(getService(IRestApiFilePresentation.class).startFilePresentation(getConferenceId(), body_sent)));
        });
    }


    /**
     * Stops presenting the file.
     *
     * @param fileId ID of the file
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<FilePresentation> stop(@NonNull final String fileId) {
        return new Promise<>(solver -> {
            IRestApiFilePresentation.FilePresentationId body_sent = new IRestApiFilePresentation.FilePresentationId(fileId);

            consumeInternalCall(solver, fileId, mCacheStoppedSolvers,
                    internalCall(getService(IRestApiFilePresentation.class).startFilePresentation(getConferenceId(), body_sent)));
        });
    }

    /**
     * Changes the position (page of the presented file) to the specific one in the current presentation.
     *
     * @param fileId   ID of the file
     * @param position the new position (page of the presented file)
     * @return the promise to resolve.
     */
    @NonNull
    public Promise<FilePresentation> update(@NonNull final String fileId,
                                            final int position) {
        return new Promise<>(solver -> {

            IRestApiFilePresentation.FilePresentationId body_sent = new IRestApiFilePresentation.FilePresentationId(fileId, position);

            consumeInternalCall(solver, fileId, mCacheUpdatedSolvers,
                    internalCall(getService(IRestApiFilePresentation.class).startFilePresentation(getConferenceId(), body_sent)));
        });
    }

    @Nullable
    private String findSolverFor(@NonNull String name) {
        for (String value : mCacheSolvers.keySet())
            if (name.indexOf(value) == 0) return value;
        return null;
    }

    @NonNull
    private FilePresentation getPresentationInformation(@NonNull String key) {
        for (FilePresentation information : presentations) {
            if (key.equals(information.key)) return information;
        }
        return new FilePresentation(key, "");
    }
}