package com.voxeet.sdk.models;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.services.SessionService;
import com.voxeet.sdk.json.UserInfo;
import com.voxeet.sdk.models.v1.ConferenceInfos;
import com.voxeet.sdk.models.v1.ConferenceUser;
import com.voxeet.sdk.models.v1.ConferenceUserStatus;
import com.voxeet.sdk.models.v1.Participant;
import com.voxeet.sdk.models.v1.RecordingStatus;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.Date;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Representation of a Conference from a SDK Point of View. Holds the various information regarding the conference
 * that can be reflected from the server's own point of view
 */
@Annotate
public class Conference {

    private static final String TAG = "Conference";
    @NonNull
    private String id;

    @Nullable
    private String alias;

    @NonNull
    private CopyOnWriteArrayList<User> users;

    @Nullable
    private User mixer;

    @Nullable
    private ConferenceInfos conferenceInfos;

    @Nullable
    private RecordingInformation recordingInformation;

    @NoDocumentation
    public Conference() {
        users = new CopyOnWriteArrayList<>();
        id = "";
    }

    @NoDocumentation
    public Conference(com.voxeet.sdk.models.v1.Conference fromConference) {
        this();

        updateUsers(fromConference.getConferenceUsers());
        id = fromConference.getConferenceId();
        alias = fromConference.getConferenceAlias();

    }

    /**
     * Get the conference id
     *
     * @return the current
     */
    @NonNull
    public String getId() {
        return id;
    }

    /**
     * Get the conference alias set by developers
     *
     * @return the conference alias
     */
    @Nullable
    public String getAlias() {
        return alias;
    }

    /**
     * Get the reference to the users in the conference
     *
     * @return the direct reference to the array of users
     */
    @NonNull
    public CopyOnWriteArrayList<User> getUsers() {
        return users;
    }

    /**
     * Check if the current conference has any users in the given state. Practical use to help getting a fast information about :
     * - on air users
     * - left users
     * - invited
     *
     * @param status the status to check upon
     * @param local will the search include the local user
     * @return at least one user exists with the corresponding state
     */
    public boolean hasAny(ConferenceUserStatus status, boolean local) {
        String id = VoxeetSdk.session().getUserId();
        if(null == id) id = "";

        for (User user : users) {
            if (null != user && status.equals(user.getStatus()) && !id.equals(user.getId())) return true;
        }
        return false;
    }

    /**
     * Update a given user in the conference
     * to be used by the SDK
     *
     * @param user a raw user to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateUsers(ConferenceUser user) {
        User internalUser = getInternalUserInConference(user);
        if (null == internalUser) {
            Log.d(TAG, "updateUsers: create User from ConferenceUser := " + new User(user));
            addUser(new User(user));
        } else {
            internalUser.updateStatus(user.getConferenceStatus());
        }

        return this;
    }

    /**
     * Update a given user in the conference
     * to be used by the SDK
     *
     * @param participant a raw participant to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateUser(@NonNull Participant participant) {
        User internalUser = getInternalUserInConference(participant);

        if (null == internalUser) {
            Log.d(TAG, "updateUsers: create User from Participant := " + new User(participant));
            String externalId = participant.getExternalId();
            User existing = findUserByExternalId(externalId);
            if (null != existing) {
                Log.d(TAG, "updateUser: existing user, updating hi·er");
                existing.updateIfNeeded(participant.getName(), participant.getAvatarUrl());
            } else {
                Log.d(TAG, "updateUser: new user, adding hi·er");
                addUser(new User(participant));
            }
        } else {
            internalUser.updateStatus(ConferenceUserStatus.fromString(participant.getStatus()));
        }

        return this;
    }

    /**
     * Update a given user in the conference
     * to be used by the SDK
     *
     * @param user a raw user to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateUser(User user) {
        User internalUser = getInternalUserInConference(user);
        if (null == internalUser) {
            Log.d("ConferenceService", "updateUsers: create User from User := " + user);
            addUser(user);
        } else {
            internalUser.updateStatus(user.getStatus());
        }

        return this;
    }

    /**
     * Update a list of users in the conference
     * to be used by the SDK
     *
     * @param users a list of raw user to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateUsers(List<ConferenceUser> users) {
        try {
            for (ConferenceUser user : users) updateUsers(user);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return this;
    }

    /**
     * Update a given user in the conference
     * to be used by the SDK
     *
     * @param participants a list of raw user to add or update
     * @return the current reference to the conference
     */
    @NoDocumentation
    @NonNull
    public Conference updateParticipants(List<Participant> participants) {
        try {
            for (Participant participant : participants) updateUser(participant);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return this;
    }

    @Nullable
    private User getInternalUserInConference(ConferenceUser user) {
        try {
            int index = users.indexOf(user);
            if (index >= 0) return users.get(index);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Nullable
    private User getInternalUserInConference(User user) {
        try {
            int index = users.indexOf(user);
            if (index >= 0) return users.get(index);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Nullable
    private User getInternalUserInConference(Participant user) {
        try {
            int index = users.indexOf(user);
            if (index >= 0) return users.get(index);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Set the current conference id
     * to be used by the SDK
     *
     * @param id
     * @return the current reference to the conference
     */
    @NoDocumentation
    public Conference setConferenceId(@NonNull String id) {
        this.id = id;
        return this;
    }

    @NoDocumentation
    public Conference setConferenceAlias(String alias) {
        this.alias = alias;
        return this;
    }

    /**
     * Check if the current Conference has a Mixer user currently connected
     *
     * useful to filter out this specific user into the UI or add a label for instance
     *
     * Possible improvement is to return true if a Mixer user is known by the conference
     *
     * @return the existence of an on air mixer user
     */
    public boolean hasMixer() {
        return null != mixer && ConferenceUserStatus.ON_AIR.equals(mixer.getStatus());
    }

    private boolean isMixer(@NonNull User user) {
        UserInfo userInfo = user.getUserInfo();
        String userName = null != userInfo ? userInfo.getName() : null;

        return null != userName && "Mixer".equalsIgnoreCase(userName.trim());
    }

    /**
     * Filter the users to get the one with the corresponding id or null if none corresponds.
     * The id is the one provided by Voxeet. Not the externalId set and known by developers
     *
     * @param userId a valid id to check
     * @return the instance of such user or null
     */
    @Nullable
    public User findUserById(@Nullable String userId) {
        if (null == userId) return null;
        for (User user : users) {
            if (userId.equals(user.getId())) return user;
        }

        SessionService sessionService = VoxeetSdk.session();
        if (null == sessionService) return null;

        String localUserId = sessionService.getUserId();
        if (null != localUserId && localUserId.equals(userId)) {
            Log.d(TAG, "findUserById: user not found but local one, creating a TEMP ONE ONLY");
            return sessionService.getUser();
        }
        return null;
    }

    /**
     * Filter the users to get the one with the corresponding externalId or null if none corresponds.
     * The externalId is the one provided by the developers. Not the id known and set by Voxeet.
     *
     * @param externalId a valid externalId to check
     * @return the instance of such user or null
     */
    @Nullable
    public User findUserByExternalId(@Nullable String externalId) {
        if (null == externalId) return null;
        for (User user : users) {
            UserInfo userInfo = user.getUserInfo();
            if (null != userInfo && externalId.equals(userInfo.getExternalId())) return user;
        }
        return null;
    }

    @NoDocumentation
    public void setConferenceInfos(@NonNull ConferenceInfos conferenceInfos) {
        this.conferenceInfos = conferenceInfos;
    }

    /**
     * Get the raw information about the conference
     * @return the information or null if non existent
     */
    @Nullable
    public ConferenceInfos getConferenceInfos() {
        return conferenceInfos;
    }

    /**
     * Set the current recording information state
     * to be used internally by the SDK
     *
     * @param recordingInformation the new information
     */
    @NoDocumentation
    public void setRecordingInformation(@Nullable RecordingInformation recordingInformation) {
        this.recordingInformation = recordingInformation;
    }

    /**
     * Get the current conference recording information and state
     *
     * @return a reference to the recording information
     */
    @Nullable
    public RecordingInformation getRecordingInformation() {
        return recordingInformation;
    }

    /**
     * Add a given user to the list of users
     * <p>
     * Warning : this does not manage the case where multiple mixers could be added - bug?
     * or even mix with how developers would create users... mixing user with the mixer
     *
     * @param user a newly instance to add
     */
    private void addUser(@NonNull User user) {
        if (isMixer(user)) mixer = user;
        else users.add(user);
    }

    public static class RecordingInformation {

        private Date startRecordTimestamp;
        private RecordingStatus recordingStatus;
        private String recordingUser;

        public void setStartRecordTimestamp(Date startRecordTimestamp) {
            this.startRecordTimestamp = startRecordTimestamp;
        }

        public Date getStartRecordTimestamp() {
            return startRecordTimestamp;
        }

        public void setRecordingStatus(RecordingStatus recordingStatus) {
            this.recordingStatus = recordingStatus;
        }

        public RecordingStatus getRecordingStatus() {
            return recordingStatus;
        }

        public void setRecordingUser(String recordingUser) {
            this.recordingUser = recordingUser;
        }

        public String getRecordingUser() {
            return recordingUser;
        }
    }
}
