package com.voxeet.sdk.core.services.media;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.android.media.MediaStream;
import com.voxeet.sdk.media.MediaSDK;

import org.webrtc.VideoFrame;
import org.webrtc.VideoRenderer;
import org.webrtc.VideoSink;

import java.lang.ref.WeakReference;
import java.util.ArrayList;

public class VideoSinkHolder {
    private MediaSDK media;
    @NonNull
    private WeakReference<VideoRenderer.Callbacks> holder;

    private ArrayList<MediaStreamListener> mediaStreamListeners;

    private VideoSinkHolder() {
        mediaStreamListeners = new ArrayList<>();
    }

    public VideoSinkHolder(MediaSDK media, @NonNull VideoRenderer.Callbacks videoSink) {
        this();
        this.media = media;
        holder = new WeakReference<>(videoSink);
    }

    public boolean hasVideoSink() {
        return holder.get() != null;
    }

    public void clear() {
        unattach();
        media = null;
    }

    @Nullable
    public MediaStream getMediaStream() {
        for (MediaStreamListener listener : mediaStreamListeners) {
            if (listener.active) return listener.mediaStream;
        }
        return null;
    }

    public void attach(@NonNull MediaStream stream) {
        VideoRenderer.Callbacks videoSink = holder.get();
        unattach();

        if (null != videoSink) {
            MediaStreamListener mediaStreamListener = new MediaStreamListener(stream);
            media.attachMediaStream(mediaStreamListener, stream);
            mediaStreamListeners.add(mediaStreamListener);
        }
    }

    public void unattach() {
        VideoRenderer.Callbacks videoSink = holder.get();
        for (MediaStreamListener mediaStreamListener : mediaStreamListeners) {
            if (null != videoSink) {
                media.unattachMediaStream(mediaStreamListener, mediaStreamListener.mediaStream);
            }
            mediaStreamListener.active = false;
        }
        mediaStreamListeners.clear();
    }

    public boolean isAttachedTo(@NonNull MediaStream mediaStream) {
        for (MediaStreamListener mediaStreamListener : mediaStreamListeners) {
            if (mediaStreamListener.mediaStream == mediaStream && mediaStreamListener.active)
                return true;
        }
        return false;
    }

    @Override
    public boolean equals(@Nullable Object obj) {
        VideoRenderer.Callbacks videoSink = holder.get();

        return obj == this || (obj instanceof VideoSink && obj.equals(videoSink));
    }

    private class MediaStreamListener implements VideoRenderer.Callbacks {
        public MediaStream mediaStream;
        public boolean active;

        public MediaStreamListener(MediaStream mediaStream) {
            this.mediaStream = mediaStream;
            this.active = true;
        }

        @Override
        public void renderFrame(VideoRenderer.I420Frame frame) {
            VideoRenderer.Callbacks videoSink = holder.get();
            if (active) {
                if (null != videoSink) videoSink.renderFrame(frame);
            } else {
                if (null != videoSink && null != media)
                    media.unattachMediaStream(videoSink, mediaStream);
            }
        }
    }


}
