package com.voxeet.sdk.core.services.conference.promises;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.services.AbstractPromiseable;
import com.voxeet.sdk.core.services.ConferenceSdkObservableProvider;
import com.voxeet.sdk.core.services.ConferenceService;
import com.voxeet.sdk.core.services.MediaDeviceService;
import com.voxeet.sdk.core.services.conference.information.ConferenceInformation;
import com.voxeet.sdk.core.services.conference.information.ConferenceState;
import com.voxeet.sdk.core.services.conference.information.LocalConferenceType;
import com.voxeet.sdk.events.error.ConferenceCreatedError;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.sdk.ConferenceCreatingEvent;
import com.voxeet.sdk.events.sdk.ConferenceCreationSuccess;
import com.voxeet.sdk.json.CreateConferenceParams;
import com.voxeet.sdk.json.internal.MetadataHolder;
import com.voxeet.sdk.json.internal.ParamsHolder;
import com.voxeet.sdk.models.v1.CreateConferenceResult;
import com.voxeet.sdk.utils.HttpHelper;

import org.greenrobot.eventbus.EventBus;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import retrofit2.Call;
import retrofit2.Response;

public class CreateConferencePromiseable extends AbstractPromiseable<CreateConferenceResult> {

    private String conferenceAlias;
    private MetadataHolder metadata;
    private ParamsHolder params = null;

    public CreateConferencePromiseable(@NonNull ConferenceService parent, @NonNull MediaDeviceService mediaDeviceService, @NonNull ConferenceSdkObservableProvider provider, @Nullable ConferenceInformation information, @NonNull EventBus eventBus,
                                       @Nullable String conferenceAlias,
                                       @Nullable MetadataHolder metadata,
                                       @Nullable ParamsHolder params) {
        super(parent, mediaDeviceService, provider, information, eventBus);

        this.conferenceAlias = conferenceAlias;
        this.metadata = metadata;
        this.params = params;
    }

    @NonNull
    @Override
    public Promise<CreateConferenceResult> createPromise() {
        final Promise<CreateConferenceResult> promise = new Promise<>(new PromiseSolver<CreateConferenceResult>() {
            @Override
            public void onCall(@NonNull final Solver<CreateConferenceResult> solver) {
                //TODO SET STATE FOR CONFERENCE ALIAS

                //setIsInConference(true);

                getEventBus().post(new ConferenceCreatingEvent(conferenceAlias, null));
                log("Attempting to create mConferene alias:=" + conferenceAlias);

                CreateConferenceParams arg = new CreateConferenceParams()
                        .setMetadataHolder(metadata)
                        .setParamsHolder(params);
                //        .setStats(mEnableStats);

                if (null != conferenceAlias) arg.setConferenceAlias(conferenceAlias);

                //now create the conference and will retry in case of issue...
                internalCreate(true, arg, solver);
            }
        });

        if (VoxeetSdk.session().isSocketOpen()) {
            return promise;
        } else {
            return new Promise<>(new PromiseSolver<CreateConferenceResult>() {
                @Override
                public void onCall(@NonNull final Solver<CreateConferenceResult> solver) {
                    VoxeetSdk.session().open().then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver internal_solver) {
                            promise.then(new PromiseExec<CreateConferenceResult, Object>() {
                                @Override
                                public void onCall(@Nullable CreateConferenceResult result, @NonNull Solver<Object> internal_solver) {
                                    solver.resolve(result);
                                }
                            }).error(new ErrorPromise() {
                                @Override
                                public void onError(@NonNull Throwable error) {
                                    solver.reject(error);
                                }
                            });
                        }
                    }).error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            solver.reject(error);
                        }
                    });
                }
            });
        }
    }


    /**
     * Create a conference and retry if neede
     *
     * @param retry  retry or not
     * @param params the params to send
     * @param solver the solver to resolve
     */
    private void internalCreate(final boolean retry, final CreateConferenceParams params, final Solver<CreateConferenceResult> solver) {
        Call<CreateConferenceResult> observable = getProvider().getCreateConferenceObservable(params);
        HttpHelper.enqueue(observable, new HttpHelper.HttpCallback<CreateConferenceResult>() {
            @Override
            public void onSuccess(@NonNull CreateConferenceResult object, @NonNull Response<CreateConferenceResult> response) {
                log("onNext: having conference");
                ConferenceInformation information = createOrSetConferenceWithParams(object.conferenceId, object.conferenceAlias);

                information.setConferenceState(ConferenceState.CREATED);
                information.setConferenceType(LocalConferenceType.CONFERENCE);

                getEventBus().post(new ConferenceCreationSuccess(information.getConference()));

                log("internalCreate onNext: join with := " + object.conferenceId + " " + object.conferenceAlias);

                //TODO check for issue in here but for now, setting back id...
                information.getConference().setConferenceId(object.conferenceId);
                log("now conference is " + information.getConference().getId());

                solver.resolve(object);
            }

            @Override
            public void onFailure(@NonNull Throwable e, @Nullable Response<CreateConferenceResult> response) {
                HttpException.dumpErrorResponse(response);
                e.printStackTrace();

                if (!retry) {
                    log("internalCreate onFailure: conference creation failed ! but no retry... quit...");
                    setIsInConference(false);

                    closeMedia();

                    getEventBus().post(new ConferenceCreatedError(getParent().handleError(e)));
                    solver.reject(e);
                } else {
                    log("internalCreate onFailure: conference creation failed ! but retry... now...");
                    internalCreate(false, params, solver);
                }
            }
        });
    }
}
