package com.voxeet.sdk.core.services.conference.information;

import android.support.annotation.NonNull;
import android.util.Log;

import com.voxeet.sdk.media.audio.UserPosition;
import com.voxeet.sdk.models.Conference;
import com.voxeet.sdk.models.User;
import com.voxeet.sdk.models.v1.ConferenceUser;
import com.voxeet.sdk.models.v1.Participant;
import com.voxeet.sdk.utils.Annotate;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Holds and manage information about a given conference
 * <p>
 * Created and holds into the SDK, developers can use it to get information such as :
 * - the conference instance
 * - the list of users and their updated state
 */
@Annotate
public class ConferenceInformation {

    private static final String TAG = ConferenceInformation.class.getSimpleName();

    @NonNull
    private Conference mConference;

    @NonNull
    private Map<String, String> mUserIdsCached;

    private LocalConferenceType mConferenceType;

    @NonNull
    private List<User> mLastInvitationReceived;
    private ConferenceState state = ConferenceState.DEFAULT;

    private HashMap<String, UserPosition> mPositions;

    private boolean mOwnScreenShareStarted;
    private boolean mOwnVideoStarted;
    private ConferenceUserType mUserType;
    private boolean telecomMode;

    private ConferenceInformation() {
        telecomMode = false;
        mUserType = ConferenceUserType.NORMAL;
        mUserIdsCached = new HashMap<>();
        mLastInvitationReceived = new ArrayList<>();
        mConference = new Conference();
        mConferenceType = LocalConferenceType.NONE;
        mOwnVideoStarted = false;
        mOwnScreenShareStarted = false;
        mPositions = new HashMap<>();
    }

    /**
     * Creates a new Information holder, used internally by the SDK
     *
     * @param conferenceId the conferenceId (not the alias)
     */
    public ConferenceInformation(@NonNull User localUser, @NonNull String conferenceId) {
        this();

        mConference.updateUser(localUser);
        mConference.setConferenceId(conferenceId);
    }

    /**
     * In 3D Audio context, retrieve the user's position
     *
     * @param user a valid instance of User
     * @return the position of given user
     */
    @NonNull
    public UserPosition getPosition(@NonNull ConferenceUser user) {
        return getPosition(user.getUserId());
    }

    /**
     * In 3D Audio context, retrieve the user's position
     *
     * @param userId a valid user's id
     * @return the position of given user
     */
    @NonNull
    public UserPosition getPosition(@NonNull String userId) {
        UserPosition p = mPositions.get(userId);
        if (null == p) p = new UserPosition(0, 0);
        return p;
    }

    /**
     * Set the user's position
     *
     * @param user  a valid instance of user
     * @param point a position to set
     */
    public void setPosition(@NonNull ConferenceUser user, @NonNull UserPosition point) {
        setPosition(user.getUserId(), point);
    }

    /**
     * Set the user's position
     *
     * @param userId an user Id
     * @param point  a position to set
     */
    public void setPosition(@NonNull String userId, @NonNull UserPosition point) {
        mPositions.put(userId, point);
    }

    /**
     * Get the Conference instance from this information holder
     *
     * @return a valid instance
     */
    @NonNull
    public Conference getConference() {
        return mConference;
    }

    /**
     * Set a conference inside this ConferenceInformation holder. To be used internally b the SDK
     *
     * @param conference
     */
    public void setConference(@NonNull Conference conference) {
        mConference = conference;
    }

    /**
     * Set the current conference's type : None (default when created), Conference or Replay
     * <p>
     * This is not to be changed during the developer's app lifecycle
     *
     * @param type the conference's type, preferrably Conference or Replay
     */
    public void setConferenceType(@NonNull LocalConferenceType type) {
        mConferenceType = type;
    }

    /**
     * get the current local conference type (None, Conference, replay). It's not a presentation of a conference from the server's point of view
     *
     * @return
     */
    @NonNull
    public LocalConferenceType getConferenceType() {
        return mConferenceType;
    }

    /**
     * Gets the various of users in cache in this conference. Will be removed in favor of the list of users in the Conference object
     *
     * @return a valid list of User's Id
     */
    @Deprecated
    @NonNull
    public Map<String, String> getUserIdsCached() {
        return mUserIdsCached;
    }

    /**
     * retrieve the list of users that where invited
     *
     * @return a valid list of User
     */
    @Deprecated
    @NonNull
    public List<User> getLastInvitationReceived() {
        return mLastInvitationReceived;
    }

    /**
     * Update the current conference state
     *
     * @param state
     */
    public void setConferenceState(@NonNull ConferenceState state) {
        this.state = state;
    }

    /**
     * Get the current conference's state
     * <p>
     * The various state possible are enough to manage a pure conference UX and UI from a developer's point of view
     *
     * @return
     */
    @NonNull
    public ConferenceState getConferenceState() {
        return state;
    }

    /**
     * Set if the user asked for video in this conference state
     * Helpful to deal with new privacy features in Android P+
     *
     * @param ownVideoStarted a boolean indicating the proper state
     */
    public void setOwnVideoStarted(boolean ownVideoStarted) {
        Log.d(TAG, "setOwnVideoStarted: " + ownVideoStarted);
        mOwnVideoStarted = ownVideoStarted;
    }

    /**
     * Return if the user asked for video in this conference state
     * Helpful to deal with new privacy features in Android P+
     *
     * @return a boolean indicating the proper state
     */
    public boolean isOwnVideoStarted() {
        Log.d(TAG, "isOwnVideoStarted: " + mOwnVideoStarted);
        return mOwnVideoStarted;
    }

    /**
     * Change the current user conference type, to be used internally by the SDK
     *
     * @param userType
     */
    public void setConferenceUserType(@NonNull ConferenceUserType userType) {
        mUserType = userType;
    }

    /**
     * Retrieve the current user type in the conference (listener, normal, broadcaster etc...)
     *
     * @return
     */
    public ConferenceUserType getConferenceUserType() {
        return mUserType;
    }

    /**
     * Informs the developer if the currentlt logged in user is in broadcaster mode. Perfect for webinars
     *
     * @return
     */
    public boolean isBroadcaster() {
        return ConferenceUserType.BROADCASTER.equals(mUserType);
    }

    /**
     * Informs the developer if the currently logged in user is in listening mode
     *
     * @return
     */
    public boolean isListener() {
        return ConferenceUserType.LISTENER.equals(mUserType);
    }

    /**
     * Must be done on init
     * <p>
     * Any call afterward to this won't only be used when the "join" will be reused
     * <p>
     * It is should usefull if a user will switch from any mode to the other with proper leave -> join
     *
     * @param userType the state to use
     */
    public void setLocalUserType(ConferenceUserType userType) {
        mUserType = userType;
    }

    /**
     * Set the screen share's state
     *
     * @param enable
     */
    public void setScreenShareOn(boolean enable) {
        mOwnScreenShareStarted = enable;
    }

    /**
     * From a screen share lifecycler perspective, checks if one is being played right not
     *
     * @return the screen share's state
     */
    public boolean isScreenShareOn() {
        return mOwnScreenShareStarted;
    }

    /**
     * Transforms, from a SDK point of view, a list of participants from the REST Api into Conference's updated user
     *
     * @param participants
     */
    public void participantsToConferenceUsers(List<Participant> participants) {
        if (null != participants) {
            Map<String, String> cache = getUserIdsCached();
            for (Participant participant : participants) {
                cache.put(participant.getExternalId(), participant.getExternalId());
            }

            Conference conference = getConference();
            conference.updateParticipants(participants);
        }
    }

    /**
     * Set or unset the telecom mode into this conference
     *
     * @param telecomMode the new telecommunication mode
     */
    public void setTelecomMode(boolean telecomMode) {
        this.telecomMode = telecomMode;
    }

    /**
     * Check if this conference is in telecom mode
     * <p>
     * i.e the conference ends when someone leaves
     *
     * @return a boolean indicating if this conference is such a telecommunication mode one
     */
    public boolean isTelecomMode() {
        return telecomMode;
    }

    public void clean() {
        mPositions.clear();
        mConference.getUsers().clear();
    }

    @Override
    public String toString() {
        return "ConferenceInformation{" +
                "Conference='" + mConference + '\'' +
                '}';
    }
}
