package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.log.factory.LogFactory;
import com.voxeet.sdk.core.VoxeetHttp;
import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.endpoints.IUserRService;
import com.voxeet.sdk.core.network.websocket.VoxeetWebSocket;
import com.voxeet.sdk.core.preferences.VoxeetPreferences;
import com.voxeet.sdk.events.error.GetUploadTokenErrorEvent;
import com.voxeet.sdk.events.error.HttpException;
import com.voxeet.sdk.events.error.NewLoginRequiredEvent;
import com.voxeet.sdk.events.error.SdkLogoutErrorEvent;
import com.voxeet.sdk.events.error.SocketConnectErrorEvent;
import com.voxeet.sdk.events.sdk.SdkLogoutSuccessEvent;
import com.voxeet.sdk.events.sdk.SocketConnectEvent;
import com.voxeet.sdk.events.success.GetTokenUploadSuccessResult;
import com.voxeet.sdk.json.UserInfo;
import com.voxeet.sdk.models.User;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.HttpHelper;
import com.voxeet.sdk.utils.NoDocumentation;

import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.ErrorPromise;
import eu.codlab.simplepromise.solve.PromiseExec;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;

/**
 * Component made to manage interaction with the current User logged in or to log in into the SDK
 */
@Annotate
public class SessionService extends com.voxeet.sdk.core.AbstractVoxeetService<IUserRService> {

    private static final String TAG = SessionService.class.getSimpleName();
    private final VoxeetSdk instance;
    private VoxeetWebSocket _voxeet_websocket;
    private List<Solver<Boolean>> mWaitForLogInSocket;
    private ReentrantLock lockConnectAttempt = new ReentrantLock();

    @Nullable
    private UserInfo _user_info;

    /**
     * Instantiates a new User service.
     *
     * @param instance the parent instance
     */
    @NoDocumentation
    public SessionService(VoxeetSdk instance) {
        super(instance, IUserRService.class);

        this.instance = instance;

        mWaitForLogInSocket = new ArrayList<>();
        _voxeet_websocket = new VoxeetWebSocket(instance, instance.getVoxeetEnvironmentHolder().getSocketUrl());

        registerEventBus();
    }

    /**
     * Connect the WebSocket for the current session. Does not check for any issue with user logged in or not.
     * <p>
     * This method will be properly released for developer use in a future SDK version.
     *
     * @param provider a VoxeetHttp provider
     */
    public void connectSocket(VoxeetHttp provider) {
        _voxeet_websocket.connect(provider);
    }

    @NoDocumentation
    @Deprecated
    public final void getUploadToken() {
        final Call<GetTokenUploadSuccessResult> user = getService().getUploadToken();
        user.enqueue(new Callback<GetTokenUploadSuccessResult>() {
            @Override
            public void onResponse(Call<GetTokenUploadSuccessResult> call, Response<GetTokenUploadSuccessResult> response) {
                if (response.isSuccessful()) {
                    getEventBus().post(response);

                    VoxeetPreferences.setUploadToken(response.body().getUploadToken());
                } else {
                    getEventBus().post(new NewLoginRequiredEvent("New login required"));
                }
            }

            @Override
            public void onFailure(Call<GetTokenUploadSuccessResult> call, Throwable e) {
                getEventBus().post(new GetUploadTokenErrorEvent(handleError(e)));
            }
        });
    }

    /**
     * Open a session with the specified UserInfo. It will act as the non parameterized login but it won't reject because of invalid data since those are in this method call.
     *
     * @param userInfo the UserInfo to use. At least an UserName should be given
     * @return the promise to resolve
     */
    @NonNull
    public Promise<Boolean> open(@NonNull UserInfo userInfo) {
        _user_info = userInfo;
        return open();
    }

    /**
     * Log the user. If no parameters have been given prior to this call, a rejection will be fired. If an user is already logged in and currently connected, it will be silently trapped.
     * <p>
     * Note : if multiple login are done for different users, only the first call will be used
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<Boolean> open() {
        if (null == _user_info) rejectUserNull("login");

        if (isSocketOpen()) {
            return new Promise<Boolean>(new PromiseSolver<Boolean>() {
                @Override
                public void onCall(@NonNull Solver<Boolean> solver) {
                    Log.d(TAG, "onCall: socket opened");
                    lockConnectAttempt();
                    resolveLoginSockets();
                    unlockConnectAttempt();

                    solver.resolve(true);
                }
            });
        }

        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull Solver<Boolean> solver) {
                if (null != _voxeet_websocket && isSocketOpen()) {
                    _voxeet_websocket.close(true);
                }

                Log.d(TAG, "onCall: start login in promise");
                lockConnectAttempt();

                Log.d(TAG, "onCall: having the list with elements := " + mWaitForLogInSocket);
                // the current list if not empty, we make sure that we are not calling it more than one 1 time
                if (mWaitForLogInSocket.size() == 0) {
                    mWaitForLogInSocket.add(solver);

                    unlockConnectAttempt(); //unlock here not before
                    IdentifyInlogin().then(new PromiseExec<Boolean, Object>() {
                        @Override
                        public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                            //nothing, will be in the events
                            Log.d(TAG, "onCall: first login part done");
                        }
                    }).error(new ErrorPromise() {
                        @Override
                        public void onError(@NonNull Throwable error) {
                            Log.d(TAG, "onError: login error " + error.getMessage());
                            //solver.reject(error);

                            lockConnectAttempt();
                            rejectLoginSockets(error);
                            clearLoginSockets();
                            unlockConnectAttempt(); //unlock here not before
                            Log.d(TAG, "onError: login error managed");
                        }
                    });
                } else {
                    Log.d(TAG, "onCall: already have a login attempt in progress");
                    try {
                        throw new IllegalStateException("Can not open a session while an other one is trying to be started");
                    } catch (Exception e) {
                        solver.reject(e);
                    }
                    unlockConnectAttempt(); //unlock here not before

                    //nothing to do, it will be resolved later
                }
                Log.d(TAG, "onCall: start login done... can take a time");
            }
        });
    }

    /**
     * Close the current socket without logging out the user. Perfect when the user's information needs to still exists in the server side (invitation etc...) but a soft-logout is being required localy.
     */
    public void closeSocket() {
        _voxeet_websocket.close(true);
    }

    @NoDocumentation
    @Deprecated
    public void cleanUserSession(@NonNull String id) {
        getVoxeetHttp(instance).cleanUserSession(id);
    }

    /**
     * Check if the socket is currently opened for the session
     * <p>
     * The WebSocket is the last stage of the login process in the SDK. The first one being the authentication using the REST Api
     *
     * @return the current connectivity check
     */
    public boolean isSocketOpen() {
        return _voxeet_websocket.isOpen();
    }

    /**
     * Get the registered UserInfo for the current session
     * <p>
     * If no UserInfo have been provided, e.g. no login done, the reference will be null
     *
     * @return the internal reference or null
     */
    @Nullable
    public UserInfo getUserInfo() {
        return _user_info;
    }

    /**
     * Get the identifier for the current user obtained during the login process
     *
     * @return the identifier, a string, or null if no login or a logout has been performed
     */
    @Nullable
    public String getUserId() {
        return VoxeetPreferences.id();
    }

    /**
     * Get a corresponding User representation of the currently logged in User. It's not an object tied to any Conference
     *
     * @return a new instance aggregating the (userId, userInfo)
     */
    @Nullable
    public User getUser() {
        String userId = getUserId();
        UserInfo userInfo = getUserInfo();

        return null != userId ? new User(userId, userInfo) : null;
    }

    /**
     * Check if a given user is corresponding to the local one. userfull when trying to sort out quickly if an User is an external one
     *
     * @param user the valid user to check upon (with an id)
     * @return if both have the same Voxeet's id
     */
    public boolean isLocalUser(@NonNull User user) {
        String id = user.getId();
        return null != id && id.equals(getUserId());
    }

    /**
     * Logout the current user if logged in.
     * <p>
     * Note : logging out will cancel any login in progress, leave any conference.
     * <p>
     * Warning : if logging out and login right "during" this process can lead to issue. The SDK is currently not checking anything on this part, please check that any call to logout has been finished before trying to login your user.
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<Boolean> close() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                LogFactory.instance.disconnect();
                Log.d(TAG, "onCall: logout called");

                lockConnectAttempt();
                try {
                    throw new IllegalStateException("You have awaiting login, the logout automatically canceled those");
                } catch (Exception e) {
                    rejectLoginSockets(e);
                }
                clearLoginSockets();
                unlockConnectAttempt();

                try {
                    ConferenceService service = VoxeetSdk.conference();

                    service.clearConferencesInformation();

                    if (null != service && service.isLive()) {
                        service.leave().then(new PromiseExec<Boolean, Object>() {
                            @Override
                            public void onCall(@Nullable Boolean result, @NonNull Solver<Object> solver) {
                                Log.d(TAG, "onCall: leaving while logging out done");
                            }
                        }).error(new ErrorPromise() {
                            @Override
                            public void onError(@NonNull Throwable error) {
                                Log.d(TAG, "onError: leaving when logout error -- for information :");
                                error.printStackTrace();
                            }
                        });
                    }
                } catch (Exception e) {
                    Log.d(TAG, "onCall: WARNING :: please report this error :");
                    e.printStackTrace();
                }

                Log.d(TAG, "onCall: clean awaiting sockets done");

                internalLogout().then(new PromiseExec<Boolean, Object>() {
                    @Override
                    public void onCall(@Nullable Boolean result, @NonNull Solver<Object> s) {
                        Log.d(TAG, "onCall: logout result := " + result + " ... propagating...");
                        solver.resolve(result);
                    }
                }).error(new ErrorPromise() {
                    @Override
                    public void onError(@NonNull Throwable error) {
                        solver.reject(error);
                    }
                });
            }
        });
    }

    private Promise<Boolean> internalLogout() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                String id = getUserId();
                if (null != id) {
                    Log.d(TAG, "Attempting to logout");

                    final Call<ResponseBody> user = getService().logout(VoxeetPreferences.token());
                    HttpHelper.enqueue(user, new HttpHelper.HttpCallback<ResponseBody>() {
                        @Override
                        public void onSuccess(@NonNull ResponseBody object, @NonNull Response<ResponseBody> response) {

                            VoxeetPreferences.onLogout();
                            closeSocket();
                            cleanUserSession(id);

                            String message;
                            if (response.code() == 200) {
                                message = "Logout success";

                                getEventBus().post(new SdkLogoutSuccessEvent(message));
                                solver.resolve(true);
                            } else {
                                message = "Logout failed";

                                getEventBus().post(new SdkLogoutErrorEvent(message));
                                solver.resolve(false);
                            }
                        }

                        @Override
                        public void onFailure(@NonNull Throwable e, @Nullable Response<ResponseBody> response) {
                            HttpException.dumpErrorResponse(response);
                            e.printStackTrace();

                            String message = "Logout failed";

                            VoxeetPreferences.onLogout();
                            VoxeetSdk.session().closeSocket();
                            VoxeetSdk.session().cleanUserSession(id);

                            getEventBus().post(new SdkLogoutErrorEvent(handleError(e)));
                            solver.reject(e);
                        }
                    });
                } else {
                    final String message = "Already logged out";
                    Log.d(TAG, message);
                    getEventBus().post(new SdkLogoutSuccessEvent("Already logged out"));
                    solver.resolve(true);
                }
            }
        });
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(SocketConnectEvent event) {
        lockConnectAttempt();
        resolveLoginSockets();
        unlockConnectAttempt(); //unlock here not before
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEvent(SocketConnectErrorEvent event) {
        lockConnectAttempt();
        int i = 0;
        while (i < mWaitForLogInSocket.size()) {
            Log.d(TAG, "onEvent: calling resolved false");
            Solver<Boolean> solver = mWaitForLogInSocket.get(i);
            try {
                solver.resolve(false);
            } catch (Exception e) {
                e.printStackTrace();
            }
            i++;
        }

        clearLoginSockets();
        unlockConnectAttempt(); //unlock here not before
    }

    @NonNull
    private Promise<Boolean> IdentifyInlogin() {
        if (null == _user_info) return rejectUserNull("on login attempt");
        return getVoxeetHttp(instance).identifyChain(_user_info);
    }

    protected VoxeetWebSocket getSocket() {
        return _voxeet_websocket;
    }

    private void lockConnectAttempt() {
        try {
            lockConnectAttempt.lock();
        } catch (Exception e) {

        }
    }

    private void unlockConnectAttempt() {
        try {
            if (lockConnectAttempt.isLocked()) {
                lockConnectAttempt.unlock();
            }
        } catch (Exception e) {

        }
    }

    private void rejectLoginSockets(@Nullable Throwable error) {
        try {
            int i = 0;
            while (i < mWaitForLogInSocket.size()) {
                Log.d(TAG, "onError: calling reject");
                Solver<Boolean> solver = mWaitForLogInSocket.get(i);
                try {
                    solver.reject(error);
                } catch (Exception e) {
                    e.printStackTrace();
                }
                i++;
            }
        } catch (Exception err) {
            err.printStackTrace();
        }
    }

    private void resolveLoginSockets() {
        try {
            int i = 0;
            while (i < mWaitForLogInSocket.size()) {
                Log.d(TAG, "onEvent: calling resolved true");
                Solver<Boolean> solver = mWaitForLogInSocket.get(i);
                try {
                    solver.resolve(true);
                } catch (Exception e) {
                    e.printStackTrace();
                }
                i++;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        clearLoginSockets();
    }

    private void clearLoginSockets() {
        try {
            mWaitForLogInSocket.clear();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private Promise<Boolean> rejectUserNull(@NonNull String origin) {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull Solver<Boolean> solver) {
                try {
                    throw new IllegalStateException("Invalid ! user not set ! sent from " + origin);
                } catch (Exception e) {
                    solver.reject(e);
                }
            }
        });
    }
}