package com.voxeet.sdk.core.services;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.network.endpoints.ISdkVideoPresentationRService;
import com.voxeet.sdk.core.services.abstracts.AbstractPresentationService;
import com.voxeet.sdk.core.services.videopresentation.PresentationState;
import com.voxeet.sdk.core.services.videopresentation.VideoPresentationInformation;
import com.voxeet.sdk.json.VideoPresentationPaused;
import com.voxeet.sdk.json.VideoPresentationPlay;
import com.voxeet.sdk.json.VideoPresentationSeek;
import com.voxeet.sdk.json.VideoPresentationStarted;
import com.voxeet.sdk.json.VideoPresentationStopped;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;

import java.util.ArrayList;
import java.util.List;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;

/**
 * Manage every outgoing or ingoing call related to Video stream from third party or outside files
 */
@Annotate
public class SDKVideoPresentationService extends AbstractPresentationService<ISdkVideoPresentationRService> {

    private List<Solver<VideoPresentationStarted>> mCacheStartedSolvers;
    private List<Solver<VideoPresentationStopped>> mCacheStoppedSolvers;
    private List<Solver<VideoPresentationPaused>> mCachePausedSolvers;
    private List<Solver<VideoPresentationPlay>> mCachePlaySolvers;
    private List<Solver<VideoPresentationSeek>> mCacheSeekSolvers;

    private List<VideoPresentationInformation> presentations;
    /**
     * Instantiates a new User service.
     *
     * @param instance the parent instance
     */
    @NoDocumentation
    public SDKVideoPresentationService(VoxeetSdk instance) {
        super(instance, ISdkVideoPresentationRService.class);

        presentations = new ArrayList<>();

        mCachePausedSolvers = new ArrayList<>();
        mCacheStartedSolvers = new ArrayList<>();
        mCacheStoppedSolvers = new ArrayList<>();
        mCachePausedSolvers = new ArrayList<>();
        mCachePlaySolvers = new ArrayList<>();
        mCacheSeekSolvers = new ArrayList<>();
        registerEventBus();
    }

    void onInternalServiceEvent(VideoPresentationStarted event) {
        tryUnlock(event, mCacheStartedSolvers);

        VideoPresentationInformation information = getPresentationInformation(event.key);
        if(null == information) {
            information = new VideoPresentationInformation(event.key, event.url);
            presentations.add(information);
        }

        information.state = PresentationState.STARTED;
        getEventBus().post(event);
    }

    void onInternalServiceEvent(VideoPresentationStopped event) {
        tryUnlock(event, mCacheStoppedSolvers);
        VideoPresentationInformation information = getPresentationInformation(event.key);
        if(null != information) {
            information.state = PresentationState.STOP;
            presentations.remove(information);
        }
        getEventBus().post(event);
    }

    void onInternalServiceEvent(VideoPresentationPaused event) {
        tryUnlock(event, mCachePausedSolvers);
        VideoPresentationInformation information = getPresentationInformation(event.key);
        if(null != information) information.state = PresentationState.PAUSED;
        getEventBus().post(event);
    }

    void onInternalServiceEvent(VideoPresentationPlay event) {
        tryUnlock(event, mCachePlaySolvers);
        VideoPresentationInformation information = getPresentationInformation(event.key);
        if(null != information) information.state = PresentationState.PLAY;
        getEventBus().post(event);
    }

    void onInternalServiceEvent(VideoPresentationSeek event) {
        tryUnlock(event, mCacheSeekSolvers);
        VideoPresentationInformation information = getPresentationInformation(event.key);
        if(null != information) information.state = PresentationState.SEEK;
        getEventBus().post(event);
    }

    /**
     * Made a call to change the given video "start" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationStarted> startVideoPresentation(final String url) {
        return new Promise<>(new PromiseSolver<VideoPresentationStarted>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationStarted> solver) {
                ISdkVideoPresentationRService.VideoPresentationUrl holder = new ISdkVideoPresentationRService.VideoPresentationUrl(url);
                consumeInternalCall(solver, mCacheStartedSolvers,
                        internalCall(getService().startVideoPresentation(getConferenceId(), holder)));
            }
        });
    }

    /**
     * Made a call to change the given video "stop" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationStopped> stopVideoPresentation() {
        return new Promise<>(new PromiseSolver<VideoPresentationStopped>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationStopped> solver) {

                consumeInternalCall(solver, mCacheStoppedSolvers,
                        internalCall(getService().stopVideoPresentation(getConferenceId())));
            }
        });
    }

    /**
     * Made a call to change the given video "play" parameter
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationPlay> playVideoPresentation() {
        return new Promise<>(new PromiseSolver<VideoPresentationPlay>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationPlay> solver) {

                consumeInternalCall(solver, mCachePlaySolvers,
                        internalCall(getService().playVideoPresentation(getConferenceId())));
            }
        });
    }

    /**
     * Made a call to change the given video "pause" parameter
     *
     * @param timestamp the timestamp the video paused at
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationPaused> pauseVideoPresentation(final long timestamp) {
        return new Promise<>(new PromiseSolver<VideoPresentationPaused>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationPaused> solver) {
                ISdkVideoPresentationRService.VideoPresentationSeek body_sent = new ISdkVideoPresentationRService.VideoPresentationSeek(timestamp);

                consumeInternalCall(solver, mCachePausedSolvers,
                        internalCall(getService().pauseVideoPresentation(getConferenceId(), body_sent)));
            }
        });
    }

    /**
     * Made a call to change the given video "seek" parameter
     *
     * @param timestamp the new timestamp
     * @return the promise to resolve
     */
    @NonNull
    public Promise<VideoPresentationSeek> seekVideoPresentation(final long timestamp) {
        return new Promise<>(new PromiseSolver<VideoPresentationSeek>() {
            @Override
            public void onCall(@NonNull final Solver<VideoPresentationSeek> solver) {

                ISdkVideoPresentationRService.VideoPresentationSeek body_sent = new ISdkVideoPresentationRService.VideoPresentationSeek(timestamp);

                consumeInternalCall(solver, mCacheSeekSolvers,
                        internalCall(getService().seekVideoPresentation(getConferenceId(), body_sent)));
            }
        });
    }

    /**
     * If a presentation is currently playing, this method will return the description of this presentation
     *
     * if none (or stopped) this method will return null
     * This method exists primarily because of the possible various Activities states
     * @return a clone of the holder of the (key, url, state) or null if no presentation currently playing
     */
    @Nullable
    public VideoPresentationInformation getCurrentPresentation() {
        for (VideoPresentationInformation information : presentations) {
            if(!PresentationState.STOP.equals(information)) return information.clone();
        }
        return null;
    }

    @Nullable
    private VideoPresentationInformation getPresentationInformation(@NonNull String key) {
        for (VideoPresentationInformation information: presentations) {
            if(key.equals(information.key)) return information;
        }

        return null;
    }

}