package com.voxeet.sdk.core.services;

import android.content.Context;
import android.os.Build;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.util.Log;

import com.voxeet.android.media.MediaEngine;
import com.voxeet.android.media.MediaEngineException;
import com.voxeet.android.media.MediaStream;
import com.voxeet.sdk.core.AbstractVoxeetService;
import com.voxeet.sdk.core.VoxeetSdk;
import com.voxeet.sdk.core.services.holder.ServiceProviderHolder;
import com.voxeet.sdk.core.services.media.VideoSinkHolder;
import com.voxeet.sdk.events.error.CameraSwitchErrorEvent;
import com.voxeet.sdk.events.sdk.CameraSwitchSuccessEvent;
import com.voxeet.sdk.media.MediaSDK;
import com.voxeet.sdk.media.camera.CameraContext;
import com.voxeet.sdk.media.camera.CameraEnumeratorLollipopWrapper;
import com.voxeet.sdk.media.camera.CameraEnumeratorPreLollipopWrapper;
import com.voxeet.sdk.utils.Annotate;
import com.voxeet.sdk.utils.NoDocumentation;
import com.voxeet.sdk.utils.Validate;

import org.webrtc.CameraVideoCapturer;
import org.webrtc.EglBase;
import org.webrtc.VideoRenderer;

import java.util.concurrent.CopyOnWriteArrayList;

import eu.codlab.simplepromise.Promise;
import eu.codlab.simplepromise.solve.PromiseSolver;
import eu.codlab.simplepromise.solve.Solver;

/**
 * Local Media related management
 * - camera
 * - media library
 * - attaching and detaching stream for VideoSink
 */
@Annotate
public class MediaDeviceService extends AbstractVoxeetService {

    private CameraContext enumerator;
    private MediaSDK media;
    private boolean isAudio3DEnabled;

    private CopyOnWriteArrayList<VideoSinkHolder> videoSinkHolders;

    @NoDocumentation
    public MediaDeviceService(@NonNull VoxeetSdk instance) {
        super(instance, ServiceProviderHolder.DEFAULT);

        videoSinkHolders = new CopyOnWriteArrayList<>();

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            enumerator = new CameraEnumeratorLollipopWrapper(instance.getApplicationContext());
        } else {
            enumerator = new CameraEnumeratorPreLollipopWrapper(instance.getApplicationContext());
        }

        enumerator.setDefaultCameraFront(true);

        isAudio3DEnabled = false;
    }

    /**
     * Check the availability of a Media instance. Media are only valid during a Conference
     *
     * @return the availability indicator
     */
    public boolean hasMedia() {
        return null != media;
    }

    /**
     * Get the current instance of the Media or null if not in a Conference (or unable to create one)
     *
     * @return the instance, need to check for null value
     */
    @Nullable
    public MediaSDK getMedia() {
        return media;
    }

    /**
     * Get the WebRTC's EglBaseContext
     * <p>
     * Mostly used by the various SDK elements but also accessible for public access
     *
     * @return the current EglBaseContext or null if no media is currently created
     */
    @Nullable
    public EglBase.Context getEglContext() {
        if (null != media) {
            return media.getEglBase().getEglBaseContext();
        } else {
            return null;
        }
    }

    /**
     * Attach a stream to a given receiver.
     * <p>
     * Note : if any stream are currently attached to the given receiver, it will be detached
     *
     * @param stream    a valid stream to attach
     * @param videoSink a videosink which will receive the updates
     * @return
     */
    @MainThread
    public boolean attachMediaStream(@NonNull MediaStream stream, @NonNull VideoRenderer.Callbacks videoSink) {
        MediaEngine media = this.media;
        if (null == media) return false;

        VideoSinkHolder holder = getOrCreate(videoSink);
        if (null == holder) return false;

        if (!holder.isAttachedTo(stream)) {
            //it is not attached to the current but... to another one -> we need to unattach it
            MediaStream previous = holder.getMediaStream();
            if (null != previous) {
                holder.unattach();
            }
            holder.attach(stream);
        }

        return true;
    }

    /**
     * Unattach a stream from a given videosink
     *
     * @param videoSink the receiver instance
     * @return indicator of the release
     */
    @MainThread
    public boolean unAttachMediaStream(@NonNull VideoRenderer.Callbacks videoSink) {
        VideoSinkHolder holder = getOrCreate(videoSink);
        if (null == holder) return false;

        holder.unattach();
        return true;
    }

    /**
     * Unattach a stream from a given videosink.
     * <p>
     * Note : this method no longer use the stream. If any streams are attached to a VideoSink, it automatically releases the current attached one
     *
     * @param stream    a stream to detach
     * @param videoSink the receiver instance
     * @return indicator of the release state
     */
    @Deprecated
    @MainThread
    public boolean unAttachMediaStream(@NonNull MediaStream stream, @NonNull VideoRenderer.Callbacks videoSink) {
        Log.d("MediaDeviceService", "unAttachMediaStream: calling deprecated unattach method");
        return unAttachMediaStream(videoSink);
        /*MediaEngine media = this.media;
        if (null == media) return false;

        VideoSinkHolder holder = getOrCreate(videoSink);
        if(null == holder) return false;

        /if (holder.isAttachedTo(stream)) {
            holder.unattach();
        }
        return true;*/
    }

    /**
     * Check if the Audio 3D layer is activated
     * <p>
     * note : any modification must be done prior to any conference joined
     *
     * @return indicator about the 3d management
     */
    public boolean isAudio3DEnabled() {
        return isAudio3DEnabled;
    }

    /**
     * Set the 3D Audio management to on/off. If no conferences or the media was not initialized, the state will retain for the next calls.
     * <p>
     * The SDK keeps the value, it must be recalled with a new value to be forced-updated
     *
     * @param enable the new state for 3d audio processing
     * @return true if everything is fine, false if the media was not initialized
     */
    public boolean setAudio3DEnabled(boolean enable) {
        return false;
    }

    /**
     * Switch the current camera to another one. The system logic is to rotate against every available camera
     * On Android, 2 camera exists by default but on some devices, it may be possible to have none, 1 or 2+
     * <p>
     * This method is also updating the Camera provider's informaton with the new camera type : front or back
     *
     * @return the promise to resolve
     */
    @NonNull
    public Promise<Boolean> switchCamera() {
        return new Promise<>(new PromiseSolver<Boolean>() {
            @Override
            public void onCall(@NonNull final Solver<Boolean> solver) {
                Validate.notNull(getMedia(), "media");

                getMedia().switchCamera(new CameraVideoCapturer.CameraSwitchHandler() {
                    @Override
                    public void onCameraSwitchDone(boolean isFrontCamera) {
                        getCameraContext().setDefaultCameraFront(isFrontCamera);

                        getEventBus().post(new CameraSwitchSuccessEvent(isFrontCamera));
                        solver.resolve(true);
                    }

                    @Override
                    public void onCameraSwitchError(String errorDescription) {
                        getEventBus().post(new CameraSwitchErrorEvent(errorDescription));
                        solver.resolve(false);
                    }
                });
            }
        });
    }

    /**
     * Retrieve the instance of the Camera information
     *
     * @return
     */
    @NonNull
    public CameraContext getCameraContext() {
        return enumerator;
    }

    @MainThread
    @Nullable
    private VideoSinkHolder getOrCreate(@Nullable VideoRenderer.Callbacks videoSink) {
        if (null == videoSink) return null;
        for (VideoSinkHolder holder : videoSinkHolders) {
            if (holder.equals(videoSink)) return holder;
        }

        VideoSinkHolder holder = new VideoSinkHolder(media, videoSink);
        videoSinkHolders.add(holder);
        return holder;
    }

    void releaseMedia() {
        for (VideoSinkHolder holder : videoSinkHolders) {
            if (holder.hasVideoSink()) {
                holder.clear();
            }
        }
        videoSinkHolders.clear();

        if (null != media) media.stop();

        media = null;
    }

    void createMedia(Context context, String userId, MediaEngine.StreamListener mediaStreamListener,
                     CameraVideoCapturer.CameraEventsHandler cameraEventsHandler,
                     boolean videoOn, boolean useMic) throws MediaEngineException {
        media = new MediaSDK(context, userId, mediaStreamListener, cameraEventsHandler, videoOn, useMic);
    }

}
